import dataclasses
import typing

import attr

from infra.walle.server.proto.walle.scenario.data_storage.v1 import scenario_data_storage_pb2
from walle.scenario.data_storage.types import HostGroupSource, MaintenanceApproversGroup


class BaseScenarioDataStorage:

    scenario_parameters_class: typing.Optional[attr.Attribute] = None

    def __init__(self, data_storage: typing.Dict):
        self._data_storage = data_storage

    def read(self):
        raise NotImplementedError

    def write(self, *args):
        raise NotImplementedError

    def read_host_groups_sources(self) -> typing.List[HostGroupSource]:
        raise NotImplementedError

    def write_host_groups_sources(self, *args):
        raise NotImplementedError

    def read_scenario_parameters(self) -> scenario_parameters_class:
        raise NotImplementedError

    def write_scenario_parameters(self, *args):
        raise NotImplementedError

    def to_protobuf(self):
        raise NotImplementedError


class DefaultDataStorage(BaseScenarioDataStorage):

    scenario_parameters_class = attr.Attribute

    def read(self) -> typing.List[MaintenanceApproversGroup]:
        return [
            MaintenanceApproversGroup(**group) for group in self._data_storage.get("maintenance_approvers_groups", [])
        ]

    def write(self, groups: typing.List[MaintenanceApproversGroup]):
        self._data_storage["maintenance_approvers_groups"] = [dataclasses.asdict(group) for group in groups]

    def read_host_groups_sources(self) -> typing.List[HostGroupSource]:
        return [
            HostGroupSource.from_dict(host_group_source_dict)
            for host_group_source_dict in self._data_storage.get("host_groups_sources", [])
        ]

    def write_host_groups_sources(self, host_groups_sources: typing.List[HostGroupSource]):
        self._data_storage["host_groups_sources"] = [
            host_group_source.to_dict() for host_group_source in host_groups_sources
        ]

    def read_scenario_parameters(self) -> scenario_parameters_class:
        parameters_dict = self._data_storage.get("scenario_parameters", {})
        return self.scenario_parameters_class(**parameters_dict)

    def write_scenario_parameters(self, scenario_parameters: scenario_parameters_class):
        self._data_storage["scenario_parameters"] = attr.asdict(scenario_parameters)

    def to_protobuf(self) -> scenario_data_storage_pb2.ScenarioDataStorage:
        storage_data_as_proto_message = scenario_data_storage_pb2.ItdcMaintenanceDataStorage()

        for group in self.read():
            proto_group = storage_data_as_proto_message.maintenance_approvers_groups.add()
            proto_group.CopyFrom(group.to_protobuf())

        proto_result = scenario_data_storage_pb2.ScenarioDataStorage()
        proto_result.itdc_maintenance_data_storage.CopyFrom(storage_data_as_proto_message)

        return proto_result
