import dataclasses
import typing

from walle.maintenance_plot.model import MaintenancePlot
from walle.scenario.host_groups_builders import maintenance_plot_getters
from walle.scenario.host_groups_builders.constants import HostGroupsSourcesTypes
from walle.scenario.host_groups_builders.exceptions import HostGroupSourceException


@dataclasses.dataclass(eq=True)
class BaseHostGroupSource:
    group_source_type: typing.ClassVar = None

    def to_dict(self):
        return dataclasses.asdict(self) | {"group_source_type": self.group_source_type}

    @classmethod
    def from_dict(cls, data: dict):
        source_type = data.get("group_source_type")
        klass = HOST_GROUPS_SOURCES_TYPES_MAP.get(source_type)
        if klass is None:
            raise HostGroupSourceException("Unknown group source type %s" % source_type)
        return klass.from_dict(data)

    @classmethod
    def get_group_source_name(cls):
        raise NotImplementedError

    def get_maintenance_plot(self) -> MaintenancePlot:
        raise NotImplementedError


@dataclasses.dataclass(eq=True)
class SpecificProjectTagHostGroupSource(BaseHostGroupSource):
    specific_project_tag: str
    group_source_type: typing.ClassVar = HostGroupsSourcesTypes.SPECIFIC_PROJECT_TAG

    @classmethod
    def from_dict(cls, data: dict):
        return SpecificProjectTagHostGroupSource(specific_project_tag=data["specific_project_tag"])

    def get_group_source_name(self):
        return f"{self.group_source_type}:{self.specific_project_tag}"

    def get_maintenance_plot(self) -> MaintenancePlot:
        return maintenance_plot_getters.by_specific_project_tag(self.specific_project_tag)


@dataclasses.dataclass(eq=True)
class MaintenancePlotHostGroupSource(BaseHostGroupSource):
    maintenance_plot_id: str
    group_source_type: typing.ClassVar = HostGroupsSourcesTypes.MAINTENANCE_PLOT

    @classmethod
    def from_dict(cls, data: dict):
        return MaintenancePlotHostGroupSource(maintenance_plot_id=data["maintenance_plot_id"])

    def get_group_source_name(self):
        return f"{self.group_source_type}:{self.maintenance_plot_id}"

    def get_maintenance_plot(self) -> MaintenancePlot:
        return maintenance_plot_getters.by_maintenance_plot_id(self.maintenance_plot_id)


@dataclasses.dataclass(eq=True)
class BotProjectIdHostGroupSource(BaseHostGroupSource):
    bot_project_id: int
    abc_service_slug: str
    group_source_type: typing.ClassVar = HostGroupsSourcesTypes.BOT_PROJECT_ID

    @classmethod
    def from_dict(cls, data: dict):
        return BotProjectIdHostGroupSource(
            bot_project_id=data["bot_project_id"], abc_service_slug=data["abc_service_slug"]
        )

    def get_group_source_name(self):
        return f"{self.group_source_type}:{str(self.bot_project_id)}/{self.abc_service_slug}"

    def get_maintenance_plot(self) -> MaintenancePlot:
        return maintenance_plot_getters.by_abc_service_slug(self.abc_service_slug)


@dataclasses.dataclass(eq=True)
class HostGroup:
    hosts_invs: list[int]
    group_source: BaseHostGroupSource


HOST_GROUPS_SOURCES_TYPES_MAP = {
    HostGroupsSourcesTypes.BOT_PROJECT_ID: BotProjectIdHostGroupSource,
    HostGroupsSourcesTypes.MAINTENANCE_PLOT: MaintenancePlotHostGroupSource,
    HostGroupsSourcesTypes.SPECIFIC_PROJECT_TAG: SpecificProjectTagHostGroupSource,
}
