import typing
from collections import defaultdict

from walle.clients import abc, bot
from walle.scenario.host_groups_builders.base import HostGroup, BotProjectIdHostGroupSource
from walle.scenario.host_groups_builders.exceptions import HostGroupSourceException
from walle.scenario.host_groups_builders.hosts_properties_getter import ProjectProperties


def build_groups_by_bot_project_id(
    hosts_project_properties_map: typing.Dict[int, ProjectProperties]
) -> (typing.List[HostGroup], typing.List[int]):
    bot_broject_id_to_hosts_invs_list_map = defaultdict(list)
    hosts_without_bot_broject_id = []

    for host_inv, project_properties in hosts_project_properties_map.items():
        if project_properties.bot_project_id is None:
            hosts_without_bot_broject_id.append(host_inv)
            continue

        bot_broject_id_to_hosts_invs_list_map[project_properties.bot_project_id].append(host_inv)

    return [
        HostGroup(
            group_source=BotProjectIdHostGroupSource(
                bot_project_id=bot_project_id,
                abc_service_slug=_get_abc_project_slug_from_bot_project_id(bot_project_id),
            ),
            hosts_invs=hosts_invs_list,
        )
        for bot_project_id, hosts_invs_list in bot_broject_id_to_hosts_invs_list_map.items()
    ], hosts_without_bot_broject_id


def _get_abc_project_slug_from_bot_project_id(bot_project_id: int) -> str:
    try:
        abc_service_id = bot.get_planner_id_by_bot_project_id(bot_project_id)
    except bot.BotInternalError as e:
        raise HostGroupSourceException(
            "Can not get ABC service ID for Bot project ID '{}': {}".format(bot_project_id, e)
        )
    try:
        return abc.get_service_slug(abc_service_id)
    except abc.ABCInternalError as e:
        raise HostGroupSourceException(
            "Can not get ABC service slug for ABC service ID '{}': {}".format(abc_service_id, e)
        )
