import typing

from walle.scenario.constants import ScriptName
from walle.scenario.host_groups_builders.base import HostGroup
from walle.scenario.host_groups_builders.by_bot_project_id import build_groups_by_bot_project_id
from walle.scenario.host_groups_builders.by_maintenance_plot import build_groups_by_maintenance_plot
from walle.scenario.host_groups_builders.by_specific_project_tag import build_groups_by_specific_project_tag
from walle.scenario.host_groups_builders.hosts_properties_getter import HostsPropertiesGetter


class HostsListSplitterException(Exception):
    pass


class BaseHostsListSplitter:
    @classmethod
    def split(cls, hosts_invs: typing.List[int]):
        raise NotImplementedError


class DefaultHostsListSplitter(BaseHostsListSplitter):

    # Order matters.
    _GROUPS_BUILDERS = [
        build_groups_by_maintenance_plot,
        build_groups_by_specific_project_tag,
        build_groups_by_bot_project_id,
    ]

    @classmethod
    def split(cls, hosts_invs: typing.List[int]) -> typing.List[HostGroup]:
        hosts_project_properties_map = cls._get_hosts_project_properties_map(hosts_invs)
        result = []

        for group_builder in cls._GROUPS_BUILDERS:
            groups, _hosts_invs_not_in_groups = group_builder(hosts_project_properties_map)
            for group in groups:
                result.append(group)
                for host_inv in group.hosts_invs:
                    del hosts_project_properties_map[host_inv]
            if not hosts_project_properties_map:
                return result

        if hosts_project_properties_map:
            raise HostsListSplitterException(
                "Not all hosts were put in a host group! Hosts invs: %s" % hosts_project_properties_map.keys()
            )

        return result

    @staticmethod
    def _get_hosts_project_properties_map(hosts_invs):
        return HostsPropertiesGetter.get_projects_properties(hosts_invs)


ALL_SPLITTERS = {
    ScriptName.ITDC_MAINTENANCE: DefaultHostsListSplitter,
    ScriptName.NOC_HARD: DefaultHostsListSplitter,
}


def get_host_list_splitter(
    script_name: str, default: typing.Optional[BaseHostsListSplitter] = DefaultHostsListSplitter
) -> typing.Optional[BaseHostsListSplitter]:
    return ALL_SPLITTERS.get(script_name, default)
