import dataclasses
import typing

from walle.hosts import Host
from walle.projects import Project
from walle.util.mongo import MongoDocument


@dataclasses.dataclass(eq=True, frozen=True)
class ProjectProperties:
    bot_project_id: typing.Optional[int]
    maintenance_plot_id: typing.Optional[str] = None
    tags: typing.List[str] = dataclasses.field(default_factory=list)

    fields_names: typing.ClassVar = [
        Project.bot_project_id.name,
        Project.maintenance_plot_id.name,
        Project.tags.name,
    ]

    @classmethod
    def from_dict(cls, data: dict):
        return ProjectProperties(
            bot_project_id=data["bot_project_id"],
            maintenance_plot_id=data.get("maintenance_plot_id"),
            tags=data.get("tags") or [],
        )


class HostsPropertiesGetter:
    @classmethod
    def get_projects_properties(cls, hosts_invs: typing.List[int]) -> typing.Dict[int, ProjectProperties]:
        hosts = cls._get_hosts_with_project(hosts_invs)
        projects_set = {host.project for host in hosts}
        projects_properties_dict = cls._get_projects_fields_values(projects_set)
        return {host.inv: projects_properties_dict[host.project] for host in hosts}

    @staticmethod
    def _get_hosts_with_project(
        hosts_invs: typing.List[int],
    ) -> typing.List[Host]:  # Technically it is a list of generated 'ApiHostDocument' which has Host's fields.
        return list(
            MongoDocument.for_model(Host).find({"inv": {"$in": hosts_invs}}, {"inv": 1, "project": 1, "_id": 0})
        )

    @classmethod
    def _get_projects_fields_values(cls, projects: typing.Set[str]) -> typing.Dict[str, ProjectProperties]:
        result = {}
        for query_result_item in MongoDocument.for_model(Project).find(
            {"_id": {"$in": list(projects)}}, {field_name: 1 for field_name in ProjectProperties.fields_names + ["_id"]}
        ):
            result[query_result_item.id] = ProjectProperties.from_dict(
                {field_name: getattr(query_result_item, field_name) for field_name in ProjectProperties.fields_names}
            )
        return result
