from walle.maintenance_plot.common_settings import MaintenanceApprovers
from walle.maintenance_plot.model import (
    MaintenancePlot,
    MaintenancePlotModel,
    MaintenancePlotMetaInfo,
    MaintenancePlotCommonSettings,
    MaintenancePlotScenarioSettings,
    CommonScenarioSettings,
)
from walle.maintenance_plot.scenarios_settings.itdc_maintenance import ItdcMaintenanceMaintenancePlotSettings
from walle.maintenance_plot.scenarios_settings.noc_hard import NocHardMaintenancePlotSettings
from walle.scenario import constants
from walle.scenario.host_groups_builders.exceptions import HostGroupSourceException
from walle.clients import abc


def by_maintenance_plot_id(maintenance_plot_id: str) -> MaintenancePlot:
    return MaintenancePlotModel.objects.get(id=maintenance_plot_id).as_dataclass()


def by_abc_service_slug(abc_service_slug: str) -> MaintenancePlot:
    return MaintenancePlot(
        id=abc_service_slug,
        meta_info=MaintenancePlotMetaInfo(
            abc_service_slug=abc_service_slug,
            name=f"Auto-generated maintenance plot for ABC service '{abc_service_slug}'",
        ),
        common_settings=MaintenancePlotCommonSettings(
            maintenance_approvers=MaintenanceApprovers(
                abc_roles_codes=[abc.Role.PRODUCT_HEAD, abc.Role.HARDWARE_RESOURCES_MANAGER],
            ),
            common_scenarios_settings=CommonScenarioSettings(),
        ),
        scenarios_settings=[
            MaintenancePlotScenarioSettings(
                scenario_type=constants.ScriptName.ITDC_MAINTENANCE,
                settings=ItdcMaintenanceMaintenancePlotSettings(
                    enable_manual_approval_after_hosts_power_off=True,
                    approval_sla=1 * 24,
                ),
            ),
            MaintenancePlotScenarioSettings(
                scenario_type=constants.ScriptName.NOC_HARD,
                settings=NocHardMaintenancePlotSettings(
                    enable_manual_approval_after_hosts_power_off=True,
                    approval_sla=1 * 24,
                ),
            ),
        ],
    )


def by_specific_project_tag(specific_project_tag: str) -> MaintenancePlot:
    maintenance_plot_data = constants.SPECIFIC_PROJECT_TAG_TO_MAINTENANCE_PLOT_DATA_MAP.get(specific_project_tag)
    if maintenance_plot_data is None:
        raise HostGroupSourceException("No maintenance plot for project tag '%s' found" % specific_project_tag)
    return MaintenancePlot(
        id=maintenance_plot_data.ID,
        meta_info=MaintenancePlotMetaInfo(
            abc_service_slug=maintenance_plot_data.META_INFO.get("abc_service_slug"),
            name=maintenance_plot_data.META_INFO.get("name"),
        ),
        common_settings=MaintenancePlotCommonSettings(
            maintenance_approvers=MaintenanceApprovers(
                logins=maintenance_plot_data.COMMON_SETTINGS.get("maintenance_approvers", {}).get("logins", []),
                abc_roles_codes=maintenance_plot_data.COMMON_SETTINGS.get("maintenance_approvers", {}).get(
                    "abc_roles_codes", []
                ),
                abc_role_scope_slugs=maintenance_plot_data.COMMON_SETTINGS.get("maintenance_approvers", {}).get(
                    "abc_role_scope_slugs", []
                ),
            ),
            common_scenarios_settings=CommonScenarioSettings(),
        ),
        scenarios_settings=[
            MaintenancePlotScenarioSettings.from_dict(scenario_settings_dict)
            for scenario_settings_dict in maintenance_plot_data.SCENARIOS_SETTINGS
        ],
    )
