"""Host status management."""

import logging

from walle.hosts import Host
from walle.scenario.host_stage_info import HostStageInfo
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.mongo import MongoDocument
from walle.util.mongo import SECONDARY_LOCAL_DC_PREFERRED

log = logging.getLogger(__name__)


def _gc_old_host_stage_info():
    hosts_in_scenario = []

    host_collection = MongoDocument.for_model(Host)
    hosts = host_collection.find(
        {"scenario_id": {"$exists": True}}, ("uuid",), read_preference=SECONDARY_LOCAL_DC_PREFERRED
    )
    for host in gevent_idle_iter(hosts):
        hosts_in_scenario.append(host.uuid)

    outdated_hsis = HostStageInfo.objects(host_uuid__nin=hosts_in_scenario).only("host_uuid", "scenario_id")

    log.info("Found %s suspected stage infos in mongo...", outdated_hsis.count())

    for hsi in gevent_idle_iter(outdated_hsis):
        revision = hsi.revision

        log.info("Process stage infos for host %s", hsi.host_uuid)

        host = host_collection.find_one({"uuid": hsi.host_uuid, "scenario_id": {"$exists": True}}, silent=True)
        if host:
            log.error("%s: Got a race during gc of host's stage info.", hsi.host_uuid)
            if host.scenario_id != hsi.scenario_id:
                log.info(
                    "Host %s has incompatibility with scenario stage info: %s != %s",
                    host.uuid,
                    host.scenario_id,
                    hsi.scenario_id,
                )
        else:
            HostStageInfo.objects(host_uuid=hsi.host_uuid, revision=revision).delete()
            log.info("Successfully removed stage info for host %s from mongo.", hsi.host_uuid)
