import logging

from sepelib.core.exceptions import LogicalError
from walle.scenario.errors import ActionDoesNotExist
from walle.scenario.marker import MarkerStatus, Marker
from walle.scenario.stage_info import StageAction, StageStatus

log = logging.getLogger(__name__)


class ActionsMap:
    def __init__(self, actions):
        action_name = _action_name
        self.name_to_action_map = {action_name(action): action for action in actions}
        self.actions_list = [action_name(action) for action in actions]
        self.name_to_list_idx_map = {action_name(action): idx for idx, action in enumerate(actions)}

    def __contains__(self, key):
        return key in self.name_to_action_map

    def __getitem__(self, key):
        if key in self.name_to_action_map:
            return self.name_to_action_map[key]
        return ActionDoesNotExist("action {} does not exist".format(key))

    def get_next_key(self, key):
        if key not in self.name_to_list_idx_map:
            return ActionDoesNotExist("action {} does not exist".format(key))

        next_idx = self.name_to_list_idx_map[key] + 1

        if next_idx == len(self.actions_list):
            return None

        return self.actions_list[next_idx]

    def get_first_action_key(self):
        return self.actions_list[0]

    def __iter__(self):
        yield from self.actions_list


def _action_name(action):
    try:
        return action.name
    except AttributeError:
        return action.__name__


class SequentialIterationStageStrategyInterface:
    def process_marker(self, marker, stage_info, active_stage_info):
        if marker.status == MarkerStatus.SUCCESS:
            active_stage_info.set_stage_finished()
            return self.make_transition(marker, stage_info)
        else:
            return Marker.in_progress(message=marker.message, data=marker.data)

    @staticmethod
    def make_transition(marker, stage_info):
        if stage_info.is_on_last_stage():
            log.debug("Stage %s has successfully executed all its stages", stage_info)
            return Marker.success(message="Stage has successfully executed all of its child stages", data=marker.data)
        else:
            stage_info.seq_num += 1
            return Marker.in_progress(message="Stage runs child stages", data=marker.data)


class SequentialIterationStageStrategy(SequentialIterationStageStrategyInterface):
    @staticmethod
    def get_active_stage(stage_info, stages):
        active_stage_info = stage_info.stages[stage_info.seq_num]
        active_stage = stages[stage_info.seq_num]
        if active_stage_info.status != StageStatus.FINISHED:
            active_stage_info.set_stage_processing()
        return active_stage, active_stage_info


class HostSequentialIterationStageStrategy(SequentialIterationStageStrategyInterface):
    @staticmethod
    def get_active_stage(host_stage_info, stages, scenario_stage_info):
        active_host_stage_info = host_stage_info.stages[host_stage_info.seq_num]
        active_scenario_stage_info = scenario_stage_info.stages[host_stage_info.seq_num]
        active_stage = stages[host_stage_info.seq_num]

        if active_host_stage_info.status != StageStatus.FINISHED:
            active_host_stage_info.set_stage_processing()
            active_scenario_stage_info.set_stage_processing()

        return active_stage, active_host_stage_info, active_scenario_stage_info


class IActionIterationStrategy:
    def make_transition(self, marker, stage_info, scenario):
        raise NotImplementedError

    def get_current_func(self, stage_info):
        raise NotImplementedError

    def get_next_action(self, stage_info):
        raise NotImplementedError


class SequentialIterationActionStrategy(IActionIterationStrategy):
    def __init__(self, actions):
        self.actions_map = ActionsMap(actions)

    def get_current_func(self, stage_info):
        if stage_info.action_type not in self.actions_map:
            raise LogicalError

        return self.actions_map[stage_info.action_type]

    def get_next_action(self, stage_info):
        if stage_info.action_type not in self.actions_map:
            raise LogicalError

        return self.actions_map.get_next_key(stage_info.action_type)

    def restart(self, stage_info):
        stage_info.action_type = self.actions_map.get_first_action_key()

    def make_transition(self, marker, stage_info, scenario):
        if marker.status == MarkerStatus.SUCCESS:
            stage_info.action_type = next_action = self.get_next_action(stage_info)

            if next_action is None:
                return Marker.success(message=marker.message)
            else:
                return Marker.in_progress(message=marker.message)
        else:
            return marker


def _stage_prepare(stage_info, scenario, *args, **kwargs):
    # stub prepare action, it is an error using it. Stages that use prepare action must provide own function
    raise LogicalError


def _stage_failure(stage_info, scenario, *args, **kwargs):
    # stub failure handler, noop
    return Marker.failure()


def _stage_completed(stage_info, scenario):
    # stub checker for stage completeness, noop
    return Marker.in_progress()


def _stage_has_more(stage_info, scenario):
    # stub repeat checker, always no repeat.
    return False


class SequentialActionsStrategyWithRepeat(IActionIterationStrategy):
    def __init__(
        self,
        actions,
        stage_prepare=_stage_prepare,
        stage_failure=_stage_failure,
        stage_completed=_stage_completed,
        stage_has_more=_stage_has_more,
    ):
        """

        :param stage_prepare: This function will be executed as a prepare action.
            If stage never enters prepare action, this function is never executed.
            Should return Marker.
        :param stage_failure: This function will be executed as a failure action.
            Failure action is terminal. That means, no other actions executed after a failure,
            no transitions from failure can be done.
            Failure action happens whenever `stage_completed` function returns a marker with `failure` status.
            Return value of this action will be used directly as the stage's return value,
            e.g. `success` marker finishes the stage.
            Should return Marker.
        :param stage_completed: This function will be executed to check if the stage has completed.
            It will be executed on every call to `make transition`.
            That is, it will be called after every action attempt, and if this function returns `success` or `failure`,
            then the action's status is discarded and the stage considered finished or failed.
            For the `success` marker, stage will be completed with provided marker.
            For the `failure` marker, `failure` action will assigned to the stage. See `stage_failure` param.
            Should return Marker.
        :param stage_has_more: This function will be executed to check if `actions` have to be iterated over again.
            This function will be called after all actions finished.
            If this function returns True, the strategy will start all `actions` again.
            The `prepare` action will not be executed.
            Should return boolean.
        """
        super().__init__()
        self._stage_prepare = stage_prepare
        self._stage_failure = stage_failure

        self._stage_completed = stage_completed
        self._stage_has_more = stage_has_more

        self.actions_map = ActionsMap(actions)

    def make_transition(self, marker, stage_info, scenario):
        if stage_info.action_type == StageAction.FAILURE:
            return marker

        completion_marker = self._stage_completed(stage_info, scenario)
        if completion_marker.status in {MarkerStatus.SUCCESS, MarkerStatus.FAILURE}:
            stage_info.action_type = StageAction.FAILURE if completion_marker.status == MarkerStatus.FAILURE else None
            return completion_marker

        if marker.status == MarkerStatus.SUCCESS:
            stage_info.action_type = next_action = self.get_next_action(stage_info)
            if next_action is None:
                if self._stage_has_more(stage_info, scenario):
                    self.restart(stage_info)
                    return Marker.in_progress(message=marker.message)
                else:
                    return Marker.success(message=marker.message)
            else:
                return Marker.in_progress(message=marker.message)
        else:
            return marker

    def get_next_action(self, stage_info):
        if stage_info.action_type == StageAction.PREPARE:
            return StageAction.ACTION

        if stage_info.action_type == StageAction.FAILURE:
            # stub, should never happen
            return None

        if stage_info.action_type not in self.actions_map:
            raise LogicalError

        return self.actions_map.get_next_key(stage_info.action_type)

    def get_current_func(self, stage_info):
        if stage_info.action_type == StageAction.PREPARE:
            return self._stage_prepare

        if stage_info.action_type == StageAction.FAILURE:
            return self._stage_failure

        if stage_info.action_type not in self.actions_map:
            raise LogicalError

        return self.actions_map[stage_info.action_type]

    def restart(self, stage_info):
        stage_info.action_type = self.actions_map.get_first_action_key()
