import dataclasses
from datetime import datetime

from walle.models import timestamp
from walle.scenario.common import get_request_to_maintenance_starting_time_using_yp_sla
from walle.scenario.definitions.base import get_data_storage
from walle.scenario.scenario import Scenario


@dataclasses.dataclass(frozen=True)
class MaintenancePlotCheckResult:
    check_passed: bool
    messages: list[str] = dataclasses.field(default_factory=list)


def check_if_maintenance_start_time_is_too_soon(scenario: Scenario) -> MaintenancePlotCheckResult:
    data_storage = get_data_storage(scenario)
    scenario_parameters = data_storage.read_scenario_parameters()

    if scenario_parameters.maintenance_start_time is None:
        return MaintenancePlotCheckResult(check_passed=True)

    host_groups_sources = data_storage.read_host_groups_sources()

    fail_messages = []
    for host_group_source in host_groups_sources:
        maintenance_plot = host_group_source.source.get_maintenance_plot()
        scenario_settings = maintenance_plot.get_scenario_settings(scenario.scenario_type)
        if scenario_settings.use_yp_sla:
            start_requesting_cms_timestamp = get_request_to_maintenance_starting_time_using_yp_sla(
                scenario_parameters.maintenance_start_time
            )
            too_soon = _check_if_requesting_cms_time_is_less_than_necessary(start_requesting_cms_timestamp)
            if too_soon:
                start_requesting_cms_time_str = datetime.fromtimestamp(start_requesting_cms_timestamp).strftime(
                    "%d.%m.%Y %H:%M"
                )
                now_time_str = datetime.fromtimestamp(timestamp()).strftime("%d.%m.%Y %H:%M")
                fail_messages.append(
                    "Maintenance start time is too soon for hosts belonging to ABC service {}, "
                    "minimum start time is {}, now is {}".format(
                        maintenance_plot.meta_info.abc_service_slug, start_requesting_cms_time_str, now_time_str
                    )
                )
        elif scenario_settings.approval_sla:
            required_offset = scenario_settings.approval_sla

            if required_offset is None:
                continue
            too_soon = _check_if_maintenance_start_time_is_too_soon_by_offset(
                scenario_parameters.maintenance_start_time,
                required_offset,
            )
            if too_soon:
                fail_messages.append(
                    (
                        "Maintenance start time is too soon for hosts belonging to ABC service %s, "
                        "required offset is %d seconds"
                    )
                    % (maintenance_plot.meta_info.abc_service_slug, required_offset)
                )

    return MaintenancePlotCheckResult(check_passed=True if not fail_messages else False, messages=fail_messages)


def _check_if_maintenance_start_time_is_too_soon_by_offset(maintenance_start_time: int, required_offset: int) -> bool:
    return (timestamp() + required_offset) > maintenance_start_time


def _check_if_requesting_cms_time_is_less_than_necessary(start_requesting_cms_timestamp: int) -> bool:
    return timestamp() > start_requesting_cms_timestamp
