import logging

import walle.projects
from walle.clients import qloud

log = logging.getLogger(__name__)


def _make_host_added(qloud_client, hostname, segment):
    host = qloud_client.find_host(hostname)
    if not host:
        qloud_client.add_host(hostname, segment)
        return None
    if host.segment != segment:
        return None
    return host


def _make_host_removed(qloud_client, hostname, comment):
    qloud_host = qloud_client.find_host(hostname)
    if qloud_host is None:
        # For safety reasons host should also be removed from qloud host resolver
        resolver_installation, resolver_segment = qloud_client.find_host_installation_segment(hostname)
        if resolver_installation is None and resolver_segment is None:
            return True
    else:
        if qloud_host.state != qloud.QloudHostStates.DOWN:
            qloud_client.set_qloud_host_state(qloud_host, qloud.QloudHostStates.DOWN, comment)
            return False
        if qloud_host.slots_count == 0:
            qloud_client.remove_qloud_host(qloud_host)
    return False


def make_host_added_ready(hostname, segment, comment):
    qloud_client = qloud.get_client()
    host = _make_host_added(qloud_client, hostname, segment)
    if not host:
        return False
    if not host.is_data_filled:
        qloud_client.set_qloud_host_state(host, qloud.QloudHostStates.INITIAL, comment)
        qloud_client.update_qloud_host_metadata(host)
        return False
    if host.state == qloud.QloudHostStates.UP:
        return True
    qloud_client.set_qloud_host_state(host, qloud.QloudHostStates.UP, comment)
    return False


def make_host_added_not_ready(hostname, segment, comment):
    qloud_client = qloud.get_client()
    host = _make_host_added(qloud_client, hostname, segment)
    if not host:
        return False
    qloud_client.set_qloud_host_state(host, qloud.QloudHostStates.INITIAL, comment)
    return True


def is_qloud_project(project):
    return qloud.get_project_tag() in set(walle.projects.get_by_id(project, ("tags",)).tags or [])


def ensure_host_not_in_qloud(host, comment):
    if not is_qloud_project(host.project):
        return True
    qloud_client = qloud.get_client()
    return _make_host_removed(qloud_client, host.name, comment)
