import dataclasses
import functools
import typing as tp

from sepelib.core.constants import HOUR_SECONDS
from walle import restrictions
from walle.clients import cms
from walle.clients.eine import ProfileMode
from walle.constants import NetworkTarget
from walle.hosts import HostState, HostStatus, HostOperationState
from walle.scenario.constants import (
    SOFT_NOC_LOAD_RETURN_TIMEOUT,
    SOFT_NOC_LOAD_RETURN_TIMEOUT_JUGGLER_SERVICE_NAME,
    SOFT_NOC_WORK_TIMEOUT,
    ScriptName,
    SchedulerName,
    TemplatePath,
    TICKET_RESOLUTION,
    TicketTransition,
    ConditionFunction,
    SharedDataKey,
    TicketTags,
    WORKMATE_STATUS_LABEL_NAME,
    WORKMATE_STATUS_TARGET_LABEL_VALUE,
)
from walle.scenario.errors import ScenarioValidationError
from walle.scenario.script_args import (
    AddHostsParams,
    IdleTimeParams,
    MaintenanceParams,
    SwitchProjectParams,
    NocSoftParams,
    get_missing_and_unknown_params,
    EmptyParams,
    HostsTransferParams,
    ReservedHostsTransferParams,
)
from walle.scenario.script_builder import ScriptBuilder
from walle.scenario.stage.acquire_permission import AcquirePermission
from walle.scenario.stage.approve_stage import ApproveStage
from walle.scenario.stage.conditional_host_stage import ConditionalHostStage
from walle.scenario.stage.detect_storage_stage import DetectStorageStage
from walle.scenario.stage.ensure_dns_access_stage import EnsureDnsAccessStage
from walle.scenario.stage.host_group_approve_stage import HostGroupApproveStage
from walle.scenario.stage.host_group_scheduler_stage import HostGroupSchedulerStage
from walle.scenario.stage.host_group_wait_before_requesting_cms_stage import (
    HostGroupWaitBeforeRequestingCmsStage,
    HostWaitBeforeRequestingCmsStage,
)
from walle.scenario.stage.lambda_stage import LambdaStage, FindActiveMacAddress
from walle.scenario.stage.maintenance_approvers_workflow_stage import MaintenanceApproversWorkflowStage
from walle.scenario.stage.noc_hard_host_group_approve_stage import (
    NocHardHostGroupApproveStage,
    ManualConfirmationHostGroupApproveStage,
)
from walle.scenario.stage.noc_maintenance_stage import NocMaintenanceStage, FinishNocMaintenanceStage
from walle.scenario.stage.scheduler_stage import (
    host_scheduler_stage,
    noc_maintenance_scheduler_stage,
    HOST_IS_ASSIGNED_READY,
    HOST_IS_ACQUIRED_BY_SCENARIO,
)
from walle.scenario.stage.timeout_stage import TimeoutStage
from walle.scenario.stages import (
    StageDesc,
    CheckAndReportAboutDeadlinesStage,
    CheckAndReportAboutHostRestrictionsStage,
)
from walle.scenario.stages import (
    ScenarioRootStage,
    NoopStage,
    HostRootStage,
    AddHostsStage,
    WaitForLabelOrTimeStage,
    SetAssignedStage,
    SetStartedWorkStatusStage,
    SetApprovementWorkStatusStage,
    SetReadyWorkStatusStage,
    SetFinishingWorkStatusStage,
    PrepareForWorkStage,
    SwitchScenarioToHostUUID,
    SwitchProjectHostStage,
    PrepareHostStage,
    AddStartrekMessageStage,
    AddStartrekTagStage,
    NoopHostStage,
    NoopHostStageSuccess,
    ExecuteTicketTransitionStage,
    PowerOffHostStage,
    WaitStateStatusHostStage,
    CancelTaskStage,
    LiberateFromQloudHostStage,
    RemoveHostsStage,
    WaitEineProfileHostStage,
    ITDCWorkflowStage,
    ProfileHostStage,
    SwitchVlansHostStage,
    CheckDnsHostStage,
    RebootHostStage,
    ReleaseHostStage,
    SetABCServiceIdStage,
    CollectHealthChecksStage,
    OptionalRedeployHostStage,
    SwitchToMaintenanceHostStage,
    SetAcquiringPermissionStatusStage,
    CheckAndSkipIfSecondOkEnabledStage,
    DumpApproversListStage,
    CallOnResponsibleForLoadReturn,
)
from walle.scenario.utils import BaseRegistry


class ScriptRegistry(BaseRegistry):

    ITEMS = {}

    @classmethod
    def register_script(cls, name, attrs_cls=None, uses_uuids=True):
        def registered_script(script):
            @functools.wraps(script)
            def wrapper_prepare_params(params, _attrs_cls=attrs_cls):
                params = params or {}

                if attrs_cls:
                    missing_params, unknown_params = get_missing_and_unknown_params(params, attrs_cls)
                    if missing_params or unknown_params:
                        error_msg = ""

                        if missing_params:
                            error_msg += "Required parameters are missing: {}; ".format(missing_params)
                        if unknown_params:
                            error_msg += "Unknown parameters were sent: {}; ".format(unknown_params)
                        raise ScenarioValidationError(error_msg)

                    try:
                        clean_params = attrs_cls(**params)
                    except TypeError as e:
                        raise ScenarioValidationError(str(e))

                else:
                    clean_params = EmptyParams()

                return script(clean_params)

            cls.ITEMS[name] = wrapper_prepare_params
            wrapper_prepare_params.name = name
            wrapper_prepare_params.uses_uuids = uses_uuids
            wrapper_prepare_params.attrs_cls = attrs_cls

            return wrapper_prepare_params

        return registered_script


@ScriptRegistry.register_script(name=ScriptName.WAIT, attrs_cls=SwitchProjectParams)
def wait_script(script_params):
    return ScenarioRootStage(
        [
            AcquirePermission(),
            host_scheduler_stage(
                [
                    HostRootStage(
                        [
                            WaitStateStatusHostStage(
                                target_project=script_params.target_project_id,
                                wait_state=HostState.ASSIGNED,
                                wait_status=HostStatus.READY,
                            )
                        ]
                    )
                ],
                schedule_type=script_params.schedule_type,
            ),
        ]
    )


@ScriptRegistry.register_script(name=ScriptName.NOOP)
def noop_script(_params):
    return ScenarioRootStage([NoopStage(), NoopStage()])


@ScriptRegistry.register_script(name=ScriptName.SET_MAINTENANCE, attrs_cls=MaintenanceParams)
def switch_to_maintenance_script(params):
    return ScenarioRootStage(
        [
            AcquirePermission(),
            host_scheduler_stage(
                [
                    HostRootStage(
                        [
                            SwitchToMaintenanceHostStage(
                                ignore_cms=params.ignore_cms,
                                operation_state=params.operation_state,
                                power_off=params.power_off,
                            ),
                        ]
                    )
                ],
                schedule_type=params.schedule_type,
            ),
        ]
    )


@ScriptRegistry.register_script(name=ScriptName.APPROVE, uses_uuids=False)
def approve_hosts_script(_params):
    return ScenarioRootStage(
        [
            AddStartrekMessageStage(template_path=TemplatePath.STARTREK_START_SCENARIO),
            ExecuteTicketTransitionStage(ticket_transition_state=TicketTransition.READY_FOR_DEV),
            ApproveStage(),
            ExecuteTicketTransitionStage(ticket_transition_state=TicketTransition.IN_PROGRESS),
            AddStartrekMessageStage(template_path=TemplatePath.STARTREK_EMERGENCY),
            AddStartrekMessageStage(template_path=TemplatePath.STARTREK_END_SCENARIO),
            ExecuteTicketTransitionStage(
                ticket_transition_state=TicketTransition.CLOSE, ticket_transition_resolution=TICKET_RESOLUTION
            ),
        ]
    )


@ScriptRegistry.register_script(name=ScriptName.WAIT_TIME, attrs_cls=IdleTimeParams)
def wait_time_script(params=None):
    return ScenarioRootStage(
        [
            AcquirePermission(),
            host_scheduler_stage(
                [HostRootStage([WaitForLabelOrTimeStage(idle_time=params.idle_time)])],
                schedule_type=params.schedule_type,
            ),
        ]
    )


@ScriptRegistry.register_script(name=ScriptName.HOSTS_ADD, attrs_cls=AddHostsParams, uses_uuids=False)
def hosts_add_script(params):
    return ScenarioRootStage(
        [
            AcquirePermission(total_started_scenarios_limit_name="max_started_host_add_scenarios"),
            AddStartrekMessageStage(template_path=TemplatePath.STARTREK_START_SCENARIO),
            ExecuteTicketTransitionStage(ticket_transition_state=TicketTransition.READY_FOR_DEV),
            ApproveStage(),
            AddStartrekMessageStage(template_path=TemplatePath.STARTREK_EMERGENCY),
            ExecuteTicketTransitionStage(ticket_transition_state=TicketTransition.IN_PROGRESS),
            WaitForLabelOrTimeStage(idle_time=HOUR_SECONDS),
            AddHostsStage(intermediate_project=params.intermediate_project),
            SwitchScenarioToHostUUID(),  # Could be moved to AddHostsStage but it's not very obvious
            host_scheduler_stage(
                [
                    HostRootStage(
                        [
                            SwitchToMaintenanceHostStage(
                                ignore_cms=params.ignore_cms,
                                power_off=True,
                                workdays_only=True,
                                operation_state=HostOperationState.DECOMMISSIONED,
                            ),
                            PowerOffHostStage(
                                ignore_cms=params.ignore_cms, operation_state=HostOperationState.DECOMMISSIONED
                            ),
                            WaitForLabelOrTimeStage(idle_time=params.idle_time),
                            LiberateFromQloudHostStage(),
                            SwitchProjectHostStage(
                                ignore_cms=params.ignore_cms,
                                target_project_id=params.target_project_id,
                                target_hardware_segment=params.target_hardware_segment,
                            ),
                            PrepareHostStage(
                                ignore_cms=params.ignore_cms,
                                target_project_id=params.target_project_id,
                                target_hardware_segment=params.target_hardware_segment,
                            ),
                        ]
                    )
                ],
                schedule_type=params.schedule_type,
            ),
            AddStartrekMessageStage(template_path=TemplatePath.STARTREK_END_SCENARIO),
            ExecuteTicketTransitionStage(
                ticket_transition_state=TicketTransition.CLOSE, ticket_transition_resolution=TICKET_RESOLUTION
            ),
        ]
    )


def get_prepare_transfer_host_stages(params):
    sb = ScriptBuilder()
    sb.add_stages(
        [
            SwitchToMaintenanceHostStage(
                ignore_cms=False,
                power_off=True,
                workdays_only=params.workdays_only,
                cms_task_action=cms.CmsTaskAction.REDEPLOY,  # WALLESUPPORT-1714
                operation_state=HostOperationState.DECOMMISSIONED,
            ),
            PowerOffHostStage(ignore_cms=False, operation_state=HostOperationState.DECOMMISSIONED),
            WaitForLabelOrTimeStage(idle_time=params.idle_time),
        ]
    )
    return sb.get_stages()


def get_transfer_host_stages(params):
    sb = ScriptBuilder()
    sb.add_stage(LiberateFromQloudHostStage())
    if params.delete:
        sb.add_stage(SwitchProjectHostStage(ignore_cms=False, target_project_id=params.intermediate_project))
        if params.abc_service_id:
            sb.add_stage(SetABCServiceIdStage(abc_service_id=params.abc_service_id))

    else:
        sb.add_stages(
            [
                SwitchProjectHostStage(
                    ignore_cms=False,
                    target_project_id=params.target_project_id,
                    target_hardware_segment=params.target_hardware_segment,
                ),
                ReleaseHostStage(ignore_cms=False),
                PrepareHostStage(
                    ignore_cms=False,
                    target_project_id=params.target_project_id,
                    target_hardware_segment=params.target_hardware_segment,
                ),
            ]
        )
    return sb.get_stages()


@ScriptRegistry.register_script(name=ScriptName.HOSTS_TRANSFER, attrs_cls=HostsTransferParams, uses_uuids=False)
def hosts_transfer_script(params):
    sb = ScriptBuilder()
    sb.add_stages(
        [
            SetAcquiringPermissionStatusStage(),
            AcquirePermission(total_started_scenarios_limit_name="max_started_host_add_scenarios"),
            AddStartrekTagStage(tag=TicketTags.CREATED),
            AddStartrekMessageStage(template_path=TemplatePath.STARTREK_START_SCENARIO),
            SetApprovementWorkStatusStage(),
            ApproveStage(),
            SetStartedWorkStatusStage(),
            AddStartrekTagStage(tag=TicketTags.APPROVED),
            AddStartrekMessageStage(template_path=TemplatePath.STARTREK_EMERGENCY),
            WaitForLabelOrTimeStage(idle_time=HOUR_SECONDS),
            AddHostsStage(intermediate_project=params.intermediate_project),
            SwitchScenarioToHostUUID(),  # Could be moved to AddHostsStage but it's not very obvious
            EnsureDnsAccessStage(),
        ]
    )

    if not params.delete:
        sb.add_stage(DetectStorageStage())

    sb.add_stage(AddStartrekTagStage(tag=TicketTags.PROCESSING))

    with sb.host_stages(host_scheduler_stage, schedule_type=SchedulerName.ALL) as hsb:
        hsb.add_stages(get_prepare_transfer_host_stages(params))
        hsb.add_stages(get_transfer_host_stages(params))

    if params.delete is True:
        sb.add_stage(RemoveHostsStage(intermediate_project=params.intermediate_project))

    sb.add_stages(
        [
            AddStartrekMessageStage(template_path=TemplatePath.STARTREK_END_SCENARIO),
        ]
    )
    return ScenarioRootStage(sb.get_stages())


@ScriptRegistry.register_script(name=ScriptName.RESERVED_HOSTS_TRANSFER, attrs_cls=ReservedHostsTransferParams)
def reserved_hosts_transfer_script(params):
    sb = ScriptBuilder()
    sb.add_stages(
        [
            SetAcquiringPermissionStatusStage(),
            AcquirePermission(total_started_scenarios_limit_name="max_started_reserved_hosts_transfer_scenarios"),
            AddStartrekTagStage(tag=TicketTags.CREATED),
            AddStartrekMessageStage(template_path=TemplatePath.STARTREK_START_SCENARIO),
            SetApprovementWorkStatusStage(),
            ApproveStage(),
            SetStartedWorkStatusStage(),
            AddStartrekTagStage(tag=TicketTags.APPROVED),
            AddStartrekMessageStage(template_path=TemplatePath.STARTREK_EMERGENCY),
        ]
    )

    sb.add_stage(AddStartrekTagStage(tag=TicketTags.PROCESSING))

    with sb.host_stages(host_scheduler_stage, schedule_type=SchedulerName.ALL) as hsb:
        hsb.add_stages(
            [
                SwitchProjectHostStage(
                    ignore_cms=False,
                    target_project_id=params.target_project_id,
                    target_hardware_segment=params.target_hardware_segment,
                    monitor_on_completion=True,
                    release=False,
                ),
                PrepareHostStage(
                    ignore_cms=False,
                    target_project_id=params.target_project_id,
                    target_hardware_segment=params.target_hardware_segment,
                ),
            ]
        )

    sb.add_stages(
        [
            AddStartrekMessageStage(template_path=TemplatePath.STARTREK_END_SCENARIO),
        ]
    )
    return ScenarioRootStage(sb.get_stages())


@ScriptRegistry.register_script(name=ScriptName.NOC_SOFT, attrs_cls=NocSoftParams)
def noc_soft_maintenance_script(params):
    return ScenarioRootStage(
        [
            AcquirePermission(
                total_started_scenarios_limit_name="max_started_maintenance_scenarios",
                dc_started_scenarios_limit="max_started_maintenance_scenarios_per_dc",
                urgently=params.urgently,
            ),
            NocMaintenanceStage(
                noc_maintenance_scheduler_stage(
                    [
                        HostRootStage(
                            [
                                CollectHealthChecksStage(),
                                PrepareForWorkStage(
                                    ignore_cms=params.ignore_cms, workdays_only=False, power_off=params.power_off
                                ),
                            ]
                        )
                    ],
                    host_readiness_str=HOST_IS_ASSIGNED_READY,
                ),
                execution_timeout=params.execution_time,
                work_timeout=SOFT_NOC_WORK_TIMEOUT
                # after SoftNocMaintenanceStage status is one of READY, REJECTED, FINISH{ING,ED} or CANCEL{ING,ED}
                # wait for READY to change into one of the other three.
            ),
            TimeoutStage(
                [
                    noc_maintenance_scheduler_stage(
                        [HostRootStage([CancelTaskStage(), SetAssignedStage(use_specific_checks=True)])],
                        schedule_type=SchedulerName.ONLY_PREVIOUSLY_ACQUIRED,
                        greedy=True,
                        host_readiness_str=HOST_IS_ACQUIRED_BY_SCENARIO,
                    ),
                    FinishNocMaintenanceStage(),
                ],
                SOFT_NOC_LOAD_RETURN_TIMEOUT,
                SOFT_NOC_LOAD_RETURN_TIMEOUT_JUGGLER_SERVICE_NAME,
            ),
        ]
    )


@dataclasses.dataclass
class FullProcessMessages:
    start_template_path: str
    ready_for_work_template_path: str
    restoring_after_work_started_template_path: str
    end_message_template_path: str


def full_process(
    total_started_scenarios_limit_name: str,
    full_process_messages: FullProcessMessages,
    power_off: bool,
    maintenance_approvers_workflow_stage_map: list[StageDesc],
    additional_host_stages: tp.Optional[list] = None,
):
    host_stages = [
        SwitchToMaintenanceHostStage(
            ignore_cms=False, power_off=False, workdays_only=False, operation_state=HostOperationState.DECOMMISSIONED
        ),
    ]
    if not additional_host_stages:
        additional_host_stages = []
    if power_off:
        host_stages.extend(
            [
                HostWaitBeforeRequestingCmsStage(
                    time_field_name="start_power_off_x_seconds_before_maintenance_start_time"
                ),
                PowerOffHostStage(ignore_cms=False, operation_state=HostOperationState.DECOMMISSIONED),
            ]
        )
    return ScenarioRootStage(
        [
            SetAcquiringPermissionStatusStage(),
            AcquirePermission(total_started_scenarios_limit_name, with_check_limits_by_maintenance_plots=True),
            AddStartrekMessageStage(template_path=full_process_messages.start_template_path),
            SetApprovementWorkStatusStage(),
            DumpApproversListStage(),
            ITDCWorkflowStage(
                [
                    MaintenanceApproversWorkflowStage(
                        [],
                        stage_map=maintenance_approvers_workflow_stage_map,
                    ),
                    SetReadyWorkStatusStage(),
                    AddStartrekMessageStage(template_path=full_process_messages.ready_for_work_template_path),
                    WaitForLabelOrTimeStage(
                        user_label=WORKMATE_STATUS_LABEL_NAME,
                        user_label_target_value=WORKMATE_STATUS_TARGET_LABEL_VALUE,
                    ),
                    SetFinishingWorkStatusStage(),
                    AddStartrekMessageStage(
                        template_path=full_process_messages.restoring_after_work_started_template_path
                    ),
                    MaintenanceApproversWorkflowStage(
                        [
                            CallOnResponsibleForLoadReturn(),
                            CheckAndSkipIfSecondOkEnabledStage(
                                [
                                    HostGroupSchedulerStage(
                                        [
                                            HostRootStage(
                                                [
                                                    WaitEineProfileHostStage(),
                                                    LambdaStage(
                                                        SharedDataKey.HOST_NETWORK_INFO, FindActiveMacAddress().name
                                                    ),
                                                    ProfileHostStage(
                                                        ignore_cms=False,
                                                        profile_mode=ProfileMode.SWP_UP,
                                                        monitor_on_completion=False,
                                                        skip_on_invalid_state=True,
                                                    ),
                                                    SwitchVlansHostStage(
                                                        network_target=NetworkTarget.PROJECT, skip_on_invalid_state=True
                                                    ),
                                                    CheckDnsHostStage(
                                                        monitor_on_completion=False, skip_on_invalid_state=True
                                                    ),
                                                    *additional_host_stages,
                                                    SetAssignedStage(power_on=power_off, skip_on_invalid_state=True),
                                                ]
                                            ),
                                        ]
                                    ),
                                ]
                            ),
                        ]
                    ),
                ]
            ),
            AddStartrekMessageStage(template_path=full_process_messages.end_message_template_path),
        ]
    )


@ScriptRegistry.register_script(name=ScriptName.ITDC_MAINTENANCE)
def itdc_maintenance_script(_params):
    total_started_scenarios_limit_name = "max_started_itdc_maintenance_scenarios"
    maintenance_approvers_workflow_stage_map = [
        StageDesc(stage_class=HostGroupApproveStage),
        StageDesc(
            stage_class=CheckAndReportAboutHostRestrictionsStage,
            params={
                'restrictions_to_check': [
                    restrictions.AUTOMATED_REBOOT,
                    restrictions.AUTOMATED_PROFILE,
                    restrictions.AUTOMATED_REDEPLOY,
                    restrictions.AUTOMATED_DNS,
                    restrictions.AUTOMATION,
                ],
            },
        ),
        StageDesc(
            stage_class=HostGroupWaitBeforeRequestingCmsStage,
            params={
                'time_field_name': 'request_cms_x_seconds_before_maintenance_start_time',
            },
        ),
        StageDesc(
            stage_class=CheckAndReportAboutDeadlinesStage,
            params={
                'children': [
                    StageDesc(
                        stage_class=HostGroupSchedulerStage,
                        params={
                            'children': [
                                StageDesc(
                                    stage_class=HostRootStage,
                                    params={
                                        'children': [
                                            StageDesc(
                                                stage_class=SwitchToMaintenanceHostStage,
                                                params={
                                                    'ignore_cms': False,
                                                    'power_off': False,
                                                    'workdays_only': False,
                                                    'operation_state': HostOperationState.DECOMMISSIONED,
                                                },
                                            ),
                                            StageDesc(
                                                stage_class=HostWaitBeforeRequestingCmsStage,
                                                params={
                                                    'time_field_name': "start_power_off_x_seconds_before_maintenance_start_time",
                                                },
                                            ),
                                            StageDesc(
                                                stage_class=PowerOffHostStage,
                                                params={
                                                    'ignore_cms': False,
                                                    'operation_state': HostOperationState.DECOMMISSIONED,
                                                },
                                            ),
                                        ]
                                    },
                                ),
                            ]
                        },
                    )
                ],
            },
            conditions={'if_any': [{'enable_manual_approval_after_hosts_power_off': False}]},
        ),
        StageDesc(
            stage_class=ManualConfirmationHostGroupApproveStage,
            params={
                'option_name': 'enable_manual_approval_after_hosts_power_off',
            },
            conditions={'if_any': [{'enable_manual_approval_after_hosts_power_off': True}]},
        ),
    ]

    return full_process(
        total_started_scenarios_limit_name,
        FullProcessMessages(
            TemplatePath.STARTREK_START_ITDC_MAINTENANCE_SCENARIO,
            TemplatePath.STARTREK_ITDC_MAINTENANCE_READY_FOR_WORK,
            TemplatePath.STARTREK_ITDC_MAINTENANCE_HOST_REVIVAL_STARTED,
            TemplatePath.STARTREK_END_ITDC_MAINTENANCE_SCENARIO,
        ),
        power_off=True,
        maintenance_approvers_workflow_stage_map=maintenance_approvers_workflow_stage_map,
        additional_host_stages=[
            ConditionalHostStage(
                [RebootHostStage(ignore_cms=False)], condition_func=ConditionFunction.ITDC_MAINTENANCE_REBOOT_NEEDED
            ),
            ConditionalHostStage(
                [OptionalRedeployHostStage(ignore_cms=False, skip_on_invalid_state=True)],
                condition_func=ConditionFunction.ITDC_MAINTENANCE_REDEPLOY_NEEDED,
            ),
        ],
    )


@ScriptRegistry.register_script(name=ScriptName.NOC_HARD)
def noc_hard_script(_params):
    total_started_scenarios_limit_name = "max_started_noc_hard_scenarios"
    maintenance_approvers_workflow_stage_map = [
        StageDesc(stage_class=NocHardHostGroupApproveStage),
        StageDesc(
            stage_class=CheckAndReportAboutHostRestrictionsStage,
            params={
                'restrictions_to_check': [
                    restrictions.AUTOMATED_REBOOT,
                    restrictions.AUTOMATED_PROFILE,
                    restrictions.AUTOMATED_REDEPLOY,
                    restrictions.AUTOMATED_DNS,
                    restrictions.AUTOMATION,
                ],
            },
        ),
        StageDesc(
            stage_class=HostGroupWaitBeforeRequestingCmsStage,
            params={
                'time_field_name': 'request_cms_x_seconds_before_maintenance_start_time',
            },
        ),
        StageDesc(
            stage_class=CheckAndReportAboutDeadlinesStage,
            params={
                'children': [
                    StageDesc(
                        stage_class=HostGroupSchedulerStage,
                        params={
                            'children': [
                                StageDesc(
                                    stage_class=HostRootStage,
                                    params={
                                        'children': [
                                            StageDesc(
                                                stage_class=SwitchToMaintenanceHostStage,
                                                params={
                                                    'ignore_cms': False,
                                                    'power_off': False,
                                                    'workdays_only': False,
                                                    'operation_state': HostOperationState.DECOMMISSIONED,
                                                },
                                            )
                                        ]
                                    },
                                ),
                            ]
                        },
                    )
                ],
            },
            conditions={'if_any': [{'enable_manual_approval_after_hosts_power_off': False}]},
        ),
        StageDesc(
            stage_class=ManualConfirmationHostGroupApproveStage,
            params={
                'option_name': 'enable_manual_approval_after_hosts_power_off',
            },
            conditions={'if_any': [{'enable_manual_approval_after_hosts_power_off': True}]},
        ),
    ]

    return full_process(
        total_started_scenarios_limit_name,
        FullProcessMessages(
            TemplatePath.STARTREK_START_NOC_HARD_SCENARIO,
            TemplatePath.STARTREK_NOC_HARD_HOST_READY_FOR_WORK,
            TemplatePath.STARTREK_NOC_HARD_HOST_REVIVAL_STARTED,
            TemplatePath.STARTREK_END_NOC_HARD_SCENARIO,
        ),
        power_off=False,
        maintenance_approvers_workflow_stage_map=maintenance_approvers_workflow_stage_map,
    )


@ScriptRegistry.register_script(name=ScriptName.SET_MAINTENANCE_WITH_GROUPS_APPROVES)
def switch_to_maintenance_with_groups_approves_script(_params):
    """
    Sample scenario script to demonstrate how parallel execution of stages work.
    """
    return ScenarioRootStage(
        [
            AddStartrekMessageStage(template_path=TemplatePath.STARTREK_START_ITDC_MAINTENANCE_SCENARIO),
            ITDCWorkflowStage(
                [
                    MaintenanceApproversWorkflowStage(
                        [
                            HostGroupApproveStage(),
                            HostGroupSchedulerStage(
                                [
                                    HostRootStage(
                                        [
                                            SwitchToMaintenanceHostStage(
                                                ignore_cms=False,
                                                power_off=True,
                                                workdays_only=False,
                                                operation_state=HostOperationState.DECOMMISSIONED,
                                            ),
                                        ]
                                    )
                                ]
                            ),
                        ]
                    ),
                ]
            ),
            AddStartrekMessageStage(template_path=TemplatePath.STARTREK_END_ITDC_MAINTENANCE_SCENARIO),
        ]
    )


@ScriptRegistry.register_script(name=ScriptName.BENCHMARK, attrs_cls=EmptyParams, uses_uuids=False)
def benchmark_script(params):
    sb = ScriptBuilder()
    sb.add_stages(
        [
            AddStartrekMessageStage(template_path=TemplatePath.STARTREK_START_SCENARIO),
            SetStartedWorkStatusStage(),
            SwitchScenarioToHostUUID(),
        ]
    )

    with sb.host_stages(host_scheduler_stage, schedule_type=SchedulerName.ALL) as hsb:
        inner_sb = ScriptBuilder()
        inner_sb.add_stages(
            [
                NoopHostStageSuccess(),
                NoopHostStage(),
            ]
        )
        hsb.add_stages(inner_sb.get_stages())

    return ScenarioRootStage(sb.get_stages())


@ScriptRegistry.register_script(name=ScriptName.YT_MAINTENANCE, attrs_cls=EmptyParams)
def yt_maintenance_script(params):
    return ScenarioRootStage(
        [
            SetAcquiringPermissionStatusStage(),
            AcquirePermission("max_started_yt_maintenance_scenarios", with_check_limits_by_maintenance_plots=True),
            host_scheduler_stage(
                [
                    HostRootStage(
                        [
                            SwitchToMaintenanceHostStage(
                                ignore_cms=False,
                                power_off=False,
                                workdays_only=False,
                                operation_state=HostOperationState.DECOMMISSIONED,
                            ),
                        ]
                    )
                ],
                schedule_type=SchedulerName.ALL,
            ),
            SetReadyWorkStatusStage(),
            WaitForLabelOrTimeStage(
                user_label=WORKMATE_STATUS_LABEL_NAME,
                user_label_target_value=WORKMATE_STATUS_TARGET_LABEL_VALUE,
            ),
            SetFinishingWorkStatusStage(),
            host_scheduler_stage(
                [
                    HostRootStage(
                        [
                            SetAssignedStage(power_on=True, monitor_on_completion=False),
                        ]
                    )
                ],
                schedule_type=SchedulerName.ALL,
            ),
        ]
    )
