import typing

import attr

from sepelib.core import config
from sepelib.core.constants import DAY_SECONDS, MINUTE_SECONDS
from walle import projects
from walle.clients.qloud import get_segment_project
from walle.errors import BadRequestError
from walle.hosts import HostOperationState
from walle.models import timestamp
from walle.scenario.constants import SchedulerName, TemplatePath
from walle.scenario.errors import ScenarioValidationError

MIN_IDLE_TIME = 30
MAX_IDLE_TIME = 14 * DAY_SECONDS
# TODO Clarify periods
MIN_WORKS_TIME = 30 * MINUTE_SECONDS
MAX_WORKS_TIME = 60 * MINUTE_SECONDS
DEFAULT_NOC_SOFT_AWAIT_TIME = 37 * MINUTE_SECONDS


def get_missing_and_unknown_params(params, attrs_cls):
    if attrs_cls:
        required_params = {p.name for p in filter(lambda x: x.default is attr.NOTHING, attr.fields(attrs_cls))}
        all_params = {p.name for p in attr.fields(attrs_cls)}
    else:
        all_params = set()
        required_params = set()

    specified_params = set(params.keys())

    return list(required_params - specified_params), list(specified_params - all_params)


def check_project_exists_and_automation_enabled(instance, attribute, value):
    try:
        project = projects.get_by_id(project_id=value)
        if not project.has_fully_enabled_automation():
            raise ScenarioValidationError("Project '{}' does not have enabled dns or healing automation".format(value))
    except projects.ProjectNotFoundError:
        raise ScenarioValidationError("Project id '{}' does not exist".format(value))


def check_between(min_value, max_value):
    def validator(instance, attribute, value):
        if not (min_value < value < max_value):
            raise ScenarioValidationError(
                "Parameter '{}' must be in range of {} and {} (got {})".format(
                    attribute.name, min_value, max_value, value
                )
            )

    return validator


def check_in(check_set):
    def validator(instance, attribute, value):
        if value not in check_set:
            raise ScenarioValidationError(
                "Parameter '{}' must be in {} (got {})".format(attribute.name, check_set, value)
            )

    return validator


def check_works_timestamp(instance, attribute, value):
    now = timestamp()
    check_between(now + MIN_WORKS_TIME, now + MAX_WORKS_TIME)(instance, attribute, value)


def check_qloud_segment_exists(instance, attribute, value):
    if value is not None and ('.' not in value or get_segment_project(value) is None):
        raise ScenarioValidationError("Invalid Qloud segment. Got '{}'".format(value))


def check_optional_segment_backup_project(instance, attribute, value):
    if instance.target_hardware_segment is not None:
        check_qloud_segment_exists(instance, attribute, instance.target_hardware_segment)
    elif instance.target_project_id is not None:
        check_project_exists_and_automation_enabled(instance, attribute, instance.target_project_id)
    else:
        raise ScenarioValidationError("You must specify target_project_id or target_hardware_segment")


def check_optional_project_backup_segment(instance, attribute, value):
    if instance.target_project_id is not None:
        check_project_exists_and_automation_enabled(instance, attribute, instance.target_project_id)
    elif instance.target_hardware_segment is not None:
        check_qloud_segment_exists(instance, attribute, instance.target_hardware_segment)
    else:
        raise ScenarioValidationError("You must specify target_project_id or target_hardware_segment")


def check_transfer_project(instance, attribute, value):
    if not instance.delete:
        check_optional_segment_backup_project(instance, attribute, value)
    elif instance.target_project_id is not None:
        raise ScenarioValidationError("Don't specify target_project_id when deleting hosts.")


def check_transfer_segment(instance, attribute, value):
    if not instance.delete:
        check_optional_segment_backup_project(instance, attribute, value)
    elif instance.target_hardware_segment is not None:
        raise ScenarioValidationError("Don't specify target_hardware_segment when deleting hosts.")


def get_params_flag(params, param_name):
    return getattr(params, param_name, None) is True


@attr.s
class BaseParams:
    @staticmethod
    def get_json_schema():
        return {
            "type": "object",
            "properties": {},
            "additionalProperties": True,
            "description": "Some undefined params",
        }

    def get_cms_params(self):
        return {}


@attr.s
class SchedulerParams(BaseParams):
    schedule_type = attr.ib(default=SchedulerName.ALL, type=str, validator=check_in(SchedulerName.CHOICES))


@attr.s
class IdleTimeParams(BaseParams):
    target_project_id = attr.ib(type=str, validator=check_project_exists_and_automation_enabled)
    idle_time = attr.ib(default=DAY_SECONDS, type=int, validator=check_between(MIN_IDLE_TIME, MAX_IDLE_TIME))
    schedule_type = attr.ib(default=SchedulerName.ALL, type=str, validator=check_in(SchedulerName.CHOICES))


@attr.s
class AddHostsParams(BaseParams):
    target_project_id = attr.ib(default=None, type=str, metadata={}, validator=check_optional_project_backup_segment)
    target_hardware_segment = attr.ib(
        default=None, type=str, metadata={}, validator=check_optional_segment_backup_project
    )
    idle_time = attr.ib(default=DAY_SECONDS, type=int, validator=check_between(MIN_IDLE_TIME, MAX_IDLE_TIME))
    ignore_cms = attr.ib(default=False, type=bool)
    intermediate_project = attr.ib(type=str)
    schedule_type = attr.ib(default=SchedulerName.ALL, type=str, validator=check_in(SchedulerName.CHOICES))
    power_off = attr.ib(default=True, type=bool)
    template_path = attr.ib(type=str, default=TemplatePath.STARTREK_EMERGENCY)
    operation_state = attr.ib(type=str, default=HostOperationState.DECOMMISSIONED)

    @intermediate_project.default
    def _load_intermediate_project_from_config(self):
        return config.get_value("scenario.stages.add_hosts_stage.project")


@attr.s
class HostsTransferParams(BaseParams):
    target_project_id = attr.ib(default=None, type=str, metadata={}, validator=check_transfer_project)
    target_hardware_segment = attr.ib(default=None, type=str, metadata={}, validator=check_transfer_segment)
    idle_time = attr.ib(default=DAY_SECONDS, type=int, validator=check_between(MIN_IDLE_TIME, MAX_IDLE_TIME))
    intermediate_project = attr.ib(type=str)
    delete = attr.ib(default=False, type=bool)
    abc_service_id = attr.ib(default=None, type=int)
    workdays_only = attr.ib(default=True, type=bool)

    @intermediate_project.default
    def _load_intermediate_project_from_config(self):
        return config.get_value("scenario.stages.add_hosts_stage.project")

    @staticmethod
    def get_json_schema():
        return {
            "type": "object",
            "properties": {
                "target_project_id": {
                    "type": "string",
                    "minLength": 1,
                    "description": "The project where the hosts need to be moved",
                },
                "target_hardware_segment": {
                    "type": "string",
                    "minLength": 1,
                    "description": "The Qloud segment where the hosts need to be moved",
                },
                "idle_time": {
                    "type": "integer",
                    "minimum": MIN_IDLE_TIME,
                    "maximum": MAX_IDLE_TIME,
                    "description": "how long to wait after approval",
                },
                "workdays_only": {
                    "type": "boolean",
                    "description": "Continue the process only on workdays",
                },
                "abc_service_id": {
                    "type": "integer",
                    "description": "Transfer hosts to the balance of this service after being removed from Wall-e",
                },
            },
            "additionalProperties": True,
            "description": "Host transfer params",
        }


@attr.s
class ReservedHostsTransferParams(BaseParams):
    target_project_id = attr.ib(default=None, type=str, metadata={}, validator=check_transfer_project)
    target_hardware_segment = attr.ib(default=None, type=str, metadata={}, validator=check_transfer_segment)
    delete = attr.ib(default=False, type=bool)

    @staticmethod
    def get_json_schema():
        return {
            "type": "object",
            "properties": {
                "target_project_id": {
                    "type": "string",
                    "minLength": 1,
                    "description": "The project where the hosts need to be moved",
                },
                "target_hardware_segment": {
                    "type": "string",
                    "minLength": 1,
                    "description": "The Qloud segment where the hosts need to be moved",
                },
            },
            "additionalProperties": True,
            "description": "Reserved hosts transfer params",
        }


@attr.s
class MaintenanceParams(BaseParams):
    schedule_type = attr.ib(default=SchedulerName.ALL, type=str, validator=check_in(SchedulerName.CHOICES))
    ignore_cms = attr.ib(default=False, type=bool)
    power_off = attr.ib(default=True, type=bool)
    operation_state = attr.ib(type=str, default=HostOperationState.DECOMMISSIONED)


@attr.s
class SwitchProjectParams(BaseParams):
    target_project_id = attr.ib(type=str, validator=check_project_exists_and_automation_enabled)
    schedule_type = attr.ib(default=SchedulerName.ALL, type=str, validator=check_in(SchedulerName.CHOICES))


@attr.s
class NocSoftParams(BaseParams):
    switch = attr.ib(default=None, type=str)
    project_id = attr.ib(default=None, type=str)
    execution_time = attr.ib(
        default=DEFAULT_NOC_SOFT_AWAIT_TIME, type=int, validator=check_between(MIN_WORKS_TIME - 1, MAX_WORKS_TIME)
    )
    schedule_type = attr.ib(default=SchedulerName.ALL, type=str, validator=check_in(SchedulerName.CHOICES))
    ignore_cms = attr.ib(default=False, type=bool)
    power_off = attr.ib(default=False, type=bool)
    urgently = attr.ib(default=False, type=bool)

    @staticmethod
    def get_json_schema():
        return {
            "type": "object",
            "properties": {
                "switch": {
                    "type": "string",
                    "minLength": 1,
                    "description": "Target switch",
                },
                "project_id": {
                    "type": "string",
                    "minLength": 1,
                    "description": "Filter hosts by given project id",
                },
            },
            "additionalProperties": True,
            "description": "NocSoft params",
            "required": ["switch"],
        }


@attr.s
class ItdcMaintenanceParams(BaseParams):
    maintenance_start_time: typing.Optional[int] = attr.ib(
        default=None, validator=[attr.validators.optional(attr.validators.instance_of(int))]
    )
    maintenance_end_time: typing.Optional[int] = attr.ib(
        default=None, validator=[attr.validators.optional(attr.validators.instance_of(int))]
    )
    rack: typing.Optional[str] = attr.ib(
        default=None, validator=[attr.validators.optional(attr.validators.instance_of(str))]
    )

    def __attrs_post_init__(self):
        if (
            self.maintenance_start_time
            and self.maintenance_end_time
            and self.maintenance_start_time > self.maintenance_end_time
        ):
            raise BadRequestError("Please, enter correct maintenance end time.")

    @staticmethod
    def get_json_schema():
        return {
            "type": "object",
            "properties": {
                "rack": {
                    "type": "string",
                    "minLength": 1,
                    "description": "Target rack",
                },
                "maintenance_start_time": {
                    "type": "string",
                    "description": "Switch unloading start time",
                },
                "maintenance_end_time": {
                    "type": "string",
                    "description": "Switch unloading end time",
                },
            },
            "additionalProperties": True,
            "description": "ItdcMaintenance params",
            "required": ["maintenance_start_time"],
        }

    def get_cms_params(self):
        return {
            "maintenance_start_time": self.maintenance_start_time,
            "maintenance_end_time": self.maintenance_end_time,
        }


@attr.s
class NocHardParams(BaseParams):
    switch: typing.Optional[str] = attr.ib(
        default=None, validator=[attr.validators.optional(attr.validators.instance_of(str))]
    )
    maintenance_start_time: typing.Optional[int] = attr.ib(
        default=None, validator=[attr.validators.optional(attr.validators.instance_of(int))]
    )
    maintenance_end_time: typing.Optional[int] = attr.ib(
        default=None, validator=[attr.validators.optional(attr.validators.instance_of(int))]
    )

    def __attrs_post_init__(self):
        if (
            self.maintenance_start_time
            and self.maintenance_end_time
            and self.maintenance_start_time > self.maintenance_end_time
        ):
            raise BadRequestError("Please, enter correct maintenance end time.")

    @staticmethod
    def get_json_schema():
        return {
            "type": "object",
            "properties": {
                "switch": {
                    "type": "string",
                    "minLength": 1,
                    "description": "Target switch",
                },
                "maintenance_start_time": {
                    "type": "string",
                    "description": "Switch unloading start time",
                },
                "maintenance_end_time": {
                    "type": "string",
                    "description": "Switch unloading end time",
                },
            },
            "additionalProperties": True,
            "description": "NocHard params",
        }

    def get_cms_params(self):
        return {
            "maintenance_start_time": self.maintenance_start_time,
            "maintenance_end_time": self.maintenance_end_time,
        }


@attr.s(frozen=True)
class EmptyParams(BaseParams):
    pass
