import collections
import datetime
import itertools
import logging
import time

from cachetools.func import ttl_cache

from sepelib.core import constants, config
from sepelib.core.constants import WEEK_SECONDS
from walle.clients import abc, bot, startrek, ok
from walle.models import timestamp
from walle.scenario.constants import TicketStatus, TemplatePath, StageName
from walle.scenario.error_handlers import collect_exc_context_for_approve_stage_stage, transform_to_stage_error
from walle.scenario.errors import ErrorList, HostDoesNotHaveBotProjectIDError
from walle.scenario.iteration_strategy import SequentialIterationActionStrategy
from walle.scenario.marker import Marker, MarkerStatus
from walle.scenario.mixins import Stage
from walle.scenario.stages import StageRegistry
from walle.util.approvement_tools import ApprovalTicketResolution
from walle.util.template_loader import JinjaTemplateRenderer

log = logging.getLogger(__name__)

APPROVAL_ISSUE_TYPE = "approval"
PRODUCT_HEAD = "Product head"
HARDWARE_RESOURCES_OWNER = "Hardware resources owner"
ASSIGNEE = "Ticket assignee"

_UNKNOWN = "UNKNOWN VALUE"


class ApproveInfo:
    def __init__(
        self,
        responsibles=None,
        invs=None,
        ticket_key=None,
        comment_url=None,
        role_types=None,
        ticket_status=None,
        approvement_uuid=None,
        approvement_status=None,
        is_approvement_resolved=None,
        service_id=None,
        is_owners_unknown=False,
    ):
        self.responsibles = responsibles
        self.invs = invs
        self.role_types = role_types
        self.ticket_key = ticket_key
        self.comment_url = comment_url
        self.ticket_status = ticket_status
        self.approvement_uuid = approvement_uuid
        self.approvement_status = approvement_status
        self.is_approvement_resolved = is_approvement_resolved
        self.service_id = service_id
        self.is_owners_unknown = is_owners_unknown

    def __eq__(self, other):
        return self.__dict__ == other.__dict__

    def __ne__(self, other):
        return not self.__eq__(other)

    def _to_dict(self):
        return {
            "responsibles": self.responsibles,
            "invs": self.invs,
            "role_types": self.role_types,
            "ticket_key": self.ticket_key,
            "comment_url": self.comment_url,
            "ticket_status": self.ticket_status,
            "approvement_uuid": self.approvement_uuid,
            "approvement_status": self.approvement_status,
            "is_approvement_resolved": self.is_approvement_resolved,
            "service_id": self.service_id,
            "is_owners_unknown": self.is_owners_unknown,
        }


class HostInfo:
    def __init__(
        self, service_id=None, owners=None, timestamp=None, bot_project_id=None, host_inv=None, role_type=None
    ):
        self.service_id = service_id
        self.owners = owners
        self.timestamp = timestamp
        self.bot_project_id = bot_project_id
        self.host_inv = host_inv
        self.role_type = role_type

    def __eq__(self, other):
        return self.__dict__ == other.__dict__

    def __ne__(self, other):
        return not self.__eq__(other)

    def _to_dict(self):
        return {
            "service_id": self.service_id,
            "owners": self.owners,
            "timestamp": self.timestamp,
            "bot_project_id": self.bot_project_id,
            "host_inv": self.host_inv,
            "role_type": self.role_type,
        }


class HostDiff:
    def __init__(self, inv):
        self.inv = inv

        self.initial_responsibles = []
        self.initial_bpid = None

        self.current_responsibles = []
        self.current_bpid = None

    def __eq__(self, other):
        return self.__dict__ == other.__dict__

    def __ne__(self, other):
        return not self.__eq__(other)

    def _set_vars(self, initial_responsibles=None, current_responsibles=None, initial_bpid=None, current_bpid=None):
        self.initial_responsibles = initial_responsibles
        self.current_responsibles = current_responsibles
        self.initial_bpid = initial_bpid
        self.current_bpid = current_bpid

    @property
    def bold(self):
        return self.initial_responsibles != self.current_responsibles or self.initial_bpid != self.current_bpid

    def _to_dict(self):
        return dict(
            inv=self.inv,
            initial_responsibles=self.initial_responsibles,
            current_responsibles=self.current_responsibles,
            initial_bpid=self.initial_bpid,
            current_bpid=self.current_bpid,
            bold=self.bold,
        )


@StageRegistry.register(StageName.ApproveStage)
class ApproveStage(Stage):
    """Creates approves in Startrek and waits for confirmation"""

    IS_ROLLBACK_SET = "is_rollback_set"

    START_FROM_BEGINNING = "start_from_beggining"

    IS_CANCEL_SET = "is_cancel_set"

    STAGE_TMP_DATA = "tmp_data"
    RESOLVED_HOSTS = "resolved_hosts"

    INITIAL_APPROVE_INFOS = "initial_approve_info"
    APPROVE_INFOS_FOR_CHECK = "approve_infos_for_check"

    BOT_PROJECT_IDS_INITIAL = "hosts_bot_project_ids_initial"
    BOT_PROJECT_IDS_FOR_CHECK = "hosts_bot_project_ids_for_check"

    LAST_OWNERS_GATHERING_TIMESTAMP = "gathering_timestamp"

    def __init__(self):
        self.iteration_strategy = SequentialIterationActionStrategy(
            [
                self.action,
                self.create_tickets,
                self.create_approvements,
                self.add_comments,
                self.check_approvement_resolution,
                self.close_linked_tickets,
            ]
        )
        super().__init__()
        self.errors = []

    def run(self, stage_info, scenario, *args, **kwargs):
        if config.get_value("scenario.rtc_approve_stage.skip"):
            return Marker.success()

        if self._is_all_data_self_contained(stage_info, scenario):
            return Marker.success()

        if not self._check_owners(stage_info, scenario):
            return Marker.in_progress()

        if self._is_diff_message_needed(stage_info):
            self._add_message_with_diff(stage_info, scenario)

        if self._is_restart_needed(stage_info):
            self._rollback(stage_info, scenario)
            self.iteration_strategy.restart(stage_info)

        if self._is_cancel_needed(stage_info):
            self._cancel(scenario)
            return Marker.in_progress()

        func = self.iteration_strategy.get_current_func(stage_info)
        marker = func(stage_info, scenario, *args, **kwargs)

        self._check_errors(stage_info, scenario)

        return self.iteration_strategy.make_transition(marker, stage_info, scenario)

    def _check_errors(self, stage_info, scenario):
        if self.errors:
            stage_errors = []
            for idx, error in enumerate(self.errors):
                exc_id, exc_context = collect_exc_context_for_approve_stage_stage(error, stage_info, idx)
                stage_error = transform_to_stage_error(scenario, exc_id, exc_context, error)
                stage_errors.append(stage_error)
            raise ErrorList(stage_errors)

    def _check_owners(self, stage_info, scenario):
        if self._is_time_for_check_owners(stage_info):
            check_owners_marker = self._collect_approve_infos_for_check(stage_info, scenario)

            if check_owners_marker.status != MarkerStatus.SUCCESS:
                return False

            if not self._is_owners_and_ids_the_same(stage_info):
                stage_info.set_data(self.IS_ROLLBACK_SET, True)

            stage_info.set_data(self.LAST_OWNERS_GATHERING_TIMESTAMP, timestamp())

        return True

    def action(self, stage_info, scenario):
        result = self._collect_owners_of_hosts(
            stage_info,
            scenario_hosts=scenario.hosts,
            hosts_owners_key=self.INITIAL_APPROVE_INFOS,
            hosts_bot_project_ids_key=self.BOT_PROJECT_IDS_INITIAL,
            ticket_key=scenario.ticket_key,
        )
        stage_info.set_data(self.LAST_OWNERS_GATHERING_TIMESTAMP, timestamp())
        return result

    def create_tickets(self, stage_info, scenario):
        startrek_client = startrek.get_client()
        approve_infos = self._get_approve_data(stage_info)

        without_errors = True
        if len(approve_infos) == 1:
            # we don't need any more tickets, just use main one
            approve_infos[0].ticket_key = scenario.ticket_key
        else:
            for approve_info in approve_infos:
                try:
                    if approve_info.ticket_key:
                        continue
                    prepared_data = self._prepare_data_for_ticket_creation(
                        scenario.ticket_key, approve_info.responsibles
                    )
                    child_ticket_key = startrek_client.create_issue(prepared_data)['key']
                    approve_info.ticket_key = child_ticket_key

                except startrek.StartrekClientError as e:
                    self.errors.append(e)
                    without_errors = False

        self._set_approve_data(stage_info, approve_infos)

        if without_errors:
            return Marker.success()
        return Marker.in_progress()

    def create_approvements(self, stage_info, scenario):
        ok_client = ok.get_client()

        approve_infos = self._get_approve_data(stage_info)
        without_errors = True

        for approve_info in approve_infos:

            if approve_info.approvement_uuid:
                continue

            try:
                prepared_data = self._prepare_data_for_approvement_creation(
                    approve_info.ticket_key, responsible=approve_info.responsibles, hosts_invs=approve_info.invs
                )
                approvement = ok_client.create_approvement(**prepared_data)
                log.info("Got approvement uuid {} for scenario {}".format(approvement.uuid, scenario.scenario_id))
                approve_info.approvement_uuid = approvement.uuid
                approve_info.approvement_status = ok.ApprovementStatus.IN_PROGRESS
            except ok.OKError as e:
                self.errors.append(e)
                without_errors = False

        self._set_approve_data(stage_info, approve_infos)

        if without_errors:
            return Marker.success()
        return Marker.in_progress()

    def add_comments(self, stage_info, scenario):
        startrek_client = startrek.get_client()
        approve_infos = self._get_approve_data(stage_info)

        without_errors = True
        for approve_info in approve_infos:

            if approve_info.comment_url:
                continue

            try:
                prepared_data = self._prepare_data_for_approve_comment(approve_info, scenario.ticket_key)
                response = startrek_client.add_comment(**prepared_data)
                approve_info.comment_url = response['self']
            except startrek.StartrekClientError as e:
                self.errors.append(e)
                without_errors = False

        self._set_approve_data(stage_info, approve_infos)

        if without_errors:
            return Marker.success()
        return Marker.in_progress()

    def check_approvement_resolution(self, stage_info, scenario):
        ok_client = ok.get_client()
        approve_infos = self._get_approve_data(stage_info)

        for approve_info in approve_infos:

            if approve_info.is_approvement_resolved:
                continue

            if approve_info.approvement_status == ok.ApprovementStatus.IN_PROGRESS:

                try:
                    approvement = ok_client.get_approvement(approve_info.approvement_uuid)
                    if approvement.is_approved:
                        approve_info.is_approvement_resolved = True
                        approve_info.approvement_status = ok.ApprovementStatus.CLOSED

                    elif approvement.status in [ok.ApprovementStatus.REJECTED, ok.ApprovementStatus.CLOSED]:
                        stage_info.set_data(self.IS_ROLLBACK_SET, True)
                        stage_info.set_data(self.IS_CANCEL_SET, True)
                        stage_info.set_data(self.LAST_OWNERS_GATHERING_TIMESTAMP, 0)
                        approve_info.approvement_status = approvement.status

                except ok.OKError as e:
                    self.errors.append(e)

        self._set_approve_data(stage_info, approve_infos)

        if all([approve_info.is_approvement_resolved for approve_info in approve_infos]):
            return Marker.success()
        return Marker.in_progress()

    def close_linked_tickets(self, stage_info, scenario):
        startrek_client = startrek.get_client()
        approve_infos = self._get_approve_data(stage_info)

        is_local_closure = len(approve_infos) == 1
        for approve_info in approve_infos:
            try:
                self._close_ticket(scenario, startrek_client, approve_info, is_local_closure)
            except Exception as e:
                self.errors.append(e)

        self._set_approve_data(stage_info, approve_infos)

        if all(approve_info.ticket_status == TicketStatus.CLOSED for approve_info in approve_infos):
            return Marker.success()
        return Marker.in_progress()

    def _rollback(self, stage_info, scenario):
        ok_client = ok.get_client()
        startrek_client = startrek.get_client()
        approve_infos = self._get_approve_data(stage_info)

        is_local_closure = len(approve_infos) == 1
        is_all_cleaned_up = True

        for approve_info in approve_infos:
            self._close_approvement(scenario, ok_client, approve_info)
            self._close_ticket(
                scenario, startrek_client, approve_info, is_local_closure, resolution=ApprovalTicketResolution.REFUSAL
            )

            if (
                approve_info.ticket_status != TicketStatus.CLOSED
                or approve_info.approvement_status != ok.ApprovementStatus.CLOSED
            ):
                is_all_cleaned_up = False

        self._set_approve_data(stage_info, approve_infos)

        if is_all_cleaned_up:
            stage_info.remove_data(self.IS_ROLLBACK_SET)
            stage_info.remove_data(self.INITIAL_APPROVE_INFOS)

    def _add_message_with_diff(self, stage_info, scenario):
        try:
            prepared_data = self._prepare_data_for_diff_comment(stage_info, scenario)
            startrek.get_client().add_comment(**prepared_data)
        except startrek.StartrekClientError as e:
            self.errors.append(e)

    def _collect_approve_infos_for_check(self, stage_info, scenario):
        result = self._collect_owners_of_hosts(
            stage_info,
            scenario_hosts=scenario.hosts,
            hosts_owners_key=self.APPROVE_INFOS_FOR_CHECK,
            hosts_bot_project_ids_key=self.BOT_PROJECT_IDS_FOR_CHECK,
            ticket_key=scenario.ticket_key,
        )
        return result

    def _collect_owners_of_hosts(
        self, stage_info, scenario_hosts, hosts_owners_key, hosts_bot_project_ids_key, ticket_key
    ):
        if not stage_info.get_data(self.STAGE_TMP_DATA, default=None):
            stage_info.set_data(self.STAGE_TMP_DATA, [host_info.inv for host_info in scenario_hosts.values()])
            stage_info.set_data(self.RESOLVED_HOSTS, [])

        resolved_hosts, error_hosts = self._get_host_owners(stage_info.get_data(self.STAGE_TMP_DATA))

        self._add_host_info(stage_info, resolved_hosts)

        if error_hosts:
            stage_info.set_data(self.STAGE_TMP_DATA, error_hosts)
            return Marker.failure()
        else:
            resolved_hosts_data = self._get_host_info(stage_info)

            approve_infos = self._group_hosts_by_responsible(resolved_hosts_data)
            self._fill_empty_owners_as_ticket_assignee(approve_infos, ticket_key)
            self._set_approve_data(stage_info, approve_infos, key=hosts_owners_key)

            grouped_hosts_by_bot_project_id = self._group_hosts_by_bot_project_id(resolved_hosts_data)
            stage_info.set_data(hosts_bot_project_ids_key, grouped_hosts_by_bot_project_id)

            stage_info.remove_data(self.STAGE_TMP_DATA)
            stage_info.remove_data(self.RESOLVED_HOSTS)

            return Marker.success()

    def _fill_empty_owners_as_ticket_assignee(self, approve_infos, ticket_key):
        startrek_client = startrek.get_client()
        issue = startrek_client.get_issue(ticket_key)
        assignee = [issue['assignee']['id']] if 'assignee' in issue else None
        if not assignee:
            assignee = config.get_value("scenario.rtc_approve_stage.default_approvers")

        for approve_info in approve_infos:
            if approve_info.responsibles == [_UNKNOWN]:
                approve_info.responsibles = assignee
                approve_info.role_types = {ASSIGNEE}
                approve_info.is_owners_unknown = True

    def _get_host_owners(self, invs):
        resolved_hosts = []
        error_hosts = []
        for inv in invs:
            try:
                resolved_hosts.append(self._resolve_host_owner(inv))
            except Exception as e:
                self.errors.append(e)
                error_hosts.append(inv)
        return resolved_hosts, error_hosts

    def _resolve_host_owner(self, host_inv):
        service_id, bot_project_id = self._get_abc_service_id_and_bot_project_id(host_inv)
        last_retrieved, owners, role_type = _get_owners(service_id)
        if not owners:
            logins = [_UNKNOWN]
            role_type = _UNKNOWN
        else:
            logins = self._collect_logins_from_abc_info(owners)

        return HostInfo(
            service_id=service_id,
            owners=logins,
            timestamp=last_retrieved,
            bot_project_id=bot_project_id,
            host_inv=host_inv,
            role_type=role_type,
        )

    def _is_restart_needed(self, stage_info):
        return stage_info.get_data(self.IS_ROLLBACK_SET, default=False)

    def _is_diff_message_needed(self, stage_info):
        return stage_info.get_data(self.IS_ROLLBACK_SET, default=False) and not stage_info.get_data(
            self.IS_CANCEL_SET, default=False
        )

    def _is_owners_and_ids_the_same(self, stage_info):
        is_approve_infos_same = True
        for initial_item, check_item in itertools.zip_longest(
            self._get_approve_data(stage_info, self.INITIAL_APPROVE_INFOS),
            self._get_approve_data(stage_info, self.APPROVE_INFOS_FOR_CHECK),
        ):
            if not self._do_items_exist(initial_item, check_item) or not self._do_approve_items_have_same_fields(
                initial_item, check_item
            ):
                is_approve_infos_same = False
                break
        return is_approve_infos_same and stage_info.get_data(self.BOT_PROJECT_IDS_INITIAL) == stage_info.get_data(
            self.BOT_PROJECT_IDS_FOR_CHECK
        )

    @staticmethod
    def _do_items_exist(*items):
        return all([item is not None for item in items])

    @staticmethod
    def _do_approve_items_have_same_fields(*items):
        return all(x.responsibles == items[0].responsibles and x.invs == items[0].invs for x in items)

    def _is_time_for_check_owners(self, stage_info):
        if stage_info.contains_data(self.LAST_OWNERS_GATHERING_TIMESTAMP):
            return (
                timestamp() - stage_info.get_data(self.LAST_OWNERS_GATHERING_TIMESTAMP, default=0)
                > constants.HOUR_SECONDS
            )
        return False

    def _is_cancel_needed(self, stage_info):
        return stage_info.get_data(self.IS_CANCEL_SET, default=False)

    def _cancel(self, scenario):
        scenario.cancel()

        try:
            prepared_data = self._prepare_data_for_canceled_comment(scenario)
            startrek.get_client().add_comment(**prepared_data)
        except startrek.StartrekClientError as e:
            self.errors.append(e)

    def _is_all_data_self_contained(self, stage_info, scenario):
        if not stage_info.contains_data(self.INITIAL_APPROVE_INFOS):
            return False

        ticket_author, ticket_timestamp = self._get_ticket_author_and_timestamp(scenario.ticket_key)

        if timestamp() - ticket_timestamp > WEEK_SECONDS:
            return False

        for approve_info in self._get_approve_data(stage_info):
            if ticket_author not in approve_info.responsibles:
                return False
        return True

    @staticmethod
    def _group_hosts_by_responsible(host_infos):
        grouped_hosts = collections.defaultdict(list)
        for host_info in host_infos:
            host_responsible = tuple(sorted(host_info.owners))
            grouped_hosts[host_responsible].append(host_info)

        result = []
        for owners, host_infos in grouped_hosts.items():
            role_types = set(sorted(item.role_type for item in host_infos))
            invs = sorted([item.host_inv for item in host_infos])
            approve_info = ApproveInfo(
                responsibles=list(owners), invs=invs, role_types=role_types, service_id=host_infos[0].service_id
            )
            result.append(approve_info)
        return sorted(result, key=lambda approve_info: approve_info.responsibles)

    @staticmethod
    def _group_hosts_by_bot_project_id(host_infos):
        grouped_hosts = collections.defaultdict(list)
        for host_info in host_infos:
            bot_project_id = host_info.bot_project_id
            grouped_hosts[bot_project_id].append(host_info.host_inv)
        return {str(id): sorted(invs) for id, invs in grouped_hosts.items()}

    @staticmethod
    def _get_abc_service_id_and_bot_project_id(host_inv):
        try:
            bot_project_id = bot.get_host_info(host_inv)["bot_project_id"]
            planner_id = bot.get_oebs_projects()[bot_project_id]["planner_id"]
            return planner_id, bot_project_id
        except Exception:
            msg = (
                "Bot project id not specified for host with inv {}. "
                "In most cases, this means that the host is not bound to the ABC service.".format(host_inv)
            )
            raise HostDoesNotHaveBotProjectIDError(message=msg)

    @staticmethod
    def _collect_logins_from_abc_info(owners):
        logins = sorted({owner["person"]["login"] for owner in owners})
        return logins

    @staticmethod
    def _get_ticket_author_and_timestamp(ticket_key):
        st_client = startrek.get_client()
        ticket_info = st_client.get_issue(ticket_key)
        ticket_timestamp = transform_datetime_to_unix_time(ticket_info['createdAt'])
        ticket_author = ticket_info['createdBy']['id']
        return ticket_author, ticket_timestamp

    def _close_approvement(self, scenario, ok_client, approve_info):
        if approve_info.approvement_status != ok.ApprovementStatus.CLOSED:
            try:
                approvement = ok_client.get_approvement(approve_info.approvement_uuid)
                if approvement.status != ok.ApprovementStatus.CLOSED:
                    ok_client.close_approvement(approve_info.approvement_uuid)

                approve_info.approvement_status = ok.ApprovementStatus.CLOSED
            except ok.OKError as e:
                self.errors.append(e)

    def _close_ticket(
        self, scenario, startrek_client, approve_info, is_local_closure, resolution=ApprovalTicketResolution.SUCCESSFUL
    ):
        if approve_info.ticket_status != TicketStatus.CLOSED:
            if is_local_closure:
                approve_info.ticket_status = TicketStatus.CLOSED
                return

            try:
                ticket_status = startrek_client.get_issue(approve_info.ticket_key)['status']['key']

                if ticket_status != TicketStatus.CLOSED:
                    startrek_client.close_issue(approve_info.ticket_key, transition="closed", resolution=resolution)

                approve_info.ticket_status = TicketStatus.CLOSED
            except startrek.StartrekClientError as e:
                self.errors.append(e)

    @staticmethod
    def _add_host_info(stage_info, values):
        result = [value._to_dict() for value in values]
        return stage_info.append_data(ApproveStage.RESOLVED_HOSTS, result)

    @staticmethod
    def _get_host_info(stage_info):
        raw_dicts = stage_info.get_data(ApproveStage.RESOLVED_HOSTS)

        return [HostInfo(**raw_dict) for raw_dict in raw_dicts]

    @staticmethod
    def _get_approve_data(stage_info, key=INITIAL_APPROVE_INFOS):
        raw_dicts = stage_info.get_data(key)

        return [ApproveInfo(**raw_dict) for raw_dict in raw_dicts] if raw_dicts else []

    @staticmethod
    def _set_approve_data(stage_info, values, key=INITIAL_APPROVE_INFOS):
        result = [value._to_dict() for value in values]
        return stage_info.set_data(key, result)

    @staticmethod
    def _prepare_data_for_approvement_creation(ticket_key, responsible, hosts_invs):
        if isinstance(responsible, str):
            responsible = [responsible]

        template_renderer = JinjaTemplateRenderer()
        ok_params = dict(invs=hosts_invs)
        approval_message = template_renderer.render_template(TemplatePath.OK_APPROVE_MESSAGE, **ok_params)

        return dict(approvers=responsible, ticket_key=ticket_key, text=approval_message)

    @staticmethod
    def _prepare_data_for_ticket_creation(parent_ticket_key, responsibles):
        if isinstance(responsibles, str):
            responsibles = [responsibles]
        responsibles = ", ".join(responsibles)

        return dict(
            queue=config.get_value("scenario.rtc_approve_stage.queue"),
            summary=config.get_value("scenario.rtc_approve_stage.summary").format(parent_ticket_key, responsibles),
            parent=parent_ticket_key,
            tags=config.get_value("scenario.rtc_approve_stage.tags"),
            type=APPROVAL_ISSUE_TYPE,
        )

    @staticmethod
    def _prepare_data_for_approve_comment(approve_info, main_ticket_key):
        params = dict(
            responsibles=", ".join([responsible + "@" for responsible in approve_info.responsibles]),
            invs=approve_info.invs,
            ticket_id=main_ticket_key,
            role_types=", ".join(approve_info.role_types),
            service_id=approve_info.service_id,
            iframe=ok.STAFF_IFRAME_TEMPLATE.format(uuid=approve_info.approvement_uuid),
            is_owners_unknown=approve_info.is_owners_unknown,
        )

        template_renderer = JinjaTemplateRenderer()
        output = template_renderer.render_template(TemplatePath.STARTREK_APPROVE_MESSAGE, **params)

        return dict(issue_id=approve_info.ticket_key, text=output, summonees=approve_info.responsibles)

    @staticmethod
    def _prepare_data_for_canceled_comment(scenario):
        params = dict(scenario_id=scenario.scenario_id)

        template_renderer = JinjaTemplateRenderer()
        output = template_renderer.render_template(TemplatePath.STARTREK_APPROVE_CANCELED, **params)

        return dict(issue_id=scenario.ticket_key, text=output)

    def _create_hosts_diff(self, stage_info, scenario):
        hosts = {}

        for host_info in scenario.hosts.values():
            hosts[host_info.inv] = HostDiff(host_info.inv)

        for item in self._get_approve_data(stage_info, self.INITIAL_APPROVE_INFOS):
            for inv in item.invs:
                hosts[inv].initial_responsibles = ", ".join(item.responsibles)

        for item in self._get_approve_data(stage_info, self.APPROVE_INFOS_FOR_CHECK):
            for inv in item.invs:
                hosts[inv].current_responsibles = ", ".join(item.responsibles)

        for (bot_project_id, invs) in stage_info.get_data(self.BOT_PROJECT_IDS_INITIAL).items():
            for inv in invs:
                hosts[inv].initial_bpid = bot_project_id

        for (bot_project_id, invs) in stage_info.get_data(self.BOT_PROJECT_IDS_FOR_CHECK).items():
            for inv in invs:
                hosts[inv].current_bpid = bot_project_id

        return list(hosts.values())

    def _prepare_data_for_diff_comment(self, stage_info, scenario):
        hosts_diff = self._create_hosts_diff(stage_info, scenario)

        params = dict(diffs=[item._to_dict() for item in hosts_diff])

        template_renderer = JinjaTemplateRenderer()
        output = template_renderer.render_template(TemplatePath.STARTREK_APPROVE_HOSTS_DIFF, **params)

        return dict(issue_id=scenario.ticket_key, text=output)


@ttl_cache(maxsize=1000, ttl=constants.HOUR_SECONDS)
def _get_owners(service_id):
    """See member schema in abc.get_service_members method"""
    members = abc.get_service_members(
        service_id=service_id, role_codes=[abc.Role.PRODUCT_HEAD, abc.Role.HARDWARE_RESOURCES_OWNER]
    )
    hardware_resource_owners = []
    product_heads = []

    for member in members:
        if member['person']['is_robot']:
            continue
        if member['role']['code'] == abc.Role.HARDWARE_RESOURCES_OWNER:
            hardware_resource_owners.append(member)
        else:
            product_heads.append(member)
    return (
        timestamp(),
        hardware_resource_owners or product_heads,
        HARDWARE_RESOURCES_OWNER if hardware_resource_owners else PRODUCT_HEAD,
    )


def transform_datetime_to_unix_time(timestamp):
    timestamp = timestamp[:-5]
    return time.mktime(datetime.datetime.strptime(timestamp, "%Y-%m-%dT%H:%M:%S.%f").timetuple())
