import logging

from walle.host_network import HostNetwork
from walle.scenario.constants import SharedDataKey, ConditionFunction
from walle.scenario.constants import StageName
from walle.scenario.host_stage_info import HostStageInfo
from walle.scenario.iteration_strategy import HostSequentialIterationStageStrategy
from walle.scenario.marker import Marker
from walle.scenario.mixins import HostParentStageHandler, HostStage
from walle.scenario.stages import StageRegistry
from walle.scenario.utils import BaseRegistry

log = logging.getLogger(__name__)


class ChoiceFuncRegistry(BaseRegistry):
    ITEMS = {}


def _get_hsi(host):
    hsi = HostStageInfo.objects.get(host_uuid=host.uuid)
    return hsi


@ChoiceFuncRegistry.register(ConditionFunction.ITDC_MAINTENANCE_REDEPLOY_NEEDED)
def get_redeploy_for_itdc_maintenance_scenario(stage_info, scenario, host):
    hsi = _get_hsi(host)
    shared_key = hsi.stage_info.read_shared_data(SharedDataKey.HOST_NETWORK_INFO)
    host_network = HostNetwork.objects.get(uuid=host.uuid)
    return host_network.active_mac != shared_key


@ChoiceFuncRegistry.register(ConditionFunction.ITDC_MAINTENANCE_REBOOT_NEEDED)
def get_reboot_for_itdc_maintenance_scenario(stage_info, scenario, host):
    hsi = _get_hsi(host)
    shared_key = hsi.stage_info.read_shared_data(SharedDataKey.HOST_NETWORK_INFO)
    host_network = HostNetwork.objects.get(uuid=host.uuid)
    return host_network.active_mac == shared_key


@StageRegistry.register(StageName.ConditionalHostStage)
class ConditionalHostStage(HostParentStageHandler, HostStage):
    """Checks if condition is met, then launches child stages"""

    CONDITION_CHECKED = "condition checked"

    def __init__(self, stages, condition_func):
        super().__init__(children=stages, condition_func=condition_func)
        self.children = stages
        self.condition_func = condition_func
        self.iteration_strategy = HostSequentialIterationStageStrategy()

    def run(self, stage_info, scenario, host, scenario_stage_info):
        if self._check_condition(stage_info, scenario, host):
            return self.execute_current_stage(stage_info, scenario, host, scenario_stage_info)

        return Marker.success(message="Condition was not met, child stage was not launcked")

    def _check_condition(self, stage_info, scenario, host):
        if self.CONDITION_CHECKED not in stage_info.data:
            stage_info.data[self.CONDITION_CHECKED] = True
            func = ChoiceFuncRegistry.get(self.condition_func)
            return func(stage_info, scenario, host)
        else:
            return True

    def serialize(self, uid="0"):
        stage_info = super().serialize(uid=uid)
        return stage_info
