import logging

from sepelib.core.exceptions import LogicalError
from walle.clients import startrek
from walle.scenario.constants import StageName
from walle.scenario.constants import TicketStatus
from walle.scenario.marker import Marker
from walle.scenario.mixins import Stage
from walle.scenario.stage_info import StageRegistry
from walle.scenario.utils import BaseRegistry

log = logging.getLogger(__name__)


class StartrekTicketParamsRegistry(BaseRegistry):
    ITEMS = {}


class StartrekTicketParamsName:
    DUMMY = "dummy_handler"


#  Just an example
@StartrekTicketParamsRegistry.register(StartrekTicketParamsName.DUMMY)
def dummy_ticket_params(stage_info, scenario):
    text = "Some simple text"
    params = dict(
        queue="ALEXSMIRNOV",  # for test purposes
        summary="Just a test",
        parent="WALLE-4032",
        tags="walle",
        type="task",
        text=text,
    )
    return params


@StageRegistry.register(StageName.CreateStartrekTicketStage)
class CreateStartrekTicketStage(Stage):
    """Creates a Startrek ticket"""

    IS_TICKET_CREATED = "is_ticket_created"
    TICKET_ID = "ticket_id"

    def __init__(self, ticket_params_func_name, wait_for_ticket_to_close=False):
        super().__init__(
            ticket_params_func_name=ticket_params_func_name, wait_for_ticket_to_close=wait_for_ticket_to_close
        )
        self.ticket_params_func_name = ticket_params_func_name
        self.wait_for_ticket_to_close = wait_for_ticket_to_close

    def _is_ticket_created(self, stage_info):
        return stage_info.get_data(self.IS_TICKET_CREATED, default=False)

    def _create_ticket(self, stage_info, scenario):
        startrek_client = startrek.get_client()
        get_ticket_params_func = StartrekTicketParamsRegistry.get(self.ticket_params_func_name)
        issue_params = get_ticket_params_func(stage_info, scenario)
        ticket_id = startrek_client.create_issue(issue_params)['key']
        stage_info.set_data(self.TICKET_ID, ticket_id)
        stage_info.set_data(self.IS_TICKET_CREATED, True)
        return ticket_id

    def _is_ticket_closed(self, stage_info):
        ticket_id = stage_info.get_data(self.TICKET_ID, default=None)
        if not ticket_id:
            raise LogicalError

        startrek_client = startrek.get_client()
        ticket_status = startrek_client.get_issue(ticket_id)['status']['key']

        if ticket_status != TicketStatus.CLOSED:
            return False
        return True

    def run(self, stage_info, scenario):
        if not self._is_ticket_created(stage_info):
            ticket_id = self._create_ticket(stage_info, scenario)
            return Marker.in_progress(message="Created ticket with id `{}`".format(ticket_id))

        ticket_id = stage_info.get_data(self.TICKET_ID)
        if self.wait_for_ticket_to_close:
            if not self._is_ticket_closed(stage_info):
                return Marker.in_progress(message="Ticket `{}` is not closed yet".format(ticket_id))
            else:
                Marker.success(message="Ticket `{}` was created and closed!".format(ticket_id))
        return Marker.success(message="Ticket `{}` was created!".format(ticket_id))
