from walle.clients import bot, startrek
from walle.hosts import Host
from walle.scenario.constants import StageName, TemplatePath
from walle.scenario.marker import Marker
from walle.scenario.mixins import Stage
from walle.scenario.scenario import Scenario
from walle.scenario.stage_info import StageInfo
from walle.scenario.stages import StageRegistry
from walle.util.mongo import SECONDARY_LOCAL_DC_PREFERRED
from walle.util.template_loader import JinjaTemplateRenderer


@StageRegistry.register(StageName.DetectStorageStage)
class DetectStorageStage(Stage):
    """Detect external disk storages"""

    CURRENT_POS = "current_pos"
    STORAGE_INFO = "storage_info"
    COMMENT_ID_FIELD = "st_comment_id"

    def run(self, stage_info: StageInfo, scenario: Scenario) -> Marker:
        self._collect_data(stage_info, self._get_scenario_host_invs(scenario))

        storage_info = stage_info.get_data(self.STORAGE_INFO)
        if not storage_info:
            return Marker.success(message="Didn't found any storages")

        if not stage_info.get_data(self.COMMENT_ID_FIELD):
            self._create_comment(
                scenario.ticket_key, stage_info, storage_info, self._get_names_by_invs(list(storage_info.keys()))
            )

        return Marker.success(message="Found storages, added comment.")

    def _collect_data(self, stage_info: StageInfo, host_invs: list[int]):
        current_pos = stage_info.get_data(self.CURRENT_POS, 0)
        storage_info = {}

        try:
            while current_pos < len(host_invs):
                host_inv = host_invs[current_pos]
                storages = []

                for storage in bot.get_storages(host_inv):
                    storages.append({"inv": storage.inv, "type": storage.type})

                if storages:
                    storage_info[str(host_inv)] = storages

                current_pos += 1
        finally:
            stage_info.set_data(self.CURRENT_POS, current_pos)

            if stage_info.contains_data(self.STORAGE_INFO):
                stage_info.update_data(self.STORAGE_INFO, storage_info)
            else:
                stage_info.set_data(self.STORAGE_INFO, storage_info)

    def _create_comment(
        self, ticket_key: str, stage_info: StageInfo, storage_info: dict[str, list[dict]], host_names: dict[str, str]
    ):
        template_renderer = JinjaTemplateRenderer()
        text = template_renderer.render_template(
            TemplatePath.STARTREK_STORAGE_INFO, storage_info=storage_info, host_names=host_names
        )
        comment = dict(issue_id=ticket_key, text=text)
        comment_id = startrek.get_client().add_comment(**comment).get("id", None)
        stage_info.set_data(self.COMMENT_ID_FIELD, comment_id)

    @staticmethod
    def _get_scenario_host_invs(scenario: Scenario) -> list[int]:
        return sorted([host.inv for host in scenario.hosts.values()])

    @staticmethod
    def _get_names_by_invs(hosts_invs: list[int]) -> dict[str, str]:
        hosts_names = {}
        for host in Host.objects(inv__in=hosts_invs, read_preference=SECONDARY_LOCAL_DC_PREFERRED).only("inv", "name"):
            if host.name is not None:
                hosts_names[str(host.inv)] = host.name

        return hosts_names
