import typing
from collections import namedtuple

from walle import network
from walle.clients import startrek
from walle.clients.dns import slayer_dns_api
from walle.clients.dns.interface import DnsError
from walle.hosts import Host, HostState
from walle.scenario.constants import StageName, TemplatePath
from walle.scenario.marker import Marker
from walle.scenario.mixins import Stage
from walle.scenario.scenario import Scenario
from walle.scenario.stage_info import StageInfo
from walle.scenario.stages import StageRegistry
from walle.util.template_loader import JinjaTemplateRenderer

DnsZonesWithOwners = namedtuple("DnsZoneOwners", "name, hosts, owners")
DnsZonesWithError = namedtuple("DnsZoneError", "name, hosts, error")


@StageRegistry.register(StageName.EnsureDnsAccessStage)
class EnsureDnsAccessStage(Stage):
    """
    Checks scenario hosts' A and AAAA DNS records.

    For hosts with these DNS records, checks if Wall-e has access to modify their zones through DNS API.
    If Walle has no access to modify these zones, or if there are records in zones which are not managed by DNS API -
    creates a comment in scenario's ticket. Hosts' owners are supposed to clear these DNS records manually.

    Ignores hosts without DNS records and hosts in HostState.FREE state.

    Stage is finished when there are no hosts' DNS records in zones that Wall-e can not modify.
    """

    COMMENT_ID_FIELD = "st_comment_id"

    def __init__(self):
        self.dns_client = None
        super().__init__()

    def run(self, stage_info: StageInfo, scenario: Scenario) -> Marker:
        # TODO(rocco66): add rurikk_dns support WALLE-4366
        self.dns_client = slayer_dns_api.DnsClient()

        scenario_hosts_uuids = list(scenario.hosts.keys())
        hosts_with_dns_records = self._get_hosts_with_dns_records(scenario_hosts_uuids)
        zones_with_owners, zones_with_dns_error = self._get_zones_owners(hosts_with_dns_records)

        zones_not_managed_by_walle = [
            zone
            for zone in zones_with_owners
            if not self.dns_client.is_zone_owner(zone.name)  # Checks if 'robot-walle' is in owners in ACL.
        ]

        if not zones_not_managed_by_walle and not zones_with_dns_error:
            return Marker.success(message="Didn't found any DNS records")

        if not stage_info.get_data(self.COMMENT_ID_FIELD):
            zones_not_managed_by_walle.sort(key=lambda zone: zone.name)
            zones_with_dns_error.sort(key=lambda zone: zone.name)
            self._create_dns_request_comment(
                zones_not_managed_by_walle, zones_with_dns_error, stage_info, scenario.ticket_key
            )

        return Marker.in_progress(message="Found some DNS records, will wait until they will be deleted")

    def _get_hosts_with_dns_records(self, hosts_uuids: typing.List[str]) -> typing.List[str]:
        """
        Returns scenario hosts which have DNS records, and not in HostState.FREE state.
        """
        hosts_with_dns_records = list()
        for host in Host.objects(uuid__in=hosts_uuids, state__ne=HostState.FREE).only("name", "project", "state"):
            hosts_with_dns_records.extend(
                [name for name in network.get_host_fqdns(host) if self._is_host_dns_record_exist(name)]
            )
        return hosts_with_dns_records

    def _create_dns_request_comment(
        self,
        zones_not_managed_by_walle: typing.List[DnsZonesWithOwners],
        zones_with_dns_api_error: typing.List[DnsZonesWithError],
        stage_info: StageInfo,
        startrek_ticket_key: str,
    ):
        """
        Creates a comment in StarTrek and save its ID to stage's data.
        """
        template_renderer = JinjaTemplateRenderer()
        text = template_renderer.render_template(
            TemplatePath.STARTREK_DNS_REQUEST,
            zones_not_managed_by_walle=zones_not_managed_by_walle,
            zones_with_dns_api_error=zones_with_dns_api_error,
        )
        comment = dict(issue_id=startrek_ticket_key, text=text)
        comment_id = startrek.get_client().add_comment(**comment).get("id", None)
        stage_info.set_data(self.COMMENT_ID_FIELD, comment_id)

    def _get_zones_owners(
        self, hosts_with_dns_records: typing.List[str]
    ) -> (typing.List[DnsZonesWithOwners], typing.List[DnsZonesWithError]):
        """
        For each host determines its DNS zone, and gets zone owners from DNS API.
        """
        zones_with_owners = dict()
        zones_with_dns_error = dict()
        zone_to_owners_map = dict()

        for host_fqdn in hosts_with_dns_records:
            zone_name = str(self.dns_client._local_ns_client.get_zone_for_name(host_fqdn))

            try:
                if zone_name not in zone_to_owners_map:
                    zone_to_owners_map[zone_name] = [owner for owner in self.dns_client.get_zone_owners(zone_name)]

                zones_with_owners.setdefault(
                    zone_name, DnsZonesWithOwners(zone_name, list(), zone_to_owners_map[zone_name])
                ).hosts.append(host_fqdn)
            except DnsError as e:
                zones_with_dns_error.setdefault(zone_name, DnsZonesWithError(zone_name, list(), e)).hosts.append(
                    host_fqdn
                )

        return list(zones_with_owners.values()), list(zones_with_dns_error.values())

    def _is_host_dns_record_exist(self, hostname: str) -> bool:
        return True if (self.dns_client.get_a(hostname) or self.dns_client.get_aaaa(hostname)) else False
