import dataclasses
import logging
import typing
from datetime import datetime

from sepelib.core import constants, config
from walle.clients import abc, ok, startrek
from walle.maintenance_plot.model import MaintenancePlot
from walle.models import timestamp
from walle.scenario.constants import StageName, TemplatePath, ScenarioWorkStatus
from walle.scenario.data_storage.base import BaseScenarioDataStorage
from walle.scenario.data_storage.types import HostGroupSource
from walle.scenario.definitions.base import get_data_storage
from walle.scenario.marker import Marker, MarkerStatus
from walle.scenario.mixins import HostGroupStage
from walle.scenario.scenario import Scenario
from walle.scenario.stage_info import StageInfo, StageRegistry
from walle.util.approvement_tools import (
    ApproveClient,
    ApprovalTicketResolution,
    CreateStartrekTicketRequest,
    CreateOkApprovementRequest,
    CreateStartrekCommentRequest,
)
from walle.util.template_loader import JinjaTemplateRenderer
from enum import Enum


log = logging.getLogger(__name__)


@dataclasses.dataclass(eq=True, frozen=True)
class HostTaskStartTimes:
    start_time_of_set_to_maintenance: typing.Optional[str] = None
    start_time_of_power_off: typing.Optional[str] = None


class RenewApproversStatus(int, Enum):
    NOT_STARTED = 1
    NEED_TO_SEND_COMMENT = 2
    NEED_TO_CHANGE_APPROVERS = 3


@dataclasses.dataclass
class ApprovementInfo:
    startrek_ticket_key: str = None
    ok_uuid: str = None
    approvers_logins_in_ok_approvement: typing.List[str] = dataclasses.field(default_factory=list)
    call_approvers_startrek_comment_id: str = None
    approvement_status: str = None
    approvement_resolution: str = None
    is_escalated: bool = False
    startrek_ticket_is_closed: bool = False
    scenario_startrek_ticket_resolution_comment_id: str = None
    approvers_renew_last_ts: typing.Optional[int] = None
    renew_approvers_status: RenewApproversStatus = RenewApproversStatus.NOT_STARTED

    def to_dict(self):
        return {
            "startrek_ticket_key": self.startrek_ticket_key,
            "ok_uuid": self.ok_uuid,
            # dataclasses.asdict() fails with
            # TypeError: __init__() missing 2 required positional arguments: 'instance' and 'name'
            # See comment below and https://github.com/samuelcolvin/pydantic/issues/1221 .
            "approvers_logins_in_ok_approvement": self.approvers_logins_in_ok_approvement or [],  #
            "call_approvers_startrek_comment_id": self.call_approvers_startrek_comment_id,
            "approvement_status": self.approvement_status,
            "approvement_resolution": self.approvement_resolution,
            "is_escalated": self.is_escalated,
            "startrek_ticket_is_closed": self.startrek_ticket_is_closed,
            "scenario_startrek_ticket_resolution_comment_id": self.scenario_startrek_ticket_resolution_comment_id,
            "approvers_renew_last_ts": self.approvers_renew_last_ts,
            "renew_approvers_status": self.renew_approvers_status or RenewApproversStatus.NOT_STARTED,
        }


class HostGroupApproveStageError(Exception):
    pass


@StageRegistry.register(StageName.HostGroupApproveStage)
class HostGroupApproveStage(HostGroupStage):
    """Creates approvement in OK and tracks its resolution."""

    _approvement_info_field_name = "approvement_info"

    _approvers_cache_ttl = constants.MINUTE_SECONDS * 15

    _marker_message_wait_for_resolution = "Wait for resolution of approvement"
    _marker_message_created_comment_with_approvement = "Created comment with approvement."
    _marker_message_created_approvement_on_ok = "Created approvement in OK service."
    _marker_message_created_ticket_for_approvement = "Created ticket for approvement."
    _marker_message_approvement_received = "Approvement received."
    _marker_message_approvement_declined = "Approvement was declined by the user and closed after escalation."
    _marker_message_auto_approve_by_maintenance_plot = "Approvement received from maintenance plot rules."
    _marker_message_approvement_disabled_by_maintenance_plot = "Approvement disabled by maintenance plot"

    _startrek_ticket_type = "approval"

    ok_approve_request_text_template_path = TemplatePath.OK_ITDC_MAINTENANCE_HOST_GROUP_APPROVE_REQUEST
    startrek_approvement_text_tempate_path = TemplatePath.STARTREK_ITDC_MAINTENANCE_HOST_GROUP_APPROVEMENT_TEXT
    startrek_escalation_text_template_path = TemplatePath.STARTREK_ITDC_MAINTENANCE_HOST_GROUP_APPROVE_ESCALATION
    startrek_received_approve_text_template_path = TemplatePath.STARTREK_ITDC_MAINTENANCE_HOST_GROUP_RECEIVED_APPROVE
    startrek_approve_rejected_text_template_path = TemplatePath.STARTREK_ITDC_MAINTENANCE_HOST_GROUP_APPROVE_REJECTED

    def __init__(self, option_name: typing.Optional[str] = None, **kwargs):
        super().__init__(option_name=option_name)
        self.option_name = option_name
        # Use `self._config[LIST_VALUE] or []` when passing these values to dataclass fields as a workaround for
        # one non-trivial bug.
        #
        # Value of `self._config` is passed to the stage as a parameter. When the specific instance of the scenario is
        # created, these parameters become serialized and written to MongoDb with MongoEngine. At runtime,
        # when stage becomes deserialized, lists within these parameters are actually not `list`, but complex
        # `mongoengine.base.datastructures.BaseList` instance, which behave as `list`.
        #
        # When such value is used in dataclasses' field, and this dataclass is converted to dict with
        # `dataclasses.asdict(obj)`, and `BaseList` object represent **empty** list instances, dataclasses'
        # `asdict` machinery fails.
        #
        # Example of an error: https://paste.yandex-team.ru/4504990
        # Code at contrib/tools/python3/src/Lib/dataclasses.py:1110 -  https://paste.yandex-team.ru/4504928 .
        #
        # Solution here is to explicity check if list typed value, that was read from Mongiengine document,
        # is not empty, and set it to an honest list otherwise.
        self._config = config.get_value("scenario.host_group_approve_stage")

    def run(self, stage_info: StageInfo, scenario: Scenario, host_group_id: int) -> MarkerStatus:
        data_storage = get_data_storage(scenario)

        if self.option_name and not self._is_stage_enabled(scenario, data_storage, host_group_id, self.option_name):
            return Marker.success(message=self._marker_message_approvement_disabled_by_maintenance_plot)

        approvement_info = self._read_approvement_info(stage_info)

        host_group_source = self._get_current_host_group_source(data_storage, host_group_id)
        if host_group_source and host_group_source.approvement_decision.skip_approvement:
            self._add_comment_to_parent_ticket_about_auto_approvement_by_maintenance_plot_settings(
                scenario.ticket_key, data_storage, host_group_id, host_group_source.approvement_decision.reason
            )
            return Marker.success(message=self._marker_message_auto_approve_by_maintenance_plot)

        # Create Startrek ticket for approvement. Scenario's ticket is a parent.
        if approvement_info is None:
            approvement_info = ApprovementInfo()
            approvement_info.startrek_ticket_key = self.create_startrek_ticket(scenario, data_storage, host_group_id)
            self._write_approvement_info(stage_info, approvement_info)
            return Marker.in_progress(message=self._marker_message_created_ticket_for_approvement)

        # Create approvement in OK.
        if approvement_info.ok_uuid is None:
            approvement_info.ok_uuid, approvement_info.approvers_logins_in_ok_approvement = self.create_approvement(
                approvement_info.startrek_ticket_key, data_storage, scenario, host_group_id
            )
            approvement_info.approvers_renew_last_ts = timestamp()
            self._write_approvement_info(stage_info, approvement_info)
            scenario.set_works_status_label(ScenarioWorkStatus.APPROVEMENT)
            return Marker.in_progress(message=self._marker_message_created_approvement_on_ok)

        # Create a comment in Startrek ticket with OK approvement's iframe.
        if approvement_info.call_approvers_startrek_comment_id is None:
            approvement_info.call_approvers_startrek_comment_id = self.call_approvers_into_startrek_ticket(
                approvement_info.startrek_ticket_key, approvement_info.ok_uuid, data_storage, host_group_id
            )
            self._add_comment_to_parent_ticket_about_created_ticket_with_approvement(
                scenario.ticket_key, approvement_info.startrek_ticket_key, data_storage, host_group_id
            )
            self._write_approvement_info(stage_info, approvement_info)
            return Marker.in_progress(message=self._marker_message_created_comment_with_approvement)

        # Check approvement status and resolution.
        (
            approvement_info.approvement_status,
            approvement_info.approvement_resolution,
        ) = self.get_approvement_status_resolution(approvement_info.ok_uuid)

        # From now all we do is wait for resolution of the approvement.
        # There are two of them:
        # * ok.ApprovementResolution.APPROVED
        # * ok.ApprovementResolution.DECLINED
        #
        # We also handle specific apprvement status ok.ApprovementStatus.REJECTED - in this case we call for SRE.

        # Handle given approve.
        if approvement_info.approvement_resolution == ok.ApprovementResolution.APPROVED:
            self.handle_given_approve(approvement_info, stage_info, scenario, host_group_id)
            return Marker.success(message=self._marker_message_approvement_received)

        # Handle a decline.
        if approvement_info.approvement_resolution == ok.ApprovementResolution.DECLINED:
            self.handle_decline(approvement_info, stage_info, scenario, host_group_id)
            return Marker.failure(message=self._marker_message_approvement_declined)

        # Handle a reject that is not escalated - escalate it.
        if approvement_info.approvement_status == ok.ApprovementStatus.REJECTED and not approvement_info.is_escalated:
            approvement_info = self.escalate(approvement_info)
            self._write_approvement_info(stage_info, approvement_info)
            return Marker.in_progress(message=self._marker_message_approvement_declined)

        # Check if it is time to renew approvers and handle their change.
        if timestamp() > approvement_info.approvers_renew_last_ts + self._approvers_cache_ttl:
            data_storage = get_data_storage(scenario)
            maintenance_plot_approvers_logins = self._get_approvers_logins(data_storage, host_group_id)
            antispam_time_add = 0

            if maintenance_plot_approvers_logins != approvement_info.approvers_logins_in_ok_approvement:
                if approvement_info.renew_approvers_status == RenewApproversStatus.NOT_STARTED:
                    approvement_info.renew_approvers_status = RenewApproversStatus.NEED_TO_SEND_COMMENT

                if approvement_info.renew_approvers_status == RenewApproversStatus.NEED_TO_SEND_COMMENT:
                    try:
                        self._add_comment_about_changed_approvers(
                            approvement_info.startrek_ticket_key,
                            maintenance_plot_approvers_logins,
                            approvement_info.approvers_logins_in_ok_approvement,
                        )
                        approvement_info.renew_approvers_status = RenewApproversStatus.NEED_TO_CHANGE_APPROVERS
                    except startrek.StartrekError as e:
                        log.error("Connection to Startrek failed (or got invalid response from the server): " + str(e))
                        antispam_time_add = 15 * constants.MINUTE_SECONDS
                    except Exception as e:
                        log.error("Unknown exception occured while connecting to Startrek: " + str(e))
                        antispam_time_add = 30 * constants.MINUTE_SECONDS

                if approvement_info.renew_approvers_status == RenewApproversStatus.NEED_TO_CHANGE_APPROVERS:
                    try:
                        self.handle_change_of_approvers(approvement_info, maintenance_plot_approvers_logins)
                    except ok.OKConnectionError as e:
                        log.error("Connection to OK failed: " + str(e))
                        if not antispam_time_add:
                            antispam_time_add = 15 * constants.MINUTE_SECONDS
                    except Exception as e:
                        log.error("Unknown exception occured while connecting to OK: " + str(e))
                        if not antispam_time_add:
                            antispam_time_add = 30 * constants.MINUTE_SECONDS

            approvement_info.approvers_renew_last_ts = timestamp() + antispam_time_add

        # Wait for resolution in all other cases.
        self._write_approvement_info(stage_info, approvement_info)
        return Marker.in_progress(message=self._marker_message_wait_for_resolution)

    def call_approvers_into_startrek_ticket(
        self,
        startrek_ticket_key: ApproveClient.TStartrekTicketKey,
        ok_uuid: ApproveClient.TOkUuid,
        data_storage: BaseScenarioDataStorage,
        host_group_id: int,
    ):
        approve_client = ApproveClient(startrek_client=startrek.get_client())
        request = CreateStartrekCommentRequest(
            issue_id=startrek_ticket_key,
            text=self._get_startrek_comment_text_approve_request(ok_uuid),
            summonees=tuple(self._get_approvers_logins(data_storage, host_group_id)),
        )
        return approve_client.create_startrek_comment(request)

    @staticmethod
    def close_approvement(ok_uuid: ApproveClient.TOkUuid):
        approve_client = ApproveClient(ok_client=ok.get_client())
        approve_client.close_ok_approvement(ok_uuid)

    def create_approvement(
        self,
        startrek_ticket_key: ApproveClient.TStartrekTicketKey,
        data_storage: BaseScenarioDataStorage,
        scenario: Scenario,
        host_group_id: int,
    ) -> (ApproveClient.TOkUuid, typing.List[str]):
        approve_client = ApproveClient(ok_client=ok.get_client())
        approvers_logins = self._get_approvers_logins(data_storage, host_group_id)
        request = CreateOkApprovementRequest(
            ticket_key=startrek_ticket_key,
            text=self._get_ok_approvement_request_text(scenario, host_group_id),
            author=scenario.issuer.strip("@"),
            approvers=tuple(approvers_logins),
            groups=tuple(self._config["ok_coordinators_groups"]) or (),
        )
        return approve_client.create_ok_approvement(request), approvers_logins

    def create_startrek_ticket(
        self, scenario: Scenario, data_storage: BaseScenarioDataStorage, host_group_id: int
    ) -> ApproveClient.TStartrekTicketKey:
        approve_client = ApproveClient(startrek_client=startrek.get_client())
        host_group_name = self._get_host_group_name(data_storage, host_group_id)
        summary = self._config["startrek_ticket_summary_template"].format(
            scenario.name, scenario.ticket_key, host_group_name
        )
        description = self._get_startrek_approvement_ticket_text(scenario, data_storage, host_group_id)
        request = CreateStartrekTicketRequest(
            parent=scenario.ticket_key,
            queue=self._config["startrek_queue"],
            summary=summary,
            description=description,
            tags=tuple(self._config["startrek_ticket_tags"]) or (),
            type=self._startrek_ticket_type,
        )
        return approve_client.create_startrek_ticket(request)

    def escalate(self, approvement_info: ApprovementInfo) -> ApprovementInfo:
        approve_client = ApproveClient(startrek_client=startrek.get_client())
        abc_service_slug = config.get_value(
            "scenario.host_group_approve_stage.abc_service_slug_to_gather_duty_on_reject"
        )
        request = CreateStartrekCommentRequest(
            issue_id=approvement_info.startrek_ticket_key,
            summonees=tuple(abc.get_service_on_duty_logins(abc_service_slug)),
            text=self._get_startrek_comment_text_call_for_routineoperations(),
        )
        approvement_info.call_approvers_startrek_comment_id = approve_client.create_startrek_comment(request)
        approvement_info.is_escalated = True
        return approvement_info

    @staticmethod
    def get_approvement_status_resolution(
        ok_uuid: ApproveClient.TOkUuid,
    ) -> (ApproveClient.TOkApprovementStatus, ApproveClient.TOkApprovementResolution):
        approve_client = ApproveClient(ok_client=ok.get_client())
        return approve_client.get_ok_approvement_status_resolution(ok_uuid)

    @staticmethod
    def handle_change_of_approvers(approvement_info: ApprovementInfo, new_approvers: typing.List[str]):
        # Close current approvement and reset saved data about it.
        # May be changed to editing existing approvement after OK-34.
        approve_client = ApproveClient(ok_client=ok.get_client())

        try:
            approve_client.edit_approvers(approvement_info.ok_uuid, new_approvers)
            approvement_info.renew_approvers_status = RenewApproversStatus.NOT_STARTED
        except ok.OKBadRequest:
            approve_client.close_ok_approvement(approvement_info.ok_uuid)
            approvement_info.ok_uuid = None
            approvement_info.call_approvers_startrek_comment_id = None
            approvement_info.approvement_status = None
            approvement_info.approvement_resolution = None
            approvement_info.renew_approvers_status = RenewApproversStatus.NOT_STARTED

    def handle_decline(
        self, approvement_info: ApprovementInfo, stage_info: StageInfo, scenario: Scenario, host_group_id: int
    ):
        data_storage = get_data_storage(scenario)
        approve_client = ApproveClient(startrek_client=startrek.get_client(), ok_client=ok.get_client())

        # Close Startrek ticket with approvement.
        if not approvement_info.startrek_ticket_is_closed:
            approve_client.close_startrek_ticket(approvement_info.startrek_ticket_key, ApprovalTicketResolution.REFUSAL)
            approvement_info.startrek_ticket_is_closed = True
            self._write_approvement_info(stage_info, approvement_info)

        # Comment scenario ticket with resolution.
        if not approvement_info.scenario_startrek_ticket_resolution_comment_id:
            approvement_info.scenario_startrek_ticket_resolution_comment_id = (
                self.write_resolution_comment_to_scenario_ticket(
                    scenario.ticket_key,
                    self._get_startrek_comment_text_approve_declined(scenario, data_storage, host_group_id),
                )
            )
            self._write_approvement_info(stage_info, approvement_info)

    def handle_given_approve(
        self, approvement_info: ApprovementInfo, stage_info: StageInfo, scenario: Scenario, host_group_id: int
    ):
        data_storage = get_data_storage(scenario)
        approve_client = ApproveClient(startrek_client=startrek.get_client())

        # Close Startrek ticket with approvement.
        if not approvement_info.startrek_ticket_is_closed:
            approve_client.close_startrek_ticket(approvement_info.startrek_ticket_key)
            approvement_info.startrek_ticket_is_closed = True
            self._write_approvement_info(stage_info, approvement_info)

        # Comment scenario ticket with resolution.
        if not approvement_info.scenario_startrek_ticket_resolution_comment_id:
            approvement_info.scenario_startrek_ticket_resolution_comment_id = (
                self.write_resolution_comment_to_scenario_ticket(
                    scenario.ticket_key,
                    self._get_startrek_comment_text_received_approve(scenario, data_storage, host_group_id),
                )
            )
            self._write_approvement_info(stage_info, approvement_info)

    @staticmethod
    def write_resolution_comment_to_scenario_ticket(
        scenario_ticket_key: ApproveClient.TStartrekTicketKey, text: str
    ) -> ApproveClient.TStartrekCommentId:
        approve_client = ApproveClient(startrek_client=startrek.get_client())
        request = CreateStartrekCommentRequest(issue_id=scenario_ticket_key, summonees=(), text=text)
        return approve_client.create_startrek_comment(request)

    def _add_comment_to_parent_ticket_about_created_ticket_with_approvement(
        self,
        scenario_ticket_key: ApproveClient.TStartrekTicketKey,
        approve_ticket_key: ApproveClient.TStartrekTicketKey,
        data_storage: BaseScenarioDataStorage,
        host_group_id: int,
    ):
        host_group_name = self._get_host_group_name(data_storage, host_group_id)
        approve_client = ApproveClient(startrek_client=startrek.get_client())
        text = self._config["startrek_created_approve_ticket_message"].format(approve_ticket_key, host_group_name)
        request = CreateStartrekCommentRequest(issue_id=scenario_ticket_key, summonees=(), text=text)
        return approve_client.create_startrek_comment(request)

    def _add_comment_to_parent_ticket_about_auto_approvement_by_maintenance_plot_settings(
        self,
        scenario_ticket_key: ApproveClient.TStartrekTicketKey,
        data_storage: BaseScenarioDataStorage,
        host_group_id: int,
        skip_reason: str,
    ):
        host_group_name = self._get_host_group_name(data_storage, host_group_id)
        approve_client = ApproveClient(startrek_client=startrek.get_client())
        text = self._config["startrek_auto_approvement_ticket_message"].format(host_group_name, skip_reason)
        request = CreateStartrekCommentRequest(issue_id=scenario_ticket_key, summonees=(), text=text)
        return approve_client.create_startrek_comment(request)

    def _add_comment_about_changed_approvers(
        self,
        approve_ticket_key: ApproveClient.TStartrekTicketKey,
        new_approvers_logins: list[str, ...],
        current_approvers_logins: list[str, ...],
    ):
        approve_client = ApproveClient(startrek_client=startrek.get_client())
        new_approvers_str = ", ".join("@{}".format(login) for login in new_approvers_logins)
        current_approvers_str = ", ".join("@{}".format(login) for login in current_approvers_logins)
        text = self._config["startek_changed_approvers_logins_ticket_message"].format(
            current_approvers_str, new_approvers_str
        )
        request = CreateStartrekCommentRequest(issue_id=approve_ticket_key, summonees=(), text=text)
        return approve_client.create_startrek_comment(request)

    def _get_approvers_logins(self, data_storage: BaseScenarioDataStorage, host_group_id: int) -> list[str, ...]:
        return self._get_maintenance_plot(data_storage, host_group_id).get_approvers()

    def _get_host_group_name(self, data_storage: BaseScenarioDataStorage, host_group_id: int) -> str:
        # ABC service name as a host group name is fine for now.
        if host_group := self._get_current_host_group_source(data_storage, host_group_id):
            return host_group.source.get_group_source_name()
        return self._get_maintenance_plot(data_storage, host_group_id).meta_info.abc_service_slug

    def _get_ok_approvement_request_text(self, scenario: Scenario, host_group_id: int) -> str:
        return JinjaTemplateRenderer().render_template(
            self.ok_approve_request_text_template_path,
            invs=self._get_hosts_invs(scenario, host_group_id),
            scenario_id=scenario.scenario_id,
            host_group_id=host_group_id,
        )

    def _get_startrek_approvement_ticket_text(
        self, scenario: Scenario, data_storage: BaseScenarioDataStorage, host_group_id: int
    ) -> str:
        scenario_parameters = data_storage.read_scenario_parameters()
        if scenario_parameters.maintenance_start_time is None:
            start_time = "Unknown"
        else:
            start_time = datetime.fromtimestamp(scenario_parameters.maintenance_start_time).strftime(
                "%d.%m.%Y %H:%M:%S"
            )

        issuer = "@{}".format(scenario.issuer.strip("@"))
        host_task_start_times = self._get_maintenance_start_time_and_power_off_start_time(
            scenario, data_storage, host_group_id
        )
        return JinjaTemplateRenderer().render_template(
            self.startrek_approvement_text_tempate_path,
            invs=self._get_hosts_invs(scenario, host_group_id),
            scenario_name=scenario.name,
            parent_ticket=scenario.ticket_key,
            start_time=start_time,
            scenario_issuer=issuer,
            set_maintenance_start_time=host_task_start_times.start_time_of_set_to_maintenance,
            power_off_start_time=host_task_start_times.start_time_of_power_off,
        )

    @staticmethod
    def _get_startrek_comment_text_approve_request(ok_uuid: str) -> str:
        return ok.STAFF_IFRAME_TEMPLATE.format(uuid=ok_uuid)

    def _get_startrek_comment_text_call_for_routineoperations(self) -> str:
        return JinjaTemplateRenderer().render_template(self.startrek_escalation_text_template_path)

    def _get_startrek_comment_text_received_approve(
        self, scenario: Scenario, data_storage: BaseScenarioDataStorage, host_group_id: int
    ) -> str:
        host_task_start_times = self._get_maintenance_start_time_and_power_off_start_time(
            scenario, data_storage, host_group_id
        )
        return JinjaTemplateRenderer().render_template(
            self.startrek_received_approve_text_template_path,
            invs=self._get_hosts_invs(scenario, host_group_id),
            scenario_id=scenario.scenario_id,
            host_group_name=self._get_host_group_name(data_storage, host_group_id),
            set_maintenance_start_time=host_task_start_times.start_time_of_set_to_maintenance,
            power_off_start_time=host_task_start_times.start_time_of_power_off,
        )

    def _get_maintenance_start_time_and_power_off_start_time(
        self, scenario: Scenario, data_storage: BaseScenarioDataStorage, host_group_id: int
    ) -> HostTaskStartTimes:
        scenario_parameters = data_storage.read_scenario_parameters()

        if scenario_parameters.maintenance_start_time is None:
            return HostTaskStartTimes()

        maintenance_plot = self._get_maintenance_plot(data_storage, host_group_id)
        scenario_settings = maintenance_plot.get_scenario_settings(scenario.scenario_type)
        host_tasks_offsets = scenario_settings.get_offset_times_of_host_tasks()

        start_time_of_set_to_maintenance = self._get_start_time_of_set_to_maintenance(
            host_tasks_offsets.request_cms_x_seconds_before_maintenance_start_time,
            scenario_parameters.maintenance_start_time,
        )
        start_time_of_power_off = self._get_start_time_of_power_off(
            host_tasks_offsets.start_power_off_x_seconds_before_maintenance_start_time,
            scenario_parameters.maintenance_start_time,
        )

        return HostTaskStartTimes(
            start_time_of_set_to_maintenance=start_time_of_set_to_maintenance,
            start_time_of_power_off=start_time_of_power_off,
        )

    @staticmethod
    def _get_start_time_of_set_to_maintenance(
        offset: typing.Optional[int], maintenance_start_time: int
    ) -> typing.Optional[str]:
        if offset:
            wait_until_ts_for_maintenance = maintenance_start_time - offset
            wait_until_str_for_maintenance = datetime.fromtimestamp(wait_until_ts_for_maintenance).strftime(
                "%H:%M %d.%m.%Y"
            )
            return wait_until_str_for_maintenance

    @staticmethod
    def _get_start_time_of_power_off(offset: typing.Optional[int], maintenance_start_time: int) -> typing.Optional[str]:
        if offset:
            wait_until_ts_for_power_off = maintenance_start_time - offset
            wait_until_str_for_power_off = datetime.fromtimestamp(wait_until_ts_for_power_off).strftime(
                "%H:%M %d.%m.%Y"
            )
            return wait_until_str_for_power_off

    def _get_startrek_comment_text_approve_declined(
        self, scenario: Scenario, data_storage: BaseScenarioDataStorage, host_group_id: int
    ) -> str:
        return JinjaTemplateRenderer().render_template(
            self.startrek_approve_rejected_text_template_path,
            invs=self._get_hosts_invs(scenario, host_group_id),
            scenario_id=scenario.scenario_id,
            host_group_name=self._get_host_group_name(data_storage, host_group_id),
        )

    @staticmethod
    def _get_hosts_invs(scenario: Scenario, host_group_id: int) -> typing.List[int]:
        return [host.inv for host in scenario.hosts.values() if host.group == host_group_id]

    @staticmethod
    def _get_maintenance_plot(data_storage: BaseScenarioDataStorage, host_group_id: int) -> MaintenancePlot:
        for host_group_source in data_storage.read_host_groups_sources():
            if host_group_source.group_id == host_group_id:
                return host_group_source.source.get_maintenance_plot()
        raise HostGroupApproveStageError("No maintenance plot found for host group %d" % host_group_id)

    @staticmethod
    def _get_current_host_group_source(
        data_storage: BaseScenarioDataStorage, host_group_id: int
    ) -> typing.Optional[HostGroupSource]:
        for host_group_source in data_storage.read_host_groups_sources():
            if host_group_source.group_id == host_group_id:
                return host_group_source

    def _read_approvement_info(self, stage_info: StageInfo) -> typing.Optional[ApprovementInfo]:
        approvement_info_dict = stage_info.get_data(self._approvement_info_field_name)
        if not approvement_info_dict:
            return None
        return ApprovementInfo(**approvement_info_dict)

    def _write_approvement_info(self, stage_info: StageInfo, approvement_info: ApprovementInfo):
        stage_info.set_data(self._approvement_info_field_name, approvement_info.to_dict())

    def _is_stage_enabled(
        self,
        scenario: Scenario,
        data_storage: BaseScenarioDataStorage,
        host_group_id: int,
        option_name: typing.Optional[str],
    ) -> bool:
        maintenance_plot = self._get_maintenance_plot(data_storage, host_group_id)
        settings = maintenance_plot.get_scenario_settings(scenario_type=scenario.scenario_type)

        try:
            result = getattr(settings, option_name)
        except AttributeError:
            result = True

        return result
