from abc import abstractproperty, abstractmethod

from mongoengine.errors import DoesNotExist

from walle.host_network import HostNetwork
from walle.scenario.constants import StageName
from walle.scenario.marker import Marker, create_shared_data
from walle.scenario.mixins import HostStageWithSharedData
from walle.scenario.stages import StageRegistry


class OperationName:

    FindActiveMacAddressOperation = "FindActiveMacAddressOperation"


@StageRegistry.register(StageName.LambdaStage)
class LambdaStage(HostStageWithSharedData):
    def __init__(self, shared_data_key, operation_name, **kwargs):
        super().__init__(shared_data_key=shared_data_key, operation_name=operation_name, **kwargs)
        self.operation_name = operation_name
        self.shared_data_key = shared_data_key

    def run(self, stage_info, scenario, host, *args, **kwargs):
        self.log_info(scenario, host, msg="start running")

        operation = self._get_operation_instance()
        result = operation.run(stage_info, scenario, host, *args, **kwargs)

        shared_data = create_shared_data(self.shared_data_key, result)
        return Marker.success(data=shared_data, message="Shared data was saved successfully")

    def _get_operation_instance(self):
        operation_class = OPERATIONS[self.operation_name]
        return operation_class()


class Operation:
    @abstractproperty
    def name(self):
        raise NotImplementedError

    @abstractmethod
    def run(self, stage_info, scenario, host, *args, **kwargs):
        raise NotImplementedError


class FindActiveMacAddress(Operation):
    @property
    def name(self):
        return OperationName.FindActiveMacAddressOperation

    def run(self, stage_info, scenario, host, *args, **kwargs):
        try:
            host_network = HostNetwork.objects.get(uuid=host.uuid)
            return host_network.active_mac
        except DoesNotExist:
            return None


ALL_OPERATIONS = [FindActiveMacAddress]


OPERATIONS = {op().name: op for op in ALL_OPERATIONS}
