from typing import Optional, List

from sepelib.core.exceptions import LogicalError
from walle.clients import juggler
from walle.models import timestamp
from walle.scenario.constants import StageName
from walle.scenario.marker import MarkerStatus
from walle.scenario.mixins import StageRunInterface
from walle.scenario.scenario import Scenario
from walle.scenario.stage_info import StageInfo, StageRegistry
from walle.scenario.stages import ScenarioRootStage


@StageRegistry.register(StageName.TimeoutStage)
class TimeoutStage(ScenarioRootStage):
    STAGE_END_TIME = "stage_end_time"

    def __init__(self, children, timeout: int, juggler_service_name: str, host_name: Optional[str] = None, **params):
        children = _ensure_non_empty_stage_list(children)
        self.timeout = timeout
        self.start_time: Optional[int] = None
        self.juggler_service_name = juggler_service_name
        self.host_name = host_name
        super().__init__(
            children, timeout=timeout, juggler_service_name=juggler_service_name, host_name=host_name, **params
        )

    def run(self, stage_info: StageInfo, scenario: Scenario):
        self._prepare(stage_info)
        marker = super().run(stage_info, scenario)
        if marker.status == MarkerStatus.IN_PROGRESS and self._timed_out(stage_info):
            juggler.send_event(
                self.juggler_service_name,
                juggler.JugglerCheckStatus.CRIT,
                _timeout_message(self.start_time, self.timeout, scenario),
                host_name=self.host_name,
            )
        elif marker.status == MarkerStatus.SUCCESS:
            juggler.send_event(
                self.juggler_service_name,
                juggler.JugglerCheckStatus.OK,
                "Timeout stage for scenario https://wall-e.yandex-team.ru/scenarios/{} finished successfully".format(
                    scenario.scenario_id
                ),
                host_name=self.host_name,
            )
        elif marker.status == MarkerStatus.FAILURE:
            juggler.send_event(
                self.juggler_service_name,
                juggler.JugglerCheckStatus.OK,
                "Timeout stage for scenario https://wall-e.yandex-team.ru/scenarios/{} failed".format(
                    scenario.scenario_id
                ),
                host_name=self.host_name,
            )
        return marker

    def _prepare(self, stage_info: StageInfo):
        if self.STAGE_END_TIME not in stage_info.data:
            self.start_time = timestamp()
            stage_info.data[self.STAGE_END_TIME] = self.start_time + self.timeout

    def _timed_out(self, stage_info: StageInfo) -> bool:
        return timestamp() >= stage_info.data[self.STAGE_END_TIME]


def _timeout_message(start_time: int, timeout: int, scenario: Scenario) -> str:
    message = """Not all stages are finished before timeout {} from start time {}
Scenario: https://wall-e.yandex-team.ru/scenarios/{}""".format(
        timeout, start_time, scenario.scenario_id
    )
    if scenario.ticket_key is not None:
        message = message + "\nTicket: https://st.yandex-team.ru/{}".format(scenario.ticket_key)
    return message


def _ensure_non_empty_stage_list(children) -> List[StageRunInterface]:
    if isinstance(children, list):
        if not children:
            raise LogicalError()
        return children

    if isinstance(children, StageRunInterface):
        return [children]
    else:
        raise LogicalError()
