from mongoengine import StringField, LongField, EmbeddedDocument, DictField, EmbeddedDocumentListField

from walle.models import timestamp
from walle.scenario.constants import DEFAULT_STAGE_DESCRIPTION
from walle.scenario.utils import BaseRegistry


class StageRegistry(BaseRegistry):
    ITEMS = {}


class StageStatus:
    QUEUE = "queue"
    PROCESSING = "processing"
    FINISHED = "finished"
    CANCELLING = "cancelling"
    CANCELLED = "cancelled"
    ALL = [QUEUE, PROCESSING, FINISHED, CANCELLING, CANCELLED]


class StageAction:
    PREPARE = "prepare"
    ACTION = "action"
    CHECK = "check"
    COMPLETION = "completion"
    FAILURE = "failure"
    ALL = [PREPARE, ACTION, CHECK, COMPLETION, FAILURE]


class StageInfo(EmbeddedDocument):
    """Represents info about stages"""

    uid = StringField(help_text="Current stage UID", required=True)
    seq_num = LongField(default=0, help_text="Current active child stage")
    name = StringField(help_text="Current stage name")
    description = StringField(help_text="Current stage description", default=DEFAULT_STAGE_DESCRIPTION)
    status = StringField(choices=StageStatus.ALL, help_text="Current stage status", default=StageStatus.QUEUE)
    status_time = LongField(help_text="Time when the current status has been set")
    action_type = StringField(help_text="Name of stage's action type", default=StageAction.ACTION)
    data = DictField(help_text="Contains all modifiable generated data from stage")
    params = DictField(help_text="Constant parameters for the stage")
    shared_data = DictField(help_text="Contains all shared data between stages on same level")

    stages = EmbeddedDocumentListField("self", help_text="A list of stages assigned to this scenario")

    msg = StringField(help_text="Message with the result of the last stage execution")
    hosts = DictField(help_text="Contains all info about hosts")
    # "hosts": {
    #     "uuid": {
    #         "msg": X
    #         "status_time": Z
    #         "status": OneOf("processing", "finished")
    #     }
    # }

    revision = LongField(default=0, help_text="Current stage info revision, defence against concurrent modifications")

    def __str__(self):
        return "<StageInfo: \"{}\":{}:{}>".format(self.name, self.action_type, self.status)

    def deserialize_stage_desc(self, stage_desc):
        from walle.scenario.stages import StageDesc

        if type(stage_desc) is StageDesc:
            return stage_desc

        stage_params = self.deserialize_params(stage_desc['params'] or {})

        return StageDesc(
            stage_class=StageRegistry.get(stage_desc['stage_class']),
            params=stage_params,
            conditions=stage_desc.get('conditions'),
        )

    def deserialize_params(self, params):
        params = params or {}
        if params.get('children'):
            params['children'] = [self.deserialize_stage_desc(stage_desc) for stage_desc in params['children']]
        if params.get('stage_map'):
            params['stage_map'] = [self.deserialize_stage_desc(stage_desc) for stage_desc in params['stage_map']]
        return params

    def deserialize(self):
        from walle.scenario.mixins import CommonParentStageHandler

        self.deserialize_params(self.params)
        stage_cls = StageRegistry.get(self.name)
        if self.stages or issubclass(stage_cls, CommonParentStageHandler):
            children = [child.deserialize() for child in self.stages]
            return stage_cls(children, **self.params)
        return stage_cls(**self.params)

    def get_data(self, key, default=None):
        return self.data.get(key, default)

    def set_data(self, key, value):
        self.data[key] = value

    def update_data(self, key, value):
        self.data[key].update(value)

    def append_data(self, key, data):
        self.data[key] += data

    def contains_data(self, key):
        return key in self.data

    def remove_data(self, key):
        if key in self.data:
            del self.data[key]

    def set_stage_finished(self):
        self.status = StageStatus.FINISHED
        self.status_time = timestamp()

    def set_stage_processing(self):
        self.status = StageStatus.PROCESSING
        self.status_time = timestamp()

    def set_stage_cancelling(self):
        self.status = StageStatus.CANCELLING
        self.status_time = timestamp()

    def set_stage_cancelled(self):
        self.status = StageStatus.CANCELLED
        self.status_time = timestamp()

    def is_on_last_stage(self):
        return self.seq_num == len(self.stages) - 1

    def write_shared_data(self, key, val):
        self.shared_data[key] = val

    def read_shared_data(self, key):
        return self.shared_data[key]

    def set_stage_msg(self, msg):
        self.msg = msg
