import dataclasses
import logging
import typing as tp
from datetime import datetime

import walle.constants
import walle.host_fsm.control
import walle.network
import walle.projects
import walle.tasks
from sepelib.core import config
from sepelib.core.exceptions import LogicalError
from walle import authorization, host_operations
from walle import restrictions
from walle._tasks.task_args import (
    ProjectSwitchingArgs,
    PrepareTaskArgs,
    SetAssignedTaskArgs,
    PowerOffTaskArgs,
    VlanSwitchTaskArgs,
    RebootTaskArgs,
    HostReleaseArgs,
    SwitchToMaintenanceTaskArgs,
)
from walle._tasks.task_creator import (
    get_switch_project_task_stages,
    get_prepare_stages,
    get_assign_stages,
    get_power_off_task_stages,
    get_vlan_switch_task_stages,
    get_reboot_task_stages,
    get_release_host_stages,
    get_switch_to_maintenance_stages,
)
from walle._tasks.task_provider import schedule_task_from_api, schedule_task_from_scenario
from walle.clients import cms, startrek, eine, qloud, bot
from walle.clients.eine import ProfileMode
from walle.constants import NetworkTarget, HostType
from walle.errors import BotEmptyNameError, WalleError, InvalidHostStateError
from walle.expert.types import CheckType, CheckGroup, CheckStatus
from walle.fsm_stages.change_disk import restart_task_with_host_power_off
from walle.host_status import cancel_task
from walle.hosts import Host, HostOperationState, TaskType, HostState, HostStatus
from walle.locks import HostInterruptableLock
from walle.maintenance_plot.model import MaintenancePlot
from walle.models import timestamp
from walle.operations_log.constants import Operation
from walle.projects import Project
from walle.scenario.common import get_host_group_maintenance_plot
from walle.scenario.constants import (
    ScriptArgs,
    TicketStatus,
    ALL_CANCELATION_WORK_STATUSES,
    WORK_STATUS_LABEL_NAME,
    ScenarioWorkStatus,
    StageName,
    TemplatePath,
    CustomTemplatePath,
    ScriptName,
)
from walle.scenario.data_storage.base import BaseScenarioDataStorage
from walle.scenario.definitions.base import get_data_storage
from walle.scenario.iteration_strategy import SequentialIterationActionStrategy
from walle.scenario.marker import Marker
from walle.scenario.mixins import (
    BaseStage,
    HostGroupStage,
    HostStage,
    ParentStageHandler,
    MultiActionStage,
    Stage,
    HostParentStageHandler,
)
from walle.scenario.qloud import (
    make_host_added_not_ready,
    make_host_added_ready,
    ensure_host_not_in_qloud,
    is_qloud_project,
)
from walle.scenario.scenario import Scenario
from walle.scenario.stage_info import StageInfo, StageRegistry
from walle.scenario.utils import check_last_operation_from_oplog
from walle.tasks import schedule_profile, schedule_redeploy, schedule_dns_check
from walle.util.misc import drop_none, dummy_context
from walle.util.tasks import check_post_code_allowed
from walle.util.template_loader import JinjaTemplateRenderer
from walle.views.helpers.maintenance import change_host_maintenance
from walle.util.approvement_tools import ApproveClient, CreateStartrekCommentRequest


log = logging.getLogger(__name__)


def _resolve_target_project_id(stage):
    if stage.target_project_id is not None:
        return stage.target_project_id
    return qloud.get_segment_project(stage.target_hardware_segment)


def _log_stage(stage_info, scenario, host=None):
    if host:
        log.info(
            "Running %s.%s on host %s for scenario #%s",
            stage_info.name,
            stage_info.action_type,
            host.uuid,
            scenario.scenario_id,
        )
    else:
        log.info("Running %s.%s for scenario #%s", stage_info.name, stage_info.action_type, scenario.scenario_id)


@dataclasses.dataclass(frozen=True)
class StageDesc:
    stage_class: tp.Type[BaseStage] = None
    params: dict = None
    conditions: dict = None

    def to_mongo(self, *args, **kwargs):
        params_to_mongo = None
        if self.params:
            params_to_mongo = self.params.copy()
            if params_to_mongo.get('children'):
                params_to_mongo['children'] = [stage_desc.to_mongo() for stage_desc in params_to_mongo['children']]
            if params_to_mongo.get('stage_map'):
                params_to_mongo['stage_map'] = [stage_desc.to_mongo() for stage_desc in params_to_mongo['stage_map']]

        return {
            'stage_class': self.stage_class.__name__,
            'params': params_to_mongo,
            'conditions': self.conditions,
        }


@StageRegistry.register(StageName.SMS)
@StageRegistry.register(StageName.SRS)
class ScenarioRootStage(ParentStageHandler, Stage):
    """Starts and ends scenario execution"""

    _cancel_message_template = "Scenario was canceled. Scenario issuer was notified in StarTrek ticket {}"

    def run(self, stage_info, scenario):
        return self.execute_current_stage(stage_info, scenario)

    def cleanup(self, stage_info, scenario):
        """Leave a comment in scenario's ticket and call issuer."""
        startrek_client = startrek.get_client()
        template_renderer = JinjaTemplateRenderer()
        params = dict(scenario_id=scenario.scenario_id)
        rendered_output = template_renderer.render_template(TemplatePath.STARTREK_SCENARIO_CANCELLED, **params)
        startrek_client.add_comment(
            issue_id=scenario.ticket_key, text=rendered_output, summonees=self._get_summonees(scenario)
        )
        return Marker.success(message=self._cancel_message_template.format(scenario.ticket_key))

    @staticmethod
    def _get_summonees(scenario):
        result = [
            scenario.issuer.strip("@"),
        ]

        config_summonees = config.get_value("scenario.stages.scenario_root_stage.startrek_scenario_canceled_summonees")
        if config_summonees:
            result += config_summonees
        return result


@StageRegistry.register(StageName.DumpApproversListStage)
class DumpApproversListStage(Stage):
    def run(self, stage_info, scenario):
        data_storage = get_data_storage(scenario)
        plots = {gs.group_id: gs.source.get_maintenance_plot() for gs in data_storage.read_host_groups_sources()}

        host_groups_info = {
            group_id: {
                'maintenance_plot_id': mp.id,
                'approvers': mp.get_approvers(),
                'hosts': [],
            }
            for group_id, mp in plots.items()
        }
        hosts_invs = {host_info.inv for host_info in scenario.hosts.values()}
        scenario_hosts = {host.inv: host.name for host in Host.objects(inv__in=hosts_invs).only("inv", "name")}

        for host_info in scenario.hosts.values():
            host_groups_info[host_info.group]['hosts'].append(
                {
                    'inv': host_info.inv,
                    'name': scenario_hosts[host_info.inv],
                }
            )
        comment_info = self.add_st_comment(scenario, host_groups_info)
        comment_id = comment_info["id"]

        return Marker.success(
            message=f"Approvers list is successfully dumped to the ticket "
            f"'{scenario.ticket_key}' with comment_id '{comment_id}'"
        )

    def add_st_comment(self, scenario, host_groups_info) -> dict:
        startrek_client = startrek.get_client()
        template_renderer = JinjaTemplateRenderer()
        params = dict(scenario_id=scenario.scenario_id, host_groups_info=host_groups_info)
        rendered_output = template_renderer.render_template(CustomTemplatePath.STARTREK_DUMP_APPROVERS_LIST, **params)
        return startrek_client.add_comment(issue_id=scenario.ticket_key, text=rendered_output)


@StageRegistry.register(StageName.CallOnResponsibleForLoadReturn)
class CallOnResponsibleForLoadReturn(HostGroupStage):
    """Get responsibles to ticket after NOC/ITDC work is done."""

    def run(self, stage_info: StageInfo, scenario: Scenario, host_group_id: int):
        self.log_info(scenario, host_group_id, msg="start running")

        data_storage = get_data_storage(scenario)
        maintenance_plot = self._get_host_group_maintenance_plot(host_group_id, data_storage)
        scenario_settings = maintenance_plot.get_scenario_settings(scenario.scenario_type)
        approvers = maintenance_plot.get_approvers()
        fqdns_of_hosts = self._collect_fqdns_of_hosts(host_group_id, scenario)

        # TODO: generate stages
        try:
            if scenario_settings.enable_manual_approval_after_hosts_power_off:
                comment_id = self._add_st_comment(
                    scenario.ticket_key, data_storage, host_group_id, tuple(approvers), fqdns_of_hosts
                )

                return Marker.success(
                    message=f"Got responsibles to ticket " f"'{scenario.ticket_key}' with comment_id '{comment_id}'"
                )
        except AttributeError as e:
            log.info(
                "scenario %s, group_id %s, manual approval isn't enabled: %s", scenario.scenario_id, host_group_id, e
            )

        return Marker.success(message="Stage skipped, option `enable_manual_approval_after_hosts_power_off` disabled")

    def _collect_fqdns_of_hosts(self, host_group_id: int, scenario: Scenario) -> list[str]:
        invs = [host_info.inv for host_info in scenario.hosts.values() if host_info.group == host_group_id]
        fqdns = [host.name for host in Host.objects(inv__in=invs).only("name")]
        return fqdns

    @staticmethod
    def _get_host_group_maintenance_plot(host_group_id: int, data_storage: BaseScenarioDataStorage) -> MaintenancePlot:
        for host_group_source in data_storage.read_host_groups_sources():
            if host_group_source.group_id == host_group_id:
                return host_group_source.source.get_maintenance_plot()
        raise WalleError("Can not find host group source with ID '%d' in scenario's data storage")

    def _get_host_group_name(self, data_storage: BaseScenarioDataStorage, host_group_id: int) -> str:
        # ABC service name as a host group name is fine for now.
        return self._get_host_group_maintenance_plot(
            host_group_id,
            data_storage,
        ).meta_info.abc_service_slug

    def _add_st_comment(
        self,
        scenario_ticket_key: str,
        data_storage: BaseScenarioDataStorage,
        host_group_id: int,
        summonees: tuple[str],
        fqdns_of_hosts: list[str],
    ) -> str:
        host_group_name = self._get_host_group_name(data_storage, host_group_id)
        approve_client = ApproveClient(startrek_client=startrek.get_client())
        text = JinjaTemplateRenderer().render_template(
            TemplatePath.STARTREK_SCENARIO_GET_RESPONSIBLES_TO_TICKET,
            fqdns=",".join(fqdns_of_hosts),
            host_group_name=host_group_name,
        )
        request = CreateStartrekCommentRequest(issue_id=scenario_ticket_key, summonees=summonees, text=text)
        return approve_client.create_startrek_comment(request)


@StageRegistry.register(StageName.ITDCWS)
class ITDCWorkflowStage(ParentStageHandler, Stage):
    """Stops child stages execution if ITDC maintenance operations were cancelled"""

    def run(self, stage_info, scenario):
        if self._itdc_work_is_cancelled(scenario):
            return Marker.success(message="ITDC work is cancelled")
        return self.execute_current_stage(stage_info, scenario)

    @staticmethod
    def _itdc_work_is_cancelled(scenario):
        return scenario.labels.get(WORK_STATUS_LABEL_NAME) in ALL_CANCELATION_WORK_STATUSES


@StageRegistry.register(StageName.HMS)
@StageRegistry.register(StageName.HRS)
class HostRootStage(HostParentStageHandler, HostStage):
    """Starts and ends host stages execution"""

    def __init__(self, children, **params):
        super().__init__(children, **params)
        for stage in children:
            if not isinstance(stage, HostStage):
                raise LogicalError

    def run(self, host_stage_info, scenario, host, scenario_stage_info):
        if host.type != HostType.SERVER:
            return Marker.success(message="That's an error. Only 'server' type can use HRS")

        marker = self.execute_current_stage(host_stage_info, scenario, host, scenario_stage_info)

        if marker.data:
            key, val = next(iter(marker.data.items()))
            host_stage_info.write_shared_data(key, val)

        log.info("HRS finished")

        return marker

    def serialize(self, uid="0"):
        stage_info = super().serialize(uid=uid)
        return stage_info


@StageRegistry.register(StageName.SwitchScenarioToHostUUID)
class SwitchScenarioToHostUUID(Stage):
    """Switches scenario on using hosts' UUIDs instead of FQDNs or inventory numbers"""

    def run(self, stage_info, scenario):
        invs = scenario.get_invs_of_hosts()
        for inv in invs:
            inv = str(inv)
            try:
                host_uuid = Host.objects.get(inv=inv).uuid
            except Host.DoesNotExist:
                raise RuntimeError("Tried to resolve UUID of host #{} with changed inventory number.".format(inv))
            scenario.hosts[host_uuid] = scenario.hosts[inv]
            del scenario.hosts[inv]

        scenario.uses_uuid_keys = True
        return Marker.success(message="Scenario switched to using host's UUIDs")


@StageRegistry.register(StageName.WaitStateStatusHostStage)
class WaitStateStatusHostStage(HostStage):
    """Waits until host is in specific project, in a given state and status"""

    def __init__(self, target_project, wait_state, wait_status):
        super().__init__(target_project=target_project, wait_state=wait_state, wait_status=wait_status)
        self.target_project = target_project
        self.wait_state = wait_state
        self.wait_status = wait_status

    def run(self, stage_info, scenario, host, *args, **kwargs):
        if not all(
            (host.status == self.wait_status, host.state == self.wait_state, host.project == self.target_project)
        ):

            msg_args = (host.state, host.status, host.project, self.wait_state, self.wait_status, self.target_project)
            msg = (
                "Host's state: '{}', host's status: '{}', host's project: '{}'. "
                "Awaited state: '{}', awaited status: '{}', awaited project: '{}'".format(*msg_args)
            )
            self.log_info(scenario, host, msg)

            return Marker.in_progress(message=msg)

        self.log_info(scenario, host, msg="stage completed successfully")
        return Marker.success(
            message="Host is in target project '{}', in required state '{}'"
            " and status '{}'".format(host.project, host.state, host.status)
        )


@StageRegistry.register(StageName.NoopStage)
class NoopStage(MultiActionStage, Stage):
    """Dummy stage, does nothing"""

    def __init__(self):
        super().__init__(SequentialIterationActionStrategy([self.action, self.check]))

    def action(self, stage_info, scenario, *args, **kwargs):
        _log_stage(stage_info, scenario)
        return Marker.success(message="Stage did nothing and finished")

    def check(self, stage_info, scenario, *args, **kwargs):
        _log_stage(stage_info, scenario)
        return Marker.in_progress(message="Stage did nothing and continued")

    def cleanup(self, stage_info, scenario, *args, **kwargs):
        raise ValueError("NoopStage cleanup launched")


@StageRegistry.register(StageName.CancelTaskStage)
class CancelTaskStage(HostStage):
    """Cancels current task of the host"""

    def run(self, stage_info, scenario, host, *args, **kwargs):
        _log_stage(stage_info, scenario, host)

        message = "Host doesn't have task"
        if all((host.state == HostState.ASSIGNED, host.task, scenario.get_host_info_by_host_obj(host).is_acquired)):
            reason = "CMS has not let this host switch to maintenance"
            with HostInterruptableLock(host.uuid, host.tier):
                cancel_task(authorization.ISSUER_WALLE, host, reason, ignore_maintenance=True, lock_class=dummy_context)
            message = "Task on the host has been canceled"

        return Marker.success(message=message)


@StageRegistry.register(StageName.SetAssignedStage)
class SetAssignedStage(MultiActionStage, HostStage):
    """Sets host to 'assigned' state"""

    def __init__(
        self, power_on=False, use_specific_checks=False, monitor_on_completion=True, skip_on_invalid_state=False
    ):
        super().__init__(
            SequentialIterationActionStrategy([self.action, self.check]),
            power_on=power_on,
            use_specific_checks=use_specific_checks,
            monitor_on_completion=monitor_on_completion,
            skip_on_invalid_state=skip_on_invalid_state,
        )
        self.power_on = power_on
        self.use_specific_checks = use_specific_checks
        self.monitor_on_completion = monitor_on_completion
        self.skip_on_invalid_state = skip_on_invalid_state

    @staticmethod
    def _is_host_in_scenario_maintenance(host, scenario):
        if host.state == HostState.MAINTENANCE and host.state_author == authorization.ISSUER_WALLE:
            if host.state_expire:
                return host.state_expire.ticket == scenario.ticket_key
            else:
                return host.ticket == scenario.ticket_key
        else:
            return False

    def action(self, stage_info, scenario, host, *args, **kwargs):
        _log_stage(stage_info, scenario, host)
        if host.state == HostState.ASSIGNED:
            return Marker.success(message="Host's state is set to 'assigned'")

        if not self._is_host_in_scenario_maintenance(host, scenario):
            return Marker.success(
                message="Host's state was set to 'maintenance' state not by Wall-e, "
                "scenario has no permission to change it to 'assigned'"
            )

        if not host.task:

            checks_for_use = None
            if self.use_specific_checks:
                if scenario.uses_uuid_keys:
                    checks_for_use = scenario.hosts[host.uuid].enabled_checks
                else:
                    checks_for_use = scenario.hosts[str(host.inv)].enabled_checks

            task_args = SetAssignedTaskArgs(
                issuer=authorization.ISSUER_WALLE,
                task_type=TaskType.AUTOMATED_ACTION,
                project=host.project,
                host_inv=host.inv,
                host_name=host.name,
                host_uuid=host.uuid,
                scenario_id=host.scenario_id,
                reason=scenario.ticket_key,
                with_auto_healing=self.monitor_on_completion,
                target_status=HostStatus.READY,
                checks_to_monitor=CheckGroup.NETWORK_AVAILABILITY,
                power_on=self.power_on,
                checks_for_use=checks_for_use,
                keep_task_id=True,
                monitor_on_completion=self.monitor_on_completion,
            )
            with HostInterruptableLock(host.uuid, host.tier):
                try:
                    schedule_task_from_scenario(host, task_args, get_assign_stages)
                except InvalidHostStateError:
                    if self.skip_on_invalid_state:
                        return Marker.success(message="Host has invalid state for this task, so Wall-e skipped it")
                    raise
            return Marker.success(message="Launched task that sets host's state to 'assigned'")
        return Marker.failure(message="Host has active task, scenario wait until it finishes")

    def check(self, stage_info, scenario, host, *args, **kwargs):
        _log_stage(stage_info, scenario, host)
        if host.task:
            return Marker.in_progress(message="Host has active task, scenario wait until it finishes")

        if host.state == HostState.ASSIGNED:
            return Marker.success(message="Host is in 'assigned' state")
        elif host.state == HostState.MAINTENANCE:
            return Marker.success(
                message="Host is in 'maintenance' state performed by user. The user must remove it himself."
            )
        else:
            return Marker.success(message="Host has unmanaged state, Wall-e skipped task.")


@StageRegistry.register(StageName.RemoveHostsStage)
class RemoveHostsStage(Stage):
    """Removes hosts from Wall-e"""

    def __init__(self, intermediate_project):
        super().__init__(intermediate_project=intermediate_project)
        self.intermediate_project = intermediate_project

    def run(self, stage_info, scenario):
        self.log_info(scenario, msg="start running")
        if self.intermediate_project:
            source_project = self.intermediate_project
        else:
            source_project = config.get_value("scenario.stages.add_hosts_stage.project")
        all_invs = {host_info.inv for host_info in scenario.hosts.values()}
        in_walle_hosts = [host for host in Host.objects.filter(inv__in=all_invs)]
        if len(in_walle_hosts) == 0:
            return Marker.success(message="Stage removed all hosts from scenario")
        for host in in_walle_hosts:
            if host.project != source_project:
                self.log_info(
                    scenario, msg="Can't remove host {}, should be in project: {}".format(host.inv, source_project)
                )
                continue
            host_query = dict(inv=host.inv)
            host_operations.instant_delete_host(
                authorization.ISSUER_WALLE, host, host_query, ignore_maintenance=True, reason=scenario.ticket_key
            )
            self.log_info(scenario, "host {} successfully removed from walle".format(host.inv))
        return Marker.in_progress(
            message="Stage can't remove all hosts, some of them not in {} project".format(source_project)
        )


@StageRegistry.register(StageName.AddHostsStage)
class AddHostsStage(Stage):
    """Adds hosts to Wall-e"""

    def __init__(self, intermediate_project):
        super().__init__(intermediate_project=intermediate_project)
        self.intermediate_project = intermediate_project

    def run(self, stage_info, scenario):
        self.log_info(scenario, msg="start running")
        try:
            target_project = self.intermediate_project
        except Exception:
            target_project = config.get_value("scenario.stages.add_hosts_stage.project")

        all_invs = {host_info.inv for host_info in scenario.hosts.values()}
        existing_invs = {
            host.inv for host in Host.objects.filter(inv__in=all_invs, type__ne=HostType.SHADOW_SERVER).only("inv")
        }
        failed_invs = []

        for inv_not_in_walle in all_invs - existing_invs:
            try:
                maintenance_properties = dict(
                    ticket_key=scenario.ticket_key, operation_state=HostOperationState.DECOMMISSIONED
                )
                host_operations.add_host(
                    authorization.ISSUER_WALLE,
                    target_project,
                    inv=inv_not_in_walle,
                    instant=True,
                    state=HostState.MAINTENANCE,
                    status=HostStatus.MANUAL,
                    reason=scenario.ticket_key,
                    maintenance_properties=maintenance_properties,
                )
                self.log_info(
                    scenario, "host {} successfully added to walle in maintenance/manual".format(inv_not_in_walle)
                )
            except BotEmptyNameError as e:
                self.log_error(
                    scenario, msg="failed to add host {} as maintenance/manual: {}".format(inv_not_in_walle, e)
                )

                try:
                    host_operations.add_host(
                        authorization.ISSUER_WALLE,
                        target_project,
                        inv=inv_not_in_walle,
                        instant=True,
                        state=HostState.FREE,
                        status=HostStatus.READY,
                        reason=scenario.ticket_key,
                    )
                    self.log_info(
                        scenario, "host {} successfully added to walle in free/ready".format(inv_not_in_walle)
                    )
                except Exception as e:
                    self.log_error(scenario, msg="failed to add host {} as free/ready: {}".format(inv_not_in_walle, e))
                    failed_invs.append(inv_not_in_walle)

        if all_invs != existing_invs:
            return Marker.in_progress(message="Failed to add hosts to Wall-e: {}".format(",".join(failed_invs)))
        return Marker.success(message="All hosts in Wall-e")


@StageRegistry.register(StageName.WaitForLabelOrTimeStage)
class WaitForLabelOrTimeStage(HostStage):
    """Waits until a certain time has passed, or a certain label has been written to scenario labels"""

    def __init__(
        self, label_name=None, label_values=None, idle_time=None, user_label=None, user_label_target_value=None
    ):
        # TODO(rocco66): remove label_name, label_values
        super().__init__(
            label_name=label_name,
            label_values=label_values,
            idle_time=idle_time,
            user_label=user_label,
            user_label_target_value=user_label_target_value,
        )

        self.idle_time = idle_time

        self.user_label = user_label
        self.user_label_target_value = user_label_target_value

    STAGE_END_TIME = "stage_end_time"

    def run(self, stage_info, scenario, host=None, scenario_stage_info=None):
        _log_stage(stage_info, scenario)
        if self._user_label_matched(scenario):
            Scenario.get_collection().update_one(
                {"_id": scenario.id}, {"$set": {f"labels.{WORK_STATUS_LABEL_NAME}": ScenarioWorkStatus.STARTED}}
            )
            return Marker.success(
                message=f"Label '{self.user_label}' has expected value '{scenario.labels.get(self.user_label)}'"
            )
        elif self._time_matched(stage_info):
            return Marker.success(
                message="Stage waited until expected date {}".format(self._get_suitable_datetime_format(stage_info))
            )
        else:
            if self.idle_time:
                message = "Stage will wait until expected date {}".format(
                    self._get_suitable_datetime_format(stage_info)
                )
            else:
                message = (
                    f"Label '{self.user_label}' hasn't expected value '{scenario.labels.get(self.user_label)}': "
                    f"work isb still being done outside of Wall-e"
                )
            return Marker.in_progress(message=message)

    def _user_label_matched(self, scenario):
        if self.user_label:
            return scenario.labels.get(self.user_label) == self.user_label_target_value
        return False

    def _time_matched(self, stage_info):
        if self.idle_time:

            if self.STAGE_END_TIME not in stage_info.data:
                stage_info.data[self.STAGE_END_TIME] = timestamp() + self.idle_time

            return stage_info.data[self.STAGE_END_TIME] <= timestamp()

        return False

    def _get_suitable_datetime_format(self, stage_info):
        return datetime.fromtimestamp(stage_info.data[self.STAGE_END_TIME]).strftime("%d.%m.%Y %H:%M")


@StageRegistry.register(StageName.SetStartedWorkStatusStage)
class SetStartedWorkStatusStage(Stage):
    """Sets scenario status to 'started' value"""

    def run(self, stage_info, scenario):
        scenario.set_works_status_label(ScenarioWorkStatus.STARTED)
        return Marker.success(message="Works status set to 'started'")


@StageRegistry.register(StageName.SetApprovementWorkStatusStage)
class SetApprovementWorkStatusStage(Stage):
    """Sets scenario status to 'approvement' value"""

    def run(self, stage_info, scenario):
        scenario.set_works_status_label(ScenarioWorkStatus.APPROVEMENT)
        return Marker.success(message="Works status set to 'approvement'")


@StageRegistry.register(StageName.SetAcquiringPermissionStatusStage)
class SetAcquiringPermissionStatusStage(Stage):
    """Sets scenario status to 'acquiring permission' value"""

    def run(self, stage_info: StageInfo, scenario: Scenario) -> Marker:
        scenario.set_works_status_label(ScenarioWorkStatus.ACQUIRING_PERMISSION)
        return Marker.success(message="Works status set to 'acquiring-permission'")


@StageRegistry.register(StageName.SetReadyWorkStatusStage)
class SetReadyWorkStatusStage(Stage):
    """Sets scenario status to 'ready' value"""

    def run(self, stage_info, scenario):
        scenario.set_works_status_label(ScenarioWorkStatus.READY)
        return Marker.success(message="Works status set to 'ready'")


@StageRegistry.register(StageName.SetFinishingWorkStatusStage)
class SetFinishingWorkStatusStage(Stage):
    """Sets scenario status to 'finishing' value"""

    def run(self, stage_info, scenario):
        scenario.set_works_status_label(ScenarioWorkStatus.FINISHING)
        return Marker.success(message="Works status set to 'finishing'")


@StageRegistry.register(StageName.SetFinishedWorkStatusStage)
class SetFinishedWorkStatusStage(Stage):
    """Sets scenario status to 'finished' value"""

    def run(self, stage_info, scenario):
        scenario.set_works_status_label(ScenarioWorkStatus.FINISHED)
        return Marker.success(message="Works status set to 'finished'")


@StageRegistry.register(StageName.SetLabelStage)
class SetLabelStage(Stage):
    """Sets value of a specific scenario label"""

    def __init__(self, label_name=None, label_value=None, label_value_alternative=None):
        super().__init__(
            label_name=label_name, label_value=label_value, label_value_alternative=label_value_alternative
        )
        self.label_name = label_name
        self.label_value = label_value
        self.label_value_alternative = label_value_alternative

    def meets_condition(self, stage_info, scenario):
        return True

    def run(self, stage_info, scenario):
        if self.meets_condition(stage_info, scenario) or not self.label_value_alternative:
            scenario.labels[self.label_name] = self.label_value
            return Marker.success(
                message="Set a value '{}' for the label '{}'".format(self.label_name, self.label_value)
            )
        else:
            scenario.labels[self.label_name] = self.label_value_alternative
            return Marker.success(
                message="Set a value '{}' for the label '{}'".format(self.label_name, self.label_value_alternative)
            )


@StageRegistry.register(StageName.SetLabelIfAllScheduledStage)
class SetLabelIfAllScheduledStage(SetLabelStage):
    def meets_condition(self, stage_info, scenario):
        return all(host_info.is_acquired for host_info in scenario.hosts.values())


@StageRegistry.register(StageName.AddStartrekMessageStage)
class AddStartrekMessageStage(Stage):
    """Writes a comment to Startrek ticket"""

    def __init__(self, template_path):
        super().__init__(template_path=template_path)
        self.template_path = template_path

    def run(self, stage_info, scenario):
        startrek_client = startrek.get_client()
        template_renderer = JinjaTemplateRenderer()
        params = dict(scenario_id=scenario.scenario_id)
        rendered_output = template_renderer.render_template(self.template_path, **params)
        startrek_client.add_comment(issue_id=scenario.ticket_key, text=rendered_output)
        return Marker.success(message="Added a StarTrek message in the ticket '{}'".format(scenario.ticket_key))


@StageRegistry.register(StageName.AddStartrekHostGroupMessageStage)
class AddStartrekHostGroupMessageStage(HostGroupStage):
    """Writes a comment to Startrek ticket, supports number of host group."""

    def __init__(self, template_path):
        super().__init__(template_path=template_path)
        self.template_path = template_path

    def run(self, stage_info, scenario, host_group_id):
        startrek_client = startrek.get_client()
        template_renderer = JinjaTemplateRenderer()
        params = dict(scenario_id=scenario.scenario_id, host_group_id=host_group_id)
        rendered_output = template_renderer.render_template(self.template_path, **params)
        startrek_client.add_comment(issue_id=scenario.ticket_key, text=rendered_output)
        return Marker.success()


@StageRegistry.register(StageName.AddStartrekTagStage)
class AddStartrekTagStage(Stage):
    """Adds a tag to Startrek ticket"""

    def __init__(self, tag):
        super().__init__(tag=tag)

        self.tag = tag

    def run(self, stage_info, scenario):
        startrek_client = startrek.get_client()
        startrek_client.modify_issue(scenario.ticket_key, {"tags": {"add": self.tag}})

        return Marker.success(message="Added a tag '{}' to the ticket '{}'".format(self.tag, scenario.ticket_key))


@StageRegistry.register(StageName.ExecuteTicketTransitionStage)
class ExecuteTicketTransitionStage(Stage):
    """Changes Startrek ticket status"""

    def __init__(self, ticket_transition_state, ticket_transition_resolution=None):
        super().__init__(
            ticket_transition_state=ticket_transition_state, ticket_transition_resolution=ticket_transition_resolution
        )
        self.transition_state = ticket_transition_state
        self.transition_resolution = ticket_transition_resolution

    def run(self, stage_info, scenario):
        issue_params = {"resolution": self.transition_resolution} if self.transition_resolution else None
        transition_args = drop_none(
            dict(issue_id=scenario.ticket_key, transition=self.transition_state, issue_params=issue_params)
        )

        startrek_client = startrek.get_client()

        ticket_state = startrek_client.get_issue(scenario.ticket_key)["status"]["key"]

        if ticket_state == TicketStatus.CLOSED:
            return Marker.success(message="Ticket is closed")

        if ticket_state != self.transition_state:
            startrek_client.execute_transition(**transition_args)

        return Marker.success(message="Transition of ticket state was performed")


@StageRegistry.register(StageName.SwitchToMaintenanceHostStage)
class SwitchToMaintenanceHostStage(HostStage):
    """Launches a task that sets host's state to 'maintenance'."""

    def __init__(
        self,
        ignore_cms=False,
        power_off=True,
        workdays_only=True,
        cms_task_action=cms.CmsTaskAction.PROFILE,
        operation_state=HostOperationState.DECOMMISSIONED,
        **kwargs,
    ):

        super().__init__(
            ignore_cms=ignore_cms,
            cms_task_action=cms_task_action,
            power_off=power_off,
            workdays_only=workdays_only,
            operation_state=operation_state,
            **kwargs,
        )

        self.ignore_cms = ignore_cms
        self.cms_task_action = cms_task_action
        self.power_off = power_off
        self.workdays_only = workdays_only
        self.operation_state = operation_state

    def _schedule_switch_to_maintenance(self, scenario: Scenario, host: Host) -> Marker:
        task_args = SwitchToMaintenanceTaskArgs(
            allowed_states=[HostState.ASSIGNED, HostState.PROBATION],
            issuer=authorization.ISSUER_WALLE,
            task_type=TaskType.AUTOMATED_ACTION,
            project=host.project,
            host_inv=host.inv,
            host_name=host.name,
            host_uuid=host.uuid,
            scenario_id=host.scenario_id,
            ignore_cms=self.get_ignore_cms_param(scenario, host),
            cms_action=self.cms_task_action,
            task_group=self._task_group(scenario),
            disable_admin_requests=False,
            power_off=self.power_off,
            ticket_key=scenario.ticket_key,
            force_new_task=False,
            monitor_on_completion=False,
            keep_downtime=True,
            reason=scenario.ticket_key,
            operation_state=self.operation_state,
            workdays_only=self.workdays_only,
        )

        with HostInterruptableLock(host.uuid, host.tier):
            schedule_task_from_scenario(host, task_args, get_switch_to_maintenance_stages)
        self.log_info(scenario, host, msg="switch-to-maintenance scheduled successfully")
        return Marker.in_progress(message="Launched a task that sets host's state to 'maintenance'")

    def get_ignore_cms_param(self, scenario, host):
        return self.get_ignore_cms_value(scenario, host, self.ignore_cms)

    @staticmethod
    def _task_group(scenario: Scenario) -> str:
        return "{}".format(scenario.id)

    @staticmethod
    def _is_host_in_passable_state(host: Host) -> bool:
        return host.state in {HostState.MAINTENANCE, HostState.FREE}

    @staticmethod
    def _is_host_ticket_equal_scenario_ticket(scenario: Scenario, host: Host) -> bool:
        return (
            host.ticket == scenario.ticket_key or host.state_expire and host.state_expire.ticket == scenario.ticket_key
        )

    def _change_maintenance(self, scenario: Scenario, host: Host):
        if host.state == HostState.MAINTENANCE and not self._is_host_ticket_equal_scenario_ticket(scenario, host):
            change_params = dict(
                ticket_key=scenario.ticket_key,
                operation_state=self.operation_state,
                reason=scenario.ticket_key,
                timeout_status=HostStatus.READY,
            )
            change_host_maintenance(issuer=authorization.ISSUER_WALLE, host=host, **change_params)
            self.log_info(scenario, host, msg="change-maintenance scheduled successfully: {}".format(change_params))

    def run(self, stage_info: StageInfo, scenario: Scenario, host: Host, *args, **kwargs) -> Marker:
        self.log_info(scenario, host, msg="start running")

        if host.task:
            if scenario.scenario_type == ScriptName.SET_MAINTENANCE and host.task.allows_power_off():
                # Temporary solution, we need something better.
                # We get here only as part of the process of power off hosts during disk changes in the ITDC
                restart_task_with_host_power_off(host, scenario)
                self.log_info(scenario, host, msg="task was restarted for powering off")
                return Marker.success(message="Launched a task that reboot the host")
            else:
                self.log_info(scenario, host, msg="wait for task: {}".format(host.task.status))
                return Marker.in_progress(message="Host has active task, scenario will wait until it ends")

        if self._is_host_in_passable_state(host):
            self._change_maintenance(scenario, host)
            self.log_info(scenario, host, msg="stage completed successfully")
            return Marker.success(message="Host is in 'maintenance' or 'free' state")

        return self._schedule_switch_to_maintenance(scenario, host)


def _is_host_powered_off(host):
    try:
        return host.get_ipmi_client().is_power_on() is False
    except Exception:
        return False


@StageRegistry.register(StageName.PowerOffHostStage)
class PowerOffHostStage(HostStage):
    """Launches a task that powers off the host"""

    def __init__(self, ignore_cms=False, operation_state=HostOperationState.OPERATION):
        super().__init__(ignore_cms=ignore_cms, operation_state=operation_state)
        self.ignore_cms = ignore_cms
        self.operation_state = operation_state

    def _schedule_power_off(self, scenario, host):
        task_args = PowerOffTaskArgs(
            issuer=authorization.ISSUER_WALLE,
            task_type=TaskType.AUTOMATED_ACTION,
            project=host.project,
            host_inv=host.inv,
            host_name=host.name,
            host_uuid=host.uuid,
            scenario_id=host.scenario_id,
            reason=scenario.ticket_key,
            ignore_maintenance=True,
            with_auto_healing=False,
            ignore_cms=self.get_ignore_cms_value(scenario, host, self.ignore_cms),
            allowed_states=HostState.ALL,
        )

        with HostInterruptableLock(host.uuid, host.tier):
            schedule_task_from_api(host, task_args, get_power_off_task_stages)
            self.log_info(scenario, host, msg="power-off scheduled successfully")

            if host.state == HostState.MAINTENANCE:
                change_params = dict(
                    ticket_key=scenario.ticket_key,
                    operation_state=self.operation_state,
                    reason=scenario.ticket_key,
                    timeout_status=HostStatus.READY,
                )
                change_host_maintenance(issuer=authorization.ISSUER_WALLE, host=host, **change_params)
                self.log_info(scenario, host, msg="change-maintenance scheduled successfully: {}".format(change_params))

        return Marker.in_progress(message="Launched a task that powers off the host")

    @staticmethod
    def _is_host_ticket_equal_scenario_ticket(scenario, host):
        return (
            host.ticket == scenario.ticket_key or host.state_expire and host.state_expire.ticket == scenario.ticket_key
        )

    @classmethod
    def _is_host_in_maintenance_by_scenario(cls, scenario, host):
        return all(
            (
                host.state == HostState.MAINTENANCE,
                cls._is_host_ticket_equal_scenario_ticket(scenario, host),
                host.operation_state == HostOperationState.DECOMMISSIONED,
                _is_host_powered_off(host),
            )
        )

    @staticmethod
    def _is_host_in_free_and_power_off(host):
        return all((host.state == HostState.FREE, _is_host_powered_off(host)))

    def run(self, stage_info, scenario, host, *args, **kwargs):
        self.log_info(scenario, host, msg="start running")

        if host.task:
            self.log_info(scenario, host, msg="wait for task: {}".format(host.task.status))
            return Marker.in_progress(message="Host has active task, scenario waits until it finishes")

        elif self._is_host_in_maintenance_by_scenario(scenario, host):
            self.log_info(scenario, host, msg="Stage completed, host is in maintenance state")
            return Marker.success(message="Host is powered off")

        elif self._is_host_in_free_and_power_off(host):
            self.log_info(scenario, host, msg="Stage completed, host is in 'free' state")
            return Marker.success(message="Host is powered off")

        elif _is_host_powered_off(host):
            self.log_info(scenario, host, msg="Host is powered off right now")
            return Marker.success(message="Host is powered off")

        return self._schedule_power_off(scenario, host)


@StageRegistry.register(StageName.LiberateFromQloudHostStage)
class LiberateFromQloudHostStage(HostStage):
    """Checks if host is in use by Qloud and removes it from there"""

    def run(self, stage_info, scenario, host, *args, **kwargs):
        if ensure_host_not_in_qloud(
            host, "Removing host from qloud for scenario: [{}]{}".format(scenario.scenario_id, scenario.name)
        ):
            return Marker.success(message="Host is not in use by Qloud")
        return Marker.in_progress(message="Host is in use by Qloud, scenario is trying to remove it from there")


@StageRegistry.register(StageName.SwitchProjectHostStage)
class SwitchProjectHostStage(HostStage):
    """Launches a task that switches project of the host"""

    def __init__(
        self,
        ignore_cms=False,
        target_project_id=None,
        target_hardware_segment=None,
        release=True,
        monitor_on_completion=False,
    ):
        super().__init__(
            target_project_id=target_project_id,
            target_hardware_segment=target_hardware_segment,
            ignore_cms=ignore_cms,
            release=release,
            monitor_on_completion=monitor_on_completion,
        )
        self.target_project_id = target_project_id
        self.target_hardware_segment = target_hardware_segment
        self.ignore_cms = ignore_cms
        self.release = release
        self.monitor_on_completion = monitor_on_completion

    def _schedule_switch_project(self, scenario, host, target_project, current_project):
        # Qloud hosts should switch without cms
        ignore_cms = self.get_ignore_cms_value(scenario, host, self.ignore_cms) or is_qloud_project(host.project)

        task_args = ProjectSwitchingArgs(
            issuer=authorization.ISSUER_WALLE,
            task_type=TaskType.AUTOMATED_ACTION,
            host_inv=host.inv,
            host_name=host.name,
            host_uuid=host.uuid,
            project=host.project,
            scenario_id=host.scenario_id,
            release=self.release,
            erase_disks=self.release,
            force=False,
            ignore_maintenance=True,
            ignore_cms=ignore_cms,
            disable_admin_requests=False,
            current_project_id=current_project.id,
            target_project_id=target_project.id,
            target_project_bot_project_id=target_project.bot_project_id,
            host_state=host.state,
            monitor_on_completion=self.monitor_on_completion,
            cms_task_id=host.cms_task_id,
            force_new_cms_task=False,
            operation_restrictions=None,
            reason=scenario.ticket_key,
        )
        with HostInterruptableLock(host.uuid, host.tier):
            schedule_task_from_scenario(host, task_args, get_switch_project_task_stages)
        self.log_info(scenario, host, msg="switch-project scheduled successfully")
        return Marker.in_progress(message="Launched a task that switches project of the host")

    def run(self, stage_info, scenario, host, *args, **kwargs):
        self.log_info(scenario, host, msg="start running")

        if not ensure_host_not_in_qloud(
            host, "Removing host from qloud for scenario: [{}]{}".format(scenario.scenario_id, scenario.name)
        ):
            self.log_info(scenario, host, msg="host is not liberated from qloud")
            return Marker.in_progress(message="Host is in use by Qloud, scenario is trying to remove it from there")

        if host.task:
            self.log_info(scenario, host, msg="wait for task: {}".format(host.task.status))
            return Marker.in_progress(message="Host has active task, scenario waits until it finishes")

        current_project = host.get_project()
        target_project = walle.projects.get_by_id(_resolve_target_project_id(self), fields=["bot_project_id"])

        if current_project.id == target_project.id:
            self.log_info(scenario, host, msg="stage completed successfully")
            return Marker.success(message="Host is in target project '{}'".format(current_project.id))

        return self._schedule_switch_project(scenario, host, target_project, current_project)


@StageRegistry.register(StageName.PrepareHostStage)
class PrepareHostStage(HostStage):
    """Launches a task that runs 'prepare' on the host"""

    def __init__(self, ignore_cms=False, target_project_id=None, target_hardware_segment=None):
        super().__init__(
            target_project_id=target_project_id, target_hardware_segment=target_hardware_segment, ignore_cms=ignore_cms
        )
        self.target_project_id = target_project_id
        self.target_hardware_segment = target_hardware_segment
        self.ignore_cms = ignore_cms

    @staticmethod
    def _is_host_waits_liberation_from_probation(host):
        return host.state == HostState.PROBATION and host.status == HostStatus.READY

    def _is_host_ready(self, host):
        target_project = _resolve_target_project_id(self)
        return all((host.state == HostState.ASSIGNED, host.status == HostStatus.READY, host.project == target_project))

    @staticmethod
    def _is_host_got_new_hostname(host):
        return host.state in {HostState.ASSIGNED, HostState.PROBATION}

    def _schedule_prepare(self, scenario, host):
        checks_to_monitor = CheckGroup.NETWORK_AVAILABILITY + CheckGroup.OS_CONSISTENCY
        network_update_args = host.deduce_profile_configuration(profile_mode=eine.ProfileMode.SWP_UP)
        (
            host_provisioner,
            host_config,
            host_deploy_tags,
            host_deploy_network,
            host_deploy_config_policy,
            deploy_configuration,
        ) = host.deduce_deploy_configuration()
        profile_configuration = host.deduce_profile_configuration(profile_mode=ProfileMode.DANGEROUS_HIGHLOAD_TEST)
        bot_project_id = host.get_project(["bot_project_id"]).bot_project_id
        update_firmware_configuration = host.deduce_profile_configuration(profile_mode=ProfileMode.FIRMWARE_UPDATE)

        task_args = PrepareTaskArgs(
            issuer=authorization.ISSUER_WALLE,
            task_type=TaskType.AUTOMATED_ACTION,
            project=host.project,
            host_inv=host.inv,
            host_name=host.name,
            host_uuid=host.uuid,
            scenario_id=host.scenario_id,
            ignore_cms=self.get_ignore_cms_value(scenario, host, self.ignore_cms),
            disable_admin_requests=False,
            monitor_on_completion=True,
            with_auto_healing=True,
            keep_fqdn=False,
            checks_to_monitor=checks_to_monitor,
            ignore_maintenance=True,
            network_target=walle.constants.NetworkTarget.PROJECT,
            network_update_args=network_update_args,
            deploy_configuration=deploy_configuration,
            host_provisioner=host_provisioner,
            host_config=host_config,
            host_deploy_tags=host_deploy_tags,
            host_deploy_network=host_deploy_network,
            host_deploy_config_policy=host_deploy_config_policy,
            bot_project_id=bot_project_id,
            skip_profile=False,
            profile_configuration=profile_configuration,
            reason=scenario.ticket_key,
            update_firmware_configuration=update_firmware_configuration,
            firmware_update_needed=True,
            operation_restrictions=None,
        )

        with HostInterruptableLock(host.uuid, host.tier):
            schedule_task_from_scenario(host, task_args, get_prepare_stages)
        self.log_info(scenario, host, msg="prepare scheduled successfully")
        return Marker.in_progress(message="Launched a task that runs 'prepare' on the host")

    def run(self, stage_info, scenario, host, *args, **kwargs):
        self.log_info(scenario, host, msg="start running")

        # If we adding host to qloud, we must introduce it to qloud's scheduler
        if (
            self._is_host_got_new_hostname(host)
            and self.target_hardware_segment is not None
            and not self._is_host_ready(host)
        ):
            if not make_host_added_not_ready(
                host.name,
                self.target_hardware_segment,
                "Adding host to qloud with scenario {} [{}].".format(scenario.name, scenario.scenario_id),
            ):
                self.log_info(scenario, host, msg="wait for host to be added in qloud")
                return Marker.in_progress(message="Adding host to Qloud's scheduler")

        if host.task:
            self.log_info(scenario, host, msg="wait for task: {}".format(host.task.status))
            return Marker.in_progress(message="Host has active task, scenario waits until it finishes")

        if self._is_host_waits_liberation_from_probation(host):
            message = "Wait until host gets out from 'probation' state; host's status: '{}'".format(host.status)
            self.log_info(scenario, host, msg=message)
            return Marker.in_progress(message=message)

        if self._is_host_ready(host):
            # If we adding host to qloud, we must ensure, that host is ready for qloud's allocations
            if self.target_hardware_segment is not None:

                if not make_host_added_ready(
                    host.name,
                    self.target_hardware_segment,
                    "Adding host to qloud with scenario {} [{}]. Making ready.".format(
                        scenario.name, scenario.scenario_id
                    ),
                ):
                    self.log_info(scenario, host, msg="wait for host to be ready in qloud")
                    return Marker.in_progress(message="Adding host to Qloud's scheduler")

            self.log_info(scenario, host, msg="stage completed successfully")
            return Marker.success(
                message="Host is in target project '{}', "
                "in 'assigned' state and in 'ready' status".format(host.project)
            )

        return self._schedule_prepare(scenario, host)


@StageRegistry.register(StageName.WaitAssignedReadyHostStage)
class WaitAssignedReadyHostStage(HostStage):
    def run(self, stage_info, scenario, host, *args, **kwargs):
        target_project = scenario.script_args[ScriptArgs.TARGET_PROJECT]

        if not all((host.status == HostStatus.READY, host.state == HostState.ASSIGNED, host.project == target_project)):
            message = (
                "Stage wait until host in assigned state, ready status, in project '{}'. "
                "Now host in '{}' state, in '{}' status, in '{}' project".format(
                    target_project, host.state, host.status, host.project
                )
            )
            return Marker.in_progress(message=message)

        return Marker.success(message="Host in assigned state, in ready status, in '{}' project".format(target_project))


@StageRegistry.register(StageName.PrepareForWorkStage)
class PrepareForWorkStage(SwitchToMaintenanceHostStage):
    """Launches a task that checks if host's CMS supports NOC-scenario and sets host's state to 'maintenance'"""

    def __init__(
        self,
        ignore_cms=False,
        cms_task_action=cms.CmsTaskAction.TEMPORARY_UNREACHABLE,
        power_off=False,
        workdays_only=False,
        operation_state=HostOperationState.OPERATION,
    ):

        super().__init__(
            ignore_cms=ignore_cms,
            cms_task_action=cms_task_action,
            power_off=power_off,
            workdays_only=workdays_only,
            operation_state=operation_state,
        )

    def _change_maintenance(self, scenario, host):
        return


@StageRegistry.register(StageName.ProfileHostStage)
class ProfileHostStage(HostStage):
    """Launches a task that runs 'profile' on the host"""

    def __init__(
        self,
        ignore_cms=False,
        profile_mode=ProfileMode.SWP_UP,
        monitor_on_completion=True,
        skip_on_invalid_state=False,
        **kwargs,
    ):
        super().__init__(
            ignore_cms=ignore_cms,
            profile_mode=profile_mode,
            monitor_on_completion=monitor_on_completion,
            skip_on_invalid_state=skip_on_invalid_state,
            **kwargs,
        )
        self.ignore_cms = ignore_cms
        self.profile_mode = profile_mode
        self.monitor_on_completion = monitor_on_completion
        self.skip_on_invalid_state = skip_on_invalid_state

    def run(self, stage_info, scenario, host, *args, **kwargs):
        self.log_info(scenario, host, msg="start running")

        if host.task:
            self.log_info(scenario, host, msg="wait for task: {}".format(host.task.status))
            return Marker.in_progress(message="Host has active task, scenario waits until it finishes")

        if check_last_operation_from_oplog(host, Operation.PROFILE, scenario.scenario_id):
            self.log_info(scenario, host, msg="stage completed successfully")
            return Marker.success(message="'Profile' task finished")

        try:
            return self._schedule_profile(scenario, host)
        except InvalidHostStateError:
            if self.skip_on_invalid_state:
                return Marker.success(message="Host has invalid state for this task, so Wall-e skipped it")
            raise

    def _schedule_profile(self, scenario, host):
        kwargs = drop_none(
            {
                "issuer": authorization.ISSUER_WALLE,
                "task_type": TaskType.AUTOMATED_ACTION,
                "host": host,
                "from_current_task": False,
                "ignore_maintenance": True,
                "reason": scenario.ticket_key,
                "ignore_cms": self.get_ignore_cms_value(scenario, host, self.ignore_cms),
                "profile_mode": self.profile_mode,
                "force_update_network_location": True,
                "monitor_on_completion": self.monitor_on_completion,
            }
        )
        with HostInterruptableLock(host.uuid, host.tier):
            schedule_profile(**kwargs)
        self.log_info(scenario, host, msg="profile-host-stage scheduled successfully")
        return Marker.in_progress(message="Launched a task that runs 'profile' on the host")


@StageRegistry.register(StageName.SwitchVlansHostStage)
class SwitchVlansHostStage(HostStage):
    """Launches a task that switches VLANs of the host"""

    def __init__(self, network_target=NetworkTarget.PROJECT, skip_on_invalid_state=False, **kwargs):
        super().__init__(network_target=network_target, skip_on_invalid_state=skip_on_invalid_state, **kwargs)
        self.network_target = network_target
        self.skip_on_invalid_state = skip_on_invalid_state

    def run(self, stage_info, scenario, host, *args, **kwargs):
        self.log_info(scenario, host, msg="start running")
        if host.task:
            self.log_info(scenario, host, msg="wait for task: {}".format(host.task.status))
            return Marker.in_progress(message="Host has active task, scenario waits until it finishes")

        if check_last_operation_from_oplog(host, Operation.SWITCH_VLANS, scenario.scenario_id):
            self.log_info(scenario, host, msg="stage completed successfully")
            return Marker.success(message="Switched VLANs of the host")

        try:
            return self._schedule_switch_vlans(scenario, host)
        except InvalidHostStateError:
            if self.skip_on_invalid_state:
                return Marker.success(message="Host has invalid state for this task, so Wall-e skipped it")
            raise

    def _schedule_switch_vlans(self, scenario, host):
        network_update_args = host.deduce_profile_configuration(profile_mode=ProfileMode.SWP_UP)
        task_args = VlanSwitchTaskArgs(
            issuer=authorization.ISSUER_WALLE,
            task_type=TaskType.AUTOMATED_ACTION,
            project=host.project,
            host_inv=host.inv,
            host_name=host.name,
            host_uuid=host.uuid,
            scenario_id=host.scenario_id,
            network_target=self.network_target,
            ignore_maintenance=True,
            reason=scenario.ticket_key,
            network_update_args=network_update_args,
        )
        with HostInterruptableLock(host.uuid, host.tier):
            schedule_task_from_scenario(host, task_args, get_vlan_switch_task_stages)
        self.log_info(scenario, host, msg="switch-vlans scheduled successfully")
        return Marker.in_progress(message="Launched a task that switches VLANs of the host")


@StageRegistry.register(StageName.RedeployHostStage)
class RedeployHostStage(HostStage):
    """Launches a task that runs 'redeploy' on the host"""

    def __init__(self, ignore_cms=False, **kwargs):
        super().__init__(ignore_cms=ignore_cms, **kwargs)
        self.ignore_cms = ignore_cms

    def run(self, stage_info, scenario, host, *args, **kwargs):
        self.log_info(scenario, host, msg="start running")

        if host.task:
            self.log_info(scenario, host, msg="wait for task: {}".format(host.task.status))
            return Marker.in_progress(message="Host has active task, scenario waits until it finishes")

        if check_last_operation_from_oplog(host, Operation.REDEPLOY, scenario.scenario_id):
            self.log_info(scenario, host, msg="stage completed successfully")
            return Marker.success(message="'Redeploy' task finished")

        return self._schedule_redeploy(scenario, host)

    def _schedule_redeploy(self, scenario, host):
        kwargs = drop_none(
            {
                "issuer": authorization.ISSUER_WALLE,
                "task_type": TaskType.AUTOMATED_ACTION,
                "host": host,
                "from_current_task": False,
                "ignore_maintenance": True,
                "reason": scenario.ticket_key,
                "ignore_cms": self.get_ignore_cms_value(scenario, host, self.ignore_cms),
                "with_auto_healing": True,
            }
        )
        with HostInterruptableLock(host.uuid, host.tier):
            schedule_redeploy(**kwargs)
        self.log_info(scenario, host, msg="redeploy scheduled successfully")
        return Marker.in_progress(message="Launched a task that runs 'redeploy' on the host")


@StageRegistry.register(StageName.OptionalRedeployHostStage)
class OptionalRedeployHostStage(RedeployHostStage):
    """Launches a task that runs 'redeploy' on the host, but activated by maintenance plot settings"""

    def __init__(self, skip_on_invalid_state=False, **kwargs):
        super().__init__(skip_on_invalid_state=skip_on_invalid_state, **kwargs)
        self.skip_on_invalid_state = skip_on_invalid_state

    def run(self, stage_info, scenario, host, *args, **kwargs):
        self.log_info(scenario, host, msg="start running")

        project = host.get_project()
        maintenance_plot_model = project.get_maintenance_plot()

        if not maintenance_plot_model:
            return super().run(stage_info, scenario, host, *args, **kwargs)

        try:
            scenario_settings_obj = maintenance_plot_model.get_scenario_settings(scenario.scenario_type)

            if scenario_settings_obj.settings.enable_redeploy_after_change_of_mac_address:
                self.log_info(scenario, host, msg="stage run parent 'run' method")
                return super().run(stage_info, scenario, host, *args, **kwargs)
            else:
                self.log_info(scenario, host, msg="stage completed successfully")
                return Marker.success(
                    message="'Redeploy' task disabled by maintenance plot "
                    "with id {}".format(maintenance_plot_model.id)
                )
        except AttributeError as e:
            msg = (
                "You tried to call 'enable_redeploy_after_change_of_mac_address' field on maintenance plot settings "
                "for host '{}' in scenario '{}' in stage '{}', but this "
                "field doesn't exists: {}".format(
                    host.human_id(), scenario.scenario_id, StageName.OptionalRedeployHostStage, str(e)
                )
            )
            raise RuntimeError(msg)
        except InvalidHostStateError:
            if self.skip_on_invalid_state:
                return Marker.success(message="Host has invalid state for this task, so Wall-e skipped it")
            raise


@StageRegistry.register(StageName.RebootHostStage)
class RebootHostStage(HostStage):
    """Launches a task that reboots the host"""

    def __init__(self, ignore_cms=False, **kwargs):
        super().__init__(ignore_cms=ignore_cms, **kwargs)
        self.ignore_cms = ignore_cms

    def run(self, stage_info, scenario, host, *args, **kwargs):
        self.log_info(scenario, host, msg="start running")

        if host.task:
            self.log_info(scenario, host, msg="wait for task: {}".format(host.task.status))
            return Marker.in_progress(message="Host has active task, scenario waits until it finishes")

        if check_last_operation_from_oplog(host, Operation.REBOOT, scenario.scenario_id):
            self.log_info(scenario, host, msg="stage completed successfully")
            return Marker.success(message="Host was rebooted")

        return self._schedule_reboot(scenario, host)

    def _schedule_reboot(self, scenario, host):
        ssh = walle.constants.SshOperation.FORBID
        task_type = TaskType.AUTOMATED_ACTION
        with_auto_healing = True
        check_post_code, check_post_code_reason = check_post_code_allowed(host, task_type, with_auto_healing)
        checks_to_monitor = sorted(set(CheckGroup.NETWORK_AVAILABILITY) | {CheckType.W_META})
        failure = None
        reboot_task_args = RebootTaskArgs(
            issuer=authorization.ISSUER_WALLE,
            task_type=task_type,
            project=host.project,
            host_inv=host.inv,
            host_name=host.name,
            host_uuid=host.uuid,
            disable_admin_requests=False,
            monitor_on_completion=True,
            scenario_id=host.scenario_id,
            ssh=ssh,
            ignore_cms=self.get_ignore_cms_value(scenario, host, self.ignore_cms),
            with_auto_healing=with_auto_healing,
            reason=scenario.ticket_key,
            check_post_code=check_post_code,
            check_post_code_reason=check_post_code_reason,
            ignore_maintenance=True,
            from_current_task=False,
            checks_to_monitor=checks_to_monitor,
            failure=failure,
            check_names=None,
            operation_restrictions=None,
            without_ipmi=False,
        )
        with HostInterruptableLock(host.uuid, host.tier):
            schedule_task_from_scenario(host, reboot_task_args, get_reboot_task_stages)
        self.log_info(scenario, host, msg="reboot scheduled successfully")
        return Marker.in_progress(message="Launched a task that reboots the host")


@StageRegistry.register(StageName.CheckDnsHostStage)
class CheckDnsHostStage(HostStage):
    """Launches a task that checks host's DNS"""

    def __init__(self, monitor_on_completion=True, skip_on_invalid_state=False, **kwargs):
        super().__init__(
            monitor_on_completion=monitor_on_completion, skip_on_invalid_state=skip_on_invalid_state, **kwargs
        )
        self.monitor_on_completion = monitor_on_completion
        self.skip_on_invalid_state = skip_on_invalid_state

    def run(self, stage_info, scenario, host, *args, **kwargs):
        self.log_info(scenario, host, msg="start running")

        if host.task:
            self.log_info(scenario, host, msg="wait for task: {}".format(host.task.status))
            return Marker.in_progress(message="Host has active task, scenario waits until it finishes")

        if check_last_operation_from_oplog(host, Operation.CHECK_DNS, scenario.scenario_id):
            self.log_info(scenario, host, msg="stage completed successfully")
            return Marker.success(message="Host's DNS checked")

        try:
            return self._schedule_check_dns(scenario, host)
        except InvalidHostStateError:
            if self.skip_on_invalid_state:
                return Marker.success(message="Host has invalid state for this task, so Wall-e skipped it")
            raise

    def _schedule_check_dns(self, scenario, host):
        kwargs = drop_none(
            {
                "issuer": authorization.ISSUER_WALLE,
                "task_type": TaskType.AUTOMATED_ACTION,
                "host": host,
                "ignore_maintenance": True,
                "reason": scenario.ticket_key,
                "with_auto_healing": True,
                "monitor_on_completion": self.monitor_on_completion,
            }
        )
        with HostInterruptableLock(host.uuid, host.tier):
            schedule_dns_check(**kwargs)
        self.log_info(scenario, host, msg="check dns scheduled successfully")
        return Marker.in_progress(message="Launched a task that checks host's DNS")


@StageRegistry.register(StageName.WaitEineProfileHostStage)
class WaitEineProfileHostStage(HostStage):
    """Waits until Eine 'profile' operation on host ends"""

    def run(self, stage_info, scenario, host, *args, **kwargs):
        self.log_info(scenario, host, msg="start running")

        if host.task:
            self.log_info(scenario, host, msg="wait for task: {}".format(host.task.status))
            return Marker.in_progress(message="Host has active task, scenario waits until it finishes")

        client = eine.get_client(eine.get_eine_provider(host.get_eine_box()))
        eine_host = client.get_host_status(host.inv, profile=True)

        if (
            eine_host.has_profile()
            and eine_host.in_process()
            and eine_host.profile() != walle.constants.EINE_NOP_PROFILE
        ):
            self.log_info(scenario, host, msg="wait for ITDC profile: {}".format(eine_host.profile()))
            return Marker.in_progress(message="Waits until Eine profiles the host")

        return Marker.success(message="Eine profiled the host")


@StageRegistry.register(StageName.ReleaseHostStage)
class ReleaseHostStage(HostStage):
    """Launches a task that releases the host"""

    def __init__(self, ignore_cms=False):
        super().__init__(ignore_cms=ignore_cms)
        self.ignore_cms = ignore_cms

    def _schedule_release_host(self, scenario, host):
        task_args = HostReleaseArgs(
            issuer=authorization.ISSUER_WALLE,
            task_type=TaskType.AUTOMATED_ACTION,
            project=host.project,
            host_inv=host.inv,
            host_name=host.name,
            host_uuid=host.uuid,
            scenario_id=host.scenario_id,
            ignore_maintenance=True,
            ignore_cms=self.get_ignore_cms_value(scenario, host, self.ignore_cms),
            disable_admin_requests=False,
            reason=scenario.ticket_key,
            current_project_id=host.project,
            host_state=host.state,
            monitor_on_completion=False,
            force_new_cms_task=True,
        )

        with HostInterruptableLock(host.uuid, host.tier):
            schedule_task_from_scenario(host, task_args, get_release_host_stages)
        self.log_info(scenario, host, msg="release-host scheduled successfully")
        return Marker.in_progress(message="Launched a task that releases the host")

    def run(self, stage_info, scenario, host, *args, **kwargs):
        self.log_info(scenario, host, msg="start running")

        if host.task:
            self.log_info(scenario, host, msg="wait for task: {}".format(host.task.status))
            return Marker.in_progress(message="Host has active task, scenario waits until it finishes")

        if host.state != HostState.FREE:
            return self._schedule_release_host(scenario, host)

        self.log_info(scenario, host, msg="stage completed successfully")
        return Marker.success(message="Host released")


@StageRegistry.register(StageName.SetABCServiceIdStage)
class SetABCServiceIdStage(HostStage):
    """Sets new host's ABC service in Bot"""

    TARGET_BOT_PROJECT_ID = "target_bot_project_id"
    TRIED_TO_CHANGE_SERVICE_ID = "tried_to_change_service_id"

    def __init__(self, abc_service_id=None):
        super().__init__(abc_service_id=abc_service_id)
        self.abc_service_id = abc_service_id

    def _got_bot_project_id(self, stage_info):
        return stage_info.get_data(self.TARGET_BOT_PROJECT_ID, default=None)

    def _convert_abc_service_id_to_bot_project_id(self, stage_info):
        target_bot_project_id = bot.get_bot_project_id_by_planner_id(self.abc_service_id)
        stage_info.set_data(self.TARGET_BOT_PROJECT_ID, target_bot_project_id)

    def _tried_to_change_service_id(self, stage_info):
        return stage_info.get_data(self.TRIED_TO_CHANGE_SERVICE_ID, default=False)

    def _assign_new_service_id(self, stage_info, host):
        bot_project_id = stage_info.get_data(self.TARGET_BOT_PROJECT_ID)
        if not bot_project_id:
            raise LogicalError

        bot.assign_project_id(host.inv, bot_project_id)
        stage_info.set_data(self.TRIED_TO_CHANGE_SERVICE_ID, True)

    def _is_service_id_appointed(self, stage_info, host):
        host_info = bot.get_host_info(host.inv)
        return host_info.get("bot_project_id") == stage_info.get_data(self.TARGET_BOT_PROJECT_ID)

    def run(self, stage_info, scenario, host, *args, **kwargs):
        self.log_info(scenario, host, msg="start running")

        if host.task:
            self.log_info(scenario, host, msg="wait for task: {}".format(host.task.status))
            return Marker.in_progress(message="Host has active task, scenario waits until it finishes")

        if not self._got_bot_project_id(stage_info):
            self._convert_abc_service_id_to_bot_project_id(stage_info)

        if not self._tried_to_change_service_id(stage_info):
            self._assign_new_service_id(stage_info, host)
            return Marker.in_progress(message="Setting new host's ABC service in Bot")

        if self._is_service_id_appointed(stage_info, host):
            self.log_info(scenario, host, msg="stage completed successfully")
            return Marker.success(message="New host's ABC service in Bot was set")

        return Marker.in_progress(message="Waiting for new host's ABC service to appear in Bot")


def collect_passed_host_health_checks(host: Host):
    result = []
    if host.health:
        project = Project.objects.only("manually_disabled_checks").get(id=host.project)
        for check_type, check_status in host.health.check_statuses.items():
            if check_status == CheckStatus.PASSED:
                if not project.manually_disabled_checks or check_type not in project.manually_disabled_checks:
                    result.append(check_type)
    return result


@StageRegistry.register(StageName.CollectHealthChecksStage)
class CollectHealthChecksStage(HostStage):
    """Collects host's health checks that are in 'passed' state"""

    def run(self, stage_info, scenario, host, *args, **kwargs):
        self.log_info(scenario, host, msg="start running")

        check_names = collect_passed_host_health_checks(host)

        self.log_info(scenario, host, msg="collected check names: {}".format(", ".join(check_names)))

        host_key = host.uuid if scenario.uses_uuid_keys else host.inv
        scenario.set_host_info_checks(host_key, check_names)

        return Marker.success(message="Host's checks collected")


@dataclasses.dataclass
class StartrekCommentInfo:
    startrek_ticket_key: str
    comment_id: str


@StageRegistry.register(StageName.CheckAndReportAboutDeadlinesStage)
class CheckAndReportAboutDeadlinesStage(ParentStageHandler, HostGroupStage):
    """Adds comments to ticket if 'set-to-maintenance' task takes too long."""

    _startrek_comment_id_for_comment_in_start_time = "startrek_comment_id_for_comment_in_start_time"
    _startrek_comment_id_for_comment_in_X_hours = "startrek_comment_id_for_comment_in_X_hours"

    def run(self, stage_info: StageInfo, scenario: Scenario, host_group_id: int):
        self.log_info(scenario, host_group_id, msg="start running")

        fqdns_of_hosts_not_in_maintenance = self._collect_fqdns_of_hosts_not_in_passable_states(host_group_id, scenario)

        data_storage = get_data_storage(scenario)
        scenario_parameters = data_storage.read_scenario_parameters()
        maintenance_plot = self._get_host_group_maintenance_plot(host_group_id, data_storage)
        scenario_settings = maintenance_plot.get_scenario_settings(scenario.scenario_type)

        # TODO: generate stages
        try:
            if scenario_settings.enable_manual_approval_after_hosts_power_off:
                return Marker.success(message="Manual approve enabled, user will set maintenance manually.")
        except AttributeError:
            log.info("scenario %s, group_id %s, manual approval don't enabled", scenario.scenario_id, host_group_id)

        if fqdns_of_hosts_not_in_maintenance and scenario_parameters.maintenance_start_time is not None:
            try:
                if scenario_settings.get_approvers_to_ticket_if_hosts_not_in_maintenance_by_x_seconds:
                    self._write_comment_if_time_out(
                        stage_info,
                        data_storage,
                        maintenance_plot,
                        self._startrek_comment_id_for_comment_in_X_hours,
                        scenario.ticket_key,
                        fqdns_of_hosts_not_in_maintenance,
                        host_group_id,
                        scenario_parameters.maintenance_start_time
                        - scenario_settings.get_approvers_to_ticket_if_hosts_not_in_maintenance_by_x_seconds,
                    )

                if scenario_settings.get_approvers_to_ticket_if_hosts_not_in_maintenance_by_start_time:
                    self._write_comment_if_time_out(
                        stage_info,
                        data_storage,
                        maintenance_plot,
                        self._startrek_comment_id_for_comment_in_start_time,
                        scenario.ticket_key,
                        fqdns_of_hosts_not_in_maintenance,
                        host_group_id,
                        scenario_parameters.maintenance_start_time,
                    )

            except AttributeError as e:
                msg = (
                    "You tried to call 'get_approvers_to_ticket_if_hosts_not_in_maintenance_by_*' fields on maintenance plot settings "
                    "for host_group_id '{}' in scenario '{}' in stage '{}', but this "
                    "field doesn't exists: {}".format(
                        host_group_id, scenario.scenario_id, StageName.CheckAndReportAboutDeadlinesStage, str(e)
                    )
                )
                raise RuntimeError(msg)

        return self.execute_current_stage(stage_info, scenario, host_group_id)

    def _collect_fqdns_of_hosts_not_in_passable_states(self, host_group_id: int, scenario: Scenario) -> list[str]:
        invs = [host_info.inv for host_info in scenario.hosts.values() if host_info.group == host_group_id]
        fqdns = [
            host.name
            for host in Host.objects(inv__in=invs, state__nin=[HostState.MAINTENANCE, HostState.FREE]).only("name")
        ]
        return fqdns

    def _write_comment_if_time_out(
        self,
        stage_info: StageInfo,
        data_storage: BaseScenarioDataStorage,
        maintenance_plot: MaintenancePlot,
        field_name: str,
        scenario_ticket_key: str,
        fqdns_of_hosts_not_in_maintenance: list[str],
        host_group_id: int,
        wait_until_ts: int,
    ):
        if self._read_comment_info(stage_info, field_name):
            return

        if timestamp() >= wait_until_ts:
            comment_id = self._add_comment_to_ticket(
                scenario_ticket_key,
                data_storage,
                host_group_id,
                fqdns_of_hosts_not_in_maintenance,
                tuple(maintenance_plot.get_approvers()),
            )
            self._write_comment_info(
                stage_info,
                field_name,
                StartrekCommentInfo(startrek_ticket_key=scenario_ticket_key, comment_id=comment_id),
            )

    @staticmethod
    def _read_comment_info(stage_info: StageInfo, field_name: str) -> tp.Optional[StartrekCommentInfo]:
        startrek_info_dict = stage_info.get_data(field_name, None)
        if startrek_info_dict:
            return StartrekCommentInfo(**startrek_info_dict)

    @staticmethod
    def _write_comment_info(stage_info: StageInfo, field_name: str, startrek_comment_info: StartrekCommentInfo):
        stage_info.set_data(field_name, dataclasses.asdict(startrek_comment_info))

    def _add_comment_to_ticket(
        self,
        scenario_ticket_key: ApproveClient.TStartrekTicketKey,
        data_storage: BaseScenarioDataStorage,
        host_group_id: int,
        fqdns_of_hosts: list[str],
        summonees: tuple[str],
    ) -> ApproveClient.TStartrekCommentId:
        host_group_name = self._get_host_group_name(data_storage, host_group_id)
        approve_client = ApproveClient(startrek_client=startrek.get_client())
        text = config.get_value("scenario.stages.check_and_report_about_deadlines_stage.template").format(
            host_group_name, ",".join(fqdns_of_hosts)
        )
        request = CreateStartrekCommentRequest(issue_id=scenario_ticket_key, summonees=summonees, text=text)
        return approve_client.create_startrek_comment(request)

    @staticmethod
    def _get_host_group_maintenance_plot(host_group_id: int, data_storage: BaseScenarioDataStorage) -> MaintenancePlot:
        for host_group_source in data_storage.read_host_groups_sources():
            if host_group_source.group_id == host_group_id:
                return host_group_source.source.get_maintenance_plot()
        raise WalleError("Can not find host group source with ID '%d' in scenario's data storage")

    def _get_host_group_name(self, data_storage: BaseScenarioDataStorage, host_group_id: int) -> str:
        # ABC service name as a host group name is fine for now.
        return self._get_host_group_maintenance_plot(
            host_group_id,
            data_storage,
        ).meta_info.abc_service_slug


@StageRegistry.register(StageName.NoopHostStage)
class NoopHostStage(HostStage):
    """Noop stage for host"""

    def run(self, stage_info, scenario, host, *args, **kwargs):
        _log_stage(stage_info, scenario, host)
        return Marker.in_progress(message="{}".format(timestamp()))


@StageRegistry.register(StageName.NoopHostStageSuccess)
class NoopHostStageSuccess(HostStage):
    """Noop stage for host"""

    def run(self, stage_info, scenario, host, *args, **kwargs):
        _log_stage(stage_info, scenario, host)
        return Marker.success(message="{}".format(timestamp()))


@StageRegistry.register(StageName.CheckAndReportAboutHostRestrictionsStage)
class CheckAndReportAboutHostRestrictionsStage(HostGroupStage):
    """Adds comments to ticket if any hosts in group have restrictions."""

    def __init__(self, restrictions_to_check: tp.Optional[list[str]] = None):
        super().__init__(restrictions_to_check=restrictions_to_check)
        self.restrictions_to_check = restrictions_to_check or []

    _startrek_comment_id = "startrek_comment_id"

    def run(self, stage_info: StageInfo, scenario: Scenario, host_group_id: int):
        self.log_info(scenario, host_group_id, msg="start running")

        data_storage = get_data_storage(scenario)
        maintenance_plot = get_host_group_maintenance_plot(host_group_id, data_storage)

        restrictions_to_check = self._get_restrictions_to_check_by_maintenance_plot_settings(
            maintenance_plot, scenario.scenario_type
        )
        fqdns_of_hosts_with_restrictions = self._collect_fqdns_of_hosts_with_restrictions(
            host_group_id, scenario, restrictions_to_check
        )

        if fqdns_of_hosts_with_restrictions:

            self._write_comment(
                stage_info,
                data_storage,
                maintenance_plot,
                self._startrek_comment_id,
                scenario.ticket_key,
                fqdns_of_hosts_with_restrictions,
                host_group_id,
            )
            return Marker.success(message="Reported about hosts with restrictions")

        return Marker.success(message="Don't find any hosts with restrictions")

    def _get_restrictions_to_check_by_maintenance_plot_settings(
        self, maintenance_plot: MaintenancePlot, scenario_type: str
    ) -> list[str]:
        settings = maintenance_plot.get_scenario_settings(scenario_type)
        disabled_restrictions = restrictions.expand_restrictions(settings.get_restrictions_disabled_by_settings())
        return restrictions.expand_restrictions(set(self.restrictions_to_check) - set(disabled_restrictions))

    def _collect_fqdns_of_hosts_with_restrictions(
        self, host_group_id: int, scenario: Scenario, restrictions_to_check: list[str]
    ) -> list[str]:
        invs = [host_info.inv for host_info in scenario.hosts.values() if host_info.group == host_group_id]
        fqdns = [
            host.name
            for host in Host.objects(inv__in=invs, type=HostType.SERVER, restrictions__in=restrictions_to_check).only(
                "name"
            )
        ]
        return fqdns

    def _write_comment(
        self,
        stage_info: StageInfo,
        data_storage: BaseScenarioDataStorage,
        maintenance_plot: MaintenancePlot,
        field_name: str,
        scenario_ticket_key: str,
        fqdns_of_hosts: list[str],
        host_group_id: int,
    ) -> str:
        if comment_info := self._read_comment_info(stage_info, field_name):
            return comment_info.comment_id

        comment_id = self._add_comment_to_ticket(
            scenario_ticket_key, data_storage, host_group_id, fqdns_of_hosts, tuple(maintenance_plot.get_approvers())
        )
        self._write_comment_info(
            stage_info, field_name, StartrekCommentInfo(startrek_ticket_key=scenario_ticket_key, comment_id=comment_id)
        )
        return comment_id

    @staticmethod
    def _read_comment_info(stage_info: StageInfo, field_name: str) -> tp.Optional[StartrekCommentInfo]:
        startrek_info_dict = stage_info.get_data(field_name, None)
        if startrek_info_dict:
            return StartrekCommentInfo(**startrek_info_dict)

    @staticmethod
    def _write_comment_info(stage_info: StageInfo, field_name: str, startrek_comment_info: StartrekCommentInfo):
        stage_info.set_data(field_name, dataclasses.asdict(startrek_comment_info))

    def _add_comment_to_ticket(
        self,
        scenario_ticket_key: ApproveClient.TStartrekTicketKey,
        data_storage: BaseScenarioDataStorage,
        host_group_id: int,
        fqdns_of_hosts: list[str],
        summonees: tuple[str],
    ) -> ApproveClient.TStartrekCommentId:
        host_group_name = self._get_host_group_name(data_storage, host_group_id)
        approve_client = ApproveClient(startrek_client=startrek.get_client())
        text = JinjaTemplateRenderer().render_template(
            TemplatePath.STARTREK_CHECK_AND_REPORT_ABOUT_RESTRICTIONS,
            invs=",".join(fqdns_of_hosts),
            host_group_name=host_group_name,
        )
        request = CreateStartrekCommentRequest(issue_id=scenario_ticket_key, summonees=summonees, text=text)
        return approve_client.create_startrek_comment(request)

    def _get_host_group_name(self, data_storage: BaseScenarioDataStorage, host_group_id: int) -> str:
        # ABC service name as a host group name is fine for now.
        return get_host_group_maintenance_plot(
            host_group_id,
            data_storage,
        ).meta_info.abc_service_slug


@StageRegistry.register(StageName.CheckAndSkipIfSecondOkEnabledStage)
class CheckAndSkipIfSecondOkEnabledStage(ParentStageHandler, HostGroupStage):
    """Skip all children if second OK enabled."""

    def run(self, stage_info: StageInfo, scenario: Scenario, host_group_id: int):
        self.log_info(scenario, host_group_id, msg="start running")

        data_storage = get_data_storage(scenario)
        maintenance_plot = self._get_host_group_maintenance_plot(host_group_id, data_storage)
        scenario_settings = maintenance_plot.get_scenario_settings(scenario.scenario_type)

        # TODO: generate stages
        try:
            if scenario_settings.enable_manual_approval_after_hosts_power_off:
                return Marker.success(message="Manual approve enabled, user will return hosts manually.")
        except AttributeError:
            log.info("scenario %s, group_id %s, manual approval don't enabled", scenario.scenario_id, host_group_id)

        return self.execute_current_stage(stage_info, scenario, host_group_id)

    @staticmethod
    def _get_host_group_maintenance_plot(host_group_id: int, data_storage: BaseScenarioDataStorage) -> MaintenancePlot:
        for host_group_source in data_storage.read_host_groups_sources():
            if host_group_source.group_id == host_group_id:
                return host_group_source.source.get_maintenance_plot()
        raise WalleError("Can not find host group source with ID '%d' in scenario's data storage")
