import re

from mongoengine import DoesNotExist

from sepelib.core import config
from sepelib.yandex.startrek import StartrekRequestError, StartrekConnectionError
from walle.clients import startrek
from walle.errors import RequestValidationError
from walle.projects import Project
from walle.scenario.constants import ScenarioFsmStatus, RESERVED_LABELS
from walle.scenario.errors import ScenarioValidationError
from walle.scenario.scenario import Scenario
from walle.scenario.script import noc_soft_maintenance_script


def validate_labels_fn(labels):
    errors = []
    for key, value in labels.items():
        if key in RESERVED_LABELS:
            errors.append("Label name {} is reserved".format(key))

        if not (
            (isinstance(key, str) or (isinstance(key, int) and not isinstance(key, bool)))
            and (isinstance(value, str) or (isinstance(value, int) and not isinstance(value, bool)))
        ):
            errors.append("Key or value isn't [String]/[Integer], wrong pair - {}:{}".format(key, value))

    if errors:
        errors = "; ".join(errors)
        raise ScenarioValidationError("Labels validation errors: {}".format(errors))


def validate_ticket_key_fn(ticket_key):
    if not ticket_key:
        raise ScenarioValidationError("Ticket key validation error: no ticket key specified")

    if not re.match(r"[a-zA-Z]+-\d+", ticket_key):
        raise ScenarioValidationError("Ticket key validation error: invalid ticket format")

    st_client = startrek.get_client()

    try:
        st_client.get_issue(ticket_key)

    except StartrekConnectionError:
        raise ScenarioValidationError("Ticket key validation error: Connection with Startrek failed")

    except StartrekRequestError as e:
        if e.response.status_code == 404:
            raise ScenarioValidationError(
                "Ticket key validation error: Startrek ticket {} does not exist".format(ticket_key)
            )
        else:
            raise ScenarioValidationError(
                "Ticket key validation error: Failed to get status for startrek ticket {}: {}".format(
                    ticket_key, str(e)
                )
            )


def validate_switch_fn(switch):
    # TODO separate NOC Scenarios from ITDC, remove duplicated code
    # TODO Allow works on a switch scheduled on multiple days
    switch_scenarios_count = Scenario.objects.filter(
        scenario_type=noc_soft_maintenance_script.name,
        status__in=[ScenarioFsmStatus.CREATED, ScenarioFsmStatus.STARTED],
    ).count()
    if switch_scenarios_count >= config.get_value("scenario.max_created_maintenance_scenarios"):
        # NB If you want to change this message do not forget to notify NOC devs
        # https://st.yandex-team.ru/NOCDEV-2206#5ed0fa25ae1518122dc92114
        raise ScenarioValidationError("Too many switches are being processed right now. Please try again later.")


def validate_project_id_fn(project_id):
    try:
        Project.objects.get(id=project_id)
    except DoesNotExist:
        raise ScenarioValidationError("Project with given id does not exists.")


def validate_scenario_params(
    params,
    validate_ticket_key: bool = True,
    validate_labels: bool = True,
    validate_switch: bool = True,
    validate_project_id: bool = True,
):
    validators = {}
    if validate_ticket_key:
        validators["ticket_key"] = validate_ticket_key_fn
    if validate_labels:
        validators["labels"] = validate_labels_fn
    if validate_switch:
        validators["switch"] = validate_switch_fn
    if validate_project_id:
        validators["project_id"] = validate_project_id_fn

    errors = []
    # TODO(rocco66): refactor this place after specific soft noc handers will be removed
    for key, func in validators.items():
        if key in params:
            value = params[key]
        else:
            if script_args_value := params.get("script_args", {}).get(key):
                value = script_args_value
            else:
                continue

        try:
            func(value)
        except ScenarioValidationError as e:
            errors.append(e.error)

    if errors:
        errors = "; ".join(errors)
        raise RequestValidationError(errors)
