import logging
import dataclasses
import typing as tp

import mongoengine
import re

from walle import audit_log
from walle import constants
from walle import host_operations
from walle import projects
from walle import restrictions
from walle.authorization import ISSUER_ROBOT_WALLE, ISSUER_WALLE
from walle.clients import abc
from walle.errors import ResourceAlreadyExistsError, HostNotFoundError
from walle.expert.types import CheckType
from walle.hosts import Host, HostState, HostStatus, HostType
from walle.locks import ProjectInterruptableLock, MaintenancePlotInterruptableLock, HostInterruptableLock
from walle.maintenance_plot.crud import get_maintenance_plot
from walle.maintenance_plot.exceptions import MaintenancePlotNotFoundError
from walle.maintenance_plot.model import MaintenancePlotModel
from walle.preorders import Preorder
from walle.projects import Project
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.mongo import MongoDocument, SECONDARY_LOCAL_DC_PREFERRED

log = logging.getLogger(__name__)

SHADOW_PREFIX = "shadow-{}"


@dataclasses.dataclass
class BotHostInfo:
    inv: int
    name: tp.Optional[str]
    oebs_status: str
    planner_id: tp.Optional[int]


@dataclasses.dataclass
class ShadowProjectInfo:
    id: str
    name: str
    bot_project_id: int
    reason: str

    type: str = constants.HostType.SHADOW_SERVER
    tags: list[str] = dataclasses.field(default_factory=lambda: [constants.HostType.SHADOW_SERVER])
    default_host_restrictions: list[str] = dataclasses.field(
        default_factory=lambda: [restrictions.AUTOMATION, restrictions.REBOOT]
    )
    manually_disabled_checks: list[str] = dataclasses.field(
        default_factory=lambda: [
            CheckType.SSH,
            CheckType.UNREACHABLE,
            CheckType.NETMON,
            CheckType.WALLE_RACK,
            CheckType.META,
        ]
    )


@dataclasses.dataclass
class ShadowMaintenancePlotInfo:
    id: str
    abc_service_slug: str
    name: str
    reason: str


def get_host_bot_info(raw_dict) -> BotHostInfo:
    planner_id = raw_dict.get("planner_id")
    return BotHostInfo(
        inv=raw_dict["inv"],
        oebs_status=raw_dict["oebs_status"],
        planner_id=int(planner_id) if planner_id else None,
        name=raw_dict.get("name", None),
    )


def get_or_create_shadow_project(project_info: ShadowProjectInfo) -> projects.Project:
    try:
        return projects.get_by_id(project_info.id)
    except projects.ProjectNotFoundError:
        from walle import project_builder  # imports from api don't work properly

        builder = project_builder.ProjectBuilder(ISSUER_ROBOT_WALLE + "@")
        builder._auth_enabled = False
        builder.set_fields(
            id=project_info.id,
            owners=[],
            automation_limits=projects.get_default_shadow_project_automation_limits(),
            host_limits=projects.get_default_host_limits(),
            validate_bot_project_id=True,
        )
        builder.set_name(project_info.name)
        builder.set_type(project_info.type)
        builder.set_project_tags(project_info.tags)
        builder.set_bot_project_id(project_info.bot_project_id)
        builder.set_default_host_restrictions(project_info.default_host_restrictions)
        builder.set_manually_disabled_checks(project_info.manually_disabled_checks)
        builder.set_cms_settings(dict(url=projects.DEFAULT_CMS_NAME, max_busy_hosts=0), None)
        builder.set_enable_healing_automation(False)
        builder.set_enable_dns_automation(False)

        add_project_ctx_manager = audit_log.on_add_project(
            ISSUER_ROBOT_WALLE, project_info.id, dataclasses.asdict(project_info), project_info.reason
        )

        with add_project_ctx_manager:
            project = builder.build()

            try:
                project.save(force_insert=True)
            except mongoengine.NotUniqueError:
                msg = "Project with the specified ID '{}' or name '{}' already exists.".format(
                    project_info.id, project_info.name
                )
                raise ResourceAlreadyExistsError(msg)
            return project


def get_or_create_maintenance_plot(plot_info: ShadowMaintenancePlotInfo) -> MaintenancePlotModel:
    try:
        return get_maintenance_plot(plot_info.id)
    except MaintenancePlotNotFoundError:
        plot = MaintenancePlotModel(
            id=plot_info.id,
            meta_info=dict(abc_service_slug=plot_info.abc_service_slug, name=plot_info.name),
            common_settings=dict(
                maintenance_approvers=dict(
                    logins=[],
                    abc_roles_codes=[abc.Role.PRODUCT_HEAD, abc.Role.HARDWARE_RESOURCES_MANAGER],
                    abc_role_scope_slugs=[],
                    abc_duty_schedule_slugs=[],
                ),
                common_scenario_settings=dict(
                    total_number_of_active_hosts=0,
                ),
            ),
            scenarios_settings=[
                {
                    "scenario_type": "itdc-maintenance",
                    "settings": {
                        "enable_manual_approval_after_hosts_power_off": True,
                    },
                },
                {
                    "scenario_type": "noc-hard",
                    "settings": {
                        "enable_manual_approval_after_hosts_power_off": True,
                    },
                },
            ],
            gc_enabled=True,
        )
        add_plot_ctx_manager = audit_log.on_add_maintenance_plot(
            ISSUER_ROBOT_WALLE, plot_info.id, dataclasses.asdict(plot_info), plot_info.reason
        )

        with add_plot_ctx_manager:
            try:
                plot.save(force_insert=True)
            except mongoengine.NotUniqueError:
                msg = "Maintenance plot with the specified ID '{}' or name '{}' already exists.".format(
                    plot_info.id, plot_info.name
                )
                raise ResourceAlreadyExistsError(msg)
            return plot


def get_or_create_shadow_host(host_info: BotHostInfo, project_id: str, reason: str) -> Host:
    try:
        return Host.get_by_inv(host_info.inv)
    except HostNotFoundError:
        return host_operations.add_host(
            ISSUER_WALLE,
            project_id,
            inv=host_info.inv,
            instant=True,
            state=HostState.FREE,
            status=HostStatus.READY,
            reason=reason,
        )


def cleanup_empty_projects_and_unlinked_maintenance_plots(reason):
    project_collection = MongoDocument.for_model(Project)
    projects = project_collection.find(
        query={}, fields=("id", "type", "maintenance_plot_id"), read_preference=SECONDARY_LOCAL_DC_PREFERRED
    )

    used_maintenance_plots = set()
    for project in gevent_idle_iter(projects):
        if project.type == HostType.SHADOW_SERVER:
            if (
                not Host.objects(project=project.id).count()
                and not Preorder.objects(project=project.id, processed=False).count()
            ):
                with ProjectInterruptableLock(project.id) and audit_log.on_delete_project(
                    ISSUER_ROBOT_WALLE, project.id, reason
                ):
                    Project.objects(id=project.id, type=HostType.SHADOW_SERVER).delete()
                    continue
        used_maintenance_plots.add(project.maintenance_plot_id)

    plot_collection = MongoDocument.for_model(MaintenancePlotModel)
    plots = plot_collection.find(
        query={"gc_enabled": True}, fields=("id",), read_preference=SECONDARY_LOCAL_DC_PREFERRED
    )
    for plot in plots:
        if plot.id not in used_maintenance_plots:
            with MaintenancePlotInterruptableLock(plot.id) and audit_log.on_delete_maintenance_plot(
                ISSUER_ROBOT_WALLE, plot.id, reason
            ):
                MaintenancePlotModel.objects(id=plot.id, gc_enabled=True).delete()


def cleanup_invalid_shadow_hosts(all_shadow_hosts_inv_to_name_in_bot_map: dict[int, tp.Optional[str]], reason: str):
    for host in Host.objects(type=HostType.SHADOW_SERVER, inv__nin=all_shadow_hosts_inv_to_name_in_bot_map.keys()):
        log.info("Removing host %s from project %s", host.inv, host.project)
        with HostInterruptableLock(host.uuid) and audit_log.on_delete_host(
            ISSUER_ROBOT_WALLE, host.project, host.inv, host.name, host.uuid, reason=reason
        ):
            host.delete()

    names = [name for name in all_shadow_hosts_inv_to_name_in_bot_map.values() if name]
    for host in Host.objects(type=HostType.SHADOW_SERVER, name__nin=names, name__exists=True):
        log.info("Removing host %s from project %s", host.name, host.project)
        with HostInterruptableLock(host.uuid) and audit_log.on_delete_host(
            ISSUER_ROBOT_WALLE, host.project, host.inv, host.name, host.uuid, reason=reason
        ):
            host.delete()


def get_or_create_project_with_maintenance_plot(
    planner_id: str, planner_ids_to_bot_ids: dict[int, str], reason: str
) -> Project:
    abc_service = abc.get_service_by_id(planner_id)
    abc_service_slug = abc_service["slug"]
    abc_service_id = abc_service["id"]

    plot_info = ShadowMaintenancePlotInfo(
        id=SHADOW_PREFIX.format(abc_service_id),
        abc_service_slug=abc_service_slug,
        name=SHADOW_PREFIX.format(abc_service_slug),
        reason=reason,
    )
    maintenance_plot = get_or_create_maintenance_plot(plot_info)

    bot_project_id = planner_ids_to_bot_ids[planner_id]

    regex = re.compile('[^a-z0-9-]')
    project_key = SHADOW_PREFIX.format(regex.sub('', abc_service_slug.lower()))[:32].strip("-")
    project_info = ShadowProjectInfo(
        id=project_key,
        name=project_key,
        bot_project_id=int(bot_project_id),
        reason=reason,
    )
    project = get_or_create_shadow_project(project_info)

    if project.maintenance_plot_id != maintenance_plot.id:
        with audit_log.on_update_project(
            ISSUER_ROBOT_WALLE, project.id, {"maintenance_plot_id": maintenance_plot.id}, reason
        ):
            project.modify(set__maintenance_plot_id=maintenance_plot.id)

    return project
