import sys
import traceback


def exception_context():
    exc_type, _exc_value, exc_traceback = sys.exc_info()
    last_op = traceback.extract_tb(exc_traceback)[-1]
    context = {
        "error_class": exc_type,
        "filename": last_op[0],
        "lineno": last_op[1],
        "function": last_op[2],
    }
    return context


def host_context(host, *extra_fields):
    # Seems like get_current_stage must find its new home.
    from walle.fsm_stages.common import get_current_stage

    context = {
        "hostname": host.name,
        "inv": host.inv,
    }
    if host.project is not None:
        context['project'] = host.project
    if host.task is not None:
        context["host_task_id"] = host.task.task_id
        context["host_task_type"] = host.task.type
        if host.task is not None:
            stage = get_current_stage(host, only_if_exists=True)
            if stage is not None:
                context["stage_name"] = stage.name
                context["stage_status"] = stage.status
                context["stage_description"] = stage.description
                context["stage_uid"] = stage.uid

    for field in extra_fields:
        context[field] = getattr(host, field)

    return context


def host_failure_context(host, project, decision, failure_timestamp, repair_begin_timestamp):
    return {
        "hostname": host.name,
        "inv": host.inv,
        "project": host.project,
        "tags": project.tags,
        "failure_type": decision.failures[0] if decision.failures else None,
        "failure_timestamp": failure_timestamp,
        "repair_begin_timestamp": repair_begin_timestamp,
    }


def scenario_context(scenario, stage, event_type, host=None):
    scenario_context = {
        "name": scenario.name,
        "issuer": scenario.issuer,
        "status": scenario.status,
        "ticket_key": scenario.ticket_key,
        "labels": scenario.labels,
        "script_args": scenario.script_args,
        "current_group": scenario.current_group,
    }

    stage_context = {
        "status": stage.status,
        "status_time": stage.status_time,
        "action_type": stage.action_type,
    }

    if host:
        host_context = {
            "inv": host.inv,
            "name": host.name,
            "project": host.project,
        }
    else:
        host_context = {}

    return {
        "scenario_id": scenario.scenario_id,
        "scenario_type": scenario.scenario_type,
        "event_type": event_type,
        "scenario_context": scenario_context,
        "stage_uid": stage.uid,
        "stage_name": stage.name,
        "stage_context": stage_context,
        "host_uuid": host.uuid if host else None,
        "host_context": host_context,
    }
