"""
Helpers to write logs in formats supported by logfeller and statbox:
JSON or TSKV format (tab-separated key-value pairs).
https://wiki.yandex-team.ru/logfeller/parsers/json/
https://wiki.yandex-team.ru/statbox/logrequirements/
"""

import logging
import logging.handlers
import time
from datetime import datetime

import simplejson as json
from flask import has_request_context, request

from walle.application import app

TSKV_MAX_ENTRY_LENGTH = 1000
TSKV_REPLACE_MAP = [
    ('\\', r'\\'),
    ('\t', r'\t'),
    ('\n', r'\n'),
    ('\r', r'\r'),
    ('\0', r'\0'),
]


class NoFlushRotatingFileHandler(logging.handlers.RotatingFileHandler):
    def flush(self):
        """No-op. Don't call self.stream.flush() to speed up things."""
        return


class Serializer:
    @property
    def format(self):
        raise NotImplementedError

    def serialize(self, context):
        raise NotImplementedError


class TskvSerializer(Serializer):
    format = "tskv"

    @classmethod
    def serialize(cls, context):
        pairs = ["{k}={v}".format(k=k, v=cls._value_to_str(context[k])) for k in sorted(context)]
        output = "tskv\t{}".format("\t".join(pairs))
        return output

    @classmethod
    def _value_to_str(cls, value):
        if isinstance(value, (dict, list, tuple)):
            value = json.dumps(value)
        else:
            value = cls._escape_value(value)
        return cls._truncate_str(value)

    @staticmethod
    def _truncate_str(value):
        return value[:TSKV_MAX_ENTRY_LENGTH]

    @staticmethod
    def _escape_value(value):
        text = str(value)
        for char, replacement in TSKV_REPLACE_MAP:
            text = text.replace(char, replacement)
        return text


class JsonSerializer(Serializer):
    format = "json"

    @classmethod
    def serialize(cls, context):
        return json.dumps(context)


class ContextLogger:
    def __init__(self, logger, serializer, **initial_context):
        self._logger = logger
        self._serializer = serializer
        self._context = initial_context

    def get_context(self):
        return self._context

    def get_child(self, **additional_context):
        context = dict(self.get_context(), **additional_context)
        return self.__class__(self._logger, self._serializer, **context)

    def log(self, **context):
        message = self._serializer.serialize(dict(self.get_context(), **context))
        self._logger.log(logging.INFO, message)


class StatboxLogger(ContextLogger):
    _current_datetime_and_zone = None

    def __init__(self, log_type, logger, serializer, **initial_context):
        self.log_type = log_type
        super().__init__(logger, serializer, **initial_context)

    def get_child(self, **additional_context):
        context = dict(self.get_context(), **additional_context)
        return self.__class__(self.log_type, self._logger, self._serializer, **context)

    @classmethod
    def timestamp_cached(cls):
        """Cache formatted timestamp and timezone for a second, because formatting is pretty slow
        and we hit it about a thousand times per second."""
        current_time = int(time.time())
        if cls._current_datetime_and_zone is None or cls._current_datetime_and_zone[0] != current_time:
            now = datetime.now()
            timestamp = now.strftime("%Y-%m-%d %H:%M:%S")
            cls._current_datetime_and_zone = (current_time, timestamp, time.strftime('%z'))
        return cls._current_datetime_and_zone[1:]

    def log(self, **context):
        timestamp, timezone = self.timestamp_cached()

        extra_args = {"timestamp": timestamp, "tskv_format": self.log_type, "timezone": timezone}
        context.update(extra_args)

        super().log(**context)


class WalleStatboxLogger(StatboxLogger):
    def log(self, **context):
        context.setdefault("role", app.role)
        if has_request_context():
            context.setdefault("request_id", request.request_id)

        super().log(**context)
