"""Provides all statbox loggers used by Wall-E."""

import logging
from logging.handlers import RotatingFileHandler

from sepelib.core import config, constants
from sepelib.core.exceptions import Error
from walle.application import Service, app
from walle.statbox.handlers import TskvSerializer, JsonSerializer, WalleStatboxLogger

log = logging.getLogger(__name__)

_LOGGERS = {}


def _register_logger(
    name, services, logger_class=WalleStatboxLogger, serializer_class=TskvSerializer, handler_class=RotatingFileHandler
):
    """
    :type logger_class: type[StatboxLogger]
    :type serializer_class: Type[Serializer]
    :type handler_class: Type[RotatingFileHandler]
    """

    logger_id = name.replace("-", "_")
    if logger_id in _LOGGERS:
        raise Error("Unable to register '{}' TSKV logger: it already registered.", logger_id)

    logger = logging.getLogger("tskv." + logger_id)
    logger.level = logging.DEBUG
    logger.propagate = False

    logger_config = {
        "id": logger_id,
        "logger": logger,
        "serializer_class": serializer_class,
        "handler_class": handler_class,
        "services": services,
        "initialized": False,
        "created": False,
    }

    _LOGGERS[logger_id] = logger_config

    def get_logger(**initial_context):
        tskv_logger = logger_class(
            log_type="walle-{}-log".format(name), logger=logger, serializer=serializer_class, **initial_context
        )

        if logger_config["initialized"] and not logger_config["created"]:
            logger_config["created"] = True
            log.critical("Initialized '%s' TSKV log which is not expected to be used in '%s' role.", name, app.role)

        return tskv_logger

    return get_logger


api_logger = _register_logger("api", services=[Service.API])
cms_logger = _register_logger("cms", services=[Service.API, Service.CRON, Service.FSM])
cron_logger = _register_logger("cron", services=[Service.CRON, Service.DMC])
dmc_logger = _register_logger("dmc", services=[Service.DMC, Service.FSM])
dns_logger = _register_logger("dns", services=[Service.DNS])
fsm_logger = _register_logger("fsm", services=[Service.API, Service.FSM])
requests_logger = _register_logger("requests", services=Service.ALL)
dns_operations_logger = _register_logger("fsm-dns", services=[Service.FSM])
failures_logger = _register_logger("failures", serializer_class=JsonSerializer, services=[Service.DMC])
scenarios_logger = _register_logger(
    "scenarios", serializer_class=JsonSerializer, services=[Service.FSM, Service.SCENARIO]
)


def init():
    path_format = config.get_value("tskv_log.path_format")
    max_bytes = config.get_value("tskv_log.max_size_mb") * constants.MEGABYTE
    backup_count = config.get_value("tskv_log.backup_count")

    formatter = logging.Formatter("%(message)s")

    for logger_config in _LOGGERS.values():
        path = path_format.format(name=logger_config["id"])

        # Don't create log files for logs that aren't expected to be used
        create = not set(logger_config["services"]).isdisjoint(app.services)

        handler = logger_config["handler_class"](
            filename=path, maxBytes=max_bytes, backupCount=backup_count, delay=not create
        )

        logger_config["initialized"] = True
        if create:
            logger_config["created"] = True

        handler.setFormatter(formatter)
        logger_config["logger"].addHandler(handler)
