import logging

from walle import audit_log
from walle.clients import juggler
from walle.expert import automation
from walle.expert.automation_plot import AutomationPlot
from walle.projects import Project
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.misc import iter_chunks

JUGGLER_BATCH_SIZE = 100
log = logging.getLogger(__name__)


def report_global_automation_status():
    """Send global automation status to Juggler."""

    _send_global_automation_event(
        "healing_enabled",
        automation.GLOBAL_HEALING_AUTOMATION,
        audit_log.TYPE_ENABLE_AUTOMATED_HEALING,
        audit_log.TYPE_DISABLE_AUTOMATED_HEALING,
    )

    _send_global_automation_event(
        "dns_enabled",
        automation.GLOBAL_DNS_AUTOMATION,
        audit_log.TYPE_ENABLE_DNS_AUTOMATION,
        audit_log.TYPE_DISABLE_DNS_AUTOMATION,
    )


def _send_global_automation_event(event_name, automation_type, enabled_audit_log_type, disabled_audit_log_type):
    enabled = automation_type.is_enabled()
    status = juggler.JugglerCheckStatus.OK if enabled else juggler.JugglerCheckStatus.CRIT
    message = _get_global_event_message(enabled_audit_log_type if enabled else disabled_audit_log_type)
    tags = ["wall-e.{}".format(event_name)]

    juggler.send_event(event_name, status, message, tags)


def _get_global_event_message(audit_log_type):
    log_entry = audit_log.LogEntry.objects(type=audit_log_type, project__exists=False).order_by("-time").first()
    if log_entry:
        return log_entry.reason or "No reason message provided."
    else:
        return "Have never been switched."


def report_project_automation_status():
    """Collect project automation statuses and send to juggler."""

    log.info("Sending report for projects automation status to juggler...")

    events = []
    for project in gevent_idle_iter(_fetch_projects()):
        events.append(_mk_automation_event(project, project.healing_automation, "healing_enabled", "Healing"))
        events.append(_mk_automation_event(project, project.dns_automation, "dns_enabled", "DNS"))

    errors = _send_events_to_juggler(events)
    _handle_report_errors(errors)


def _fetch_projects():
    return Project.objects.only("id", "tags", "healing_automation", "dns_automation")


def _mk_automation_event(project, automation_status, event_name, automation_name):
    project_id = project.id
    tags = ["wall-e.{}".format(event_name), "wall-e.project.{}".format(project_id)]
    tags.extend("wall-e.tag.{}".format(juggler.normalize(tag)) for tag in project.tags or [])

    return {
        "host": "wall-e.project.{}".format(project_id),
        "service": event_name,
        "status": juggler.JugglerCheckStatus.OK if automation_status.enabled else juggler.JugglerCheckStatus.CRIT,
        "description": _mk_event_message(automation_status, project_id, automation_name),
        "tags": tags,
    }


def _mk_event_message(automation_status, project_id, automation_name):
    if automation_status.status_message:
        return automation_status.status_message

    status = "enabled" if automation_status.enabled else "disabled"

    return "{label} automation is {status} in project {project_id}.".format(
        label=automation_name, status=status, project_id=project_id
    )


def _handle_report_errors(errors):
    if errors:
        log.info("Failure while sending report for projects automation status to juggler, see logs.")

        first_error = str(errors[0])
        juggler.send_event(
            "wall-e.projects-automation-report",
            juggler.JugglerCheckStatus.CRIT,
            "Failure while sending report for projects automation status to juggler: {}.".format(first_error),
        )
    else:
        message = "Report for projects automation status sent to juggler."
        juggler.send_event("wall-e.projects-automation-report", juggler.JugglerCheckStatus.OK, message)
        log.info(message)


def report_plot_automation_status():
    """Collect plot automation statuses and send to juggler."""

    log.info("Sending report for plot automation status to juggler...")

    events = []
    for plot in gevent_idle_iter(_fetch_automation_plots()):
        for check in plot.checks:
            events.append(_mk_automation_plot_event(plot, check, "healing_enabled"))

    errors = _send_events_to_juggler(events)
    _handle_plot_report_errors(errors)


def _fetch_automation_plots():
    return AutomationPlot.objects.only("id", "checks")


def _mk_automation_plot_event(plot, check, event_name):
    tags = ["wall-e.{}".format(event_name), "wall-e.automation-plot.{}".format(plot.id)]

    return {
        "host": "wall-e.automation-plot.{}.{}".format(plot.id, check.name),
        "service": event_name,
        "status": juggler.JugglerCheckStatus.OK if check.enabled else juggler.JugglerCheckStatus.CRIT,
        "description": "Healing for check {} in automation plot {} is {}".format(
            check.name, plot.id, "enabled" if check.enabled else "disabled"
        ),
        "tags": tags,
    }


def _handle_plot_report_errors(errors):
    if errors:
        log.info("Failure while sending report for automation-plot status to juggler, see logs.")

        first_error = str(errors[0])
        juggler.send_event(
            "wall-e.automation-plot-report",
            juggler.JugglerCheckStatus.CRIT,
            "Failure while sending report for projects automation status to juggler: {}.".format(first_error),
        )
    else:
        message = "Report for automation-plot status sent to juggler."
        juggler.send_event("wall-e.automation-plot-report", juggler.JugglerCheckStatus.OK, message)
        log.info(message)


def _send_events_to_juggler(events):
    errors = []

    for batch in iter_chunks(events, JUGGLER_BATCH_SIZE):
        batch_errors = juggler.send_batch(batch)
        errors.extend(batch_errors)

    return errors
