"""Host task management."""

import logging

import walle.fsm_stages.common
import walle.host_fsm.control
import walle.profile_stat
import walle.projects
import walle.stages
from sepelib.core import config
from sepelib.core.exceptions import LogicalError
from walle import audit_log, authorization, restrictions
from walle import network  # noqa
from walle._tasks.task_args import (
    RebootTaskArgs,
    EnsureHostPreorderAcquirementTaskArgs,
    SwitchToMaintenanceTaskArgs,
    ProjectSwitchingArgs,
    PrepareTaskArgs,
    SetAssignedTaskArgs,
    FqdnDeinvalidationArgs,
    HostReleaseArgs,
)
from walle._tasks.task_creator import (
    get_reboot_task_stages,
    get_bot_acquirement_task_stages,
    get_switch_project_task_stages,
    get_switch_to_maintenance_stages,
    get_prepare_stages,
    get_assign_stages,
    get_fqdn_deinvalidation_stages,
    get_release_host_stages,
)
from walle._tasks.task_provider import schedule_task, schedule_task_from_api
from walle._tasks.utils import validate_hostname
from walle.clients import deploy
from walle.clients.cms import CmsTaskAction
from walle.clients.eine import ProfileMode
from walle.cms import is_cms_equal_in_projects, is_default_cms_used_in_projects
from walle.constants import (
    PROVISIONER_LUI,
    VLAN_SCHEME_STATIC,
    DEFAULT_DNS_TTL,
    SshOperation,
    NetworkTarget,
    HostType,
    HOST_TYPES_WITH_FULL_AUTOMATION,
    VLAN_SCHEMES_WITH_VERIFYING,
)
from walle.errors import InvalidHostStateError, RequestValidationError, BadRequestError
from walle.expert.types import CheckType, CheckGroup
from walle.fsm_stages.constants import EineProfileOperation
from walle.hosts import (
    Host,
    HostState,
    HostStatus,
    HostOperationState,
    TaskType,
    MissingDefaultProfileConfigurationError,
)
from walle.models import timestamp
from walle.operations_log.constants import Operation
from walle.restrictions import strip_restrictions
from walle.stages import Stage, Stages
from walle.util import notifications
from walle.util.misc import drop_none, fix_mongo_set_kwargs, filter_dict_keys
from walle.util.tasks import (
    get_profile_stages,
    get_deploy_stages,
    new_task,
    reject_request_if_needed,
    CheckDnsTaskHelper,
    TaskHelper,
    ReportTaskHelper,
    StageBuilder,
    UserTaskHelper,
    check_state_and_get_query,
    get_power_on_stages,
    check_post_code_allowed,
    prepare_host_adding_stages,
)

log = logging.getLogger(__name__)


def schedule_add_host(
    host,
    issuer,
    deploy_configuration,
    audit_entry,
    dns,
    check,
    ignore_cms,
    disable_admin_requests,
    with_auto_healing,
    reason=None,
):
    task = UserTaskHelper(
        issuer,
        TaskType.MANUAL,
        host,
        disable_admin_requests=disable_admin_requests,
        with_auto_healing=with_auto_healing,
        ignore_maintenance=True,
    )
    task.acquire_permission(CmsTaskAction.PREPARE, comment=reason, ignore_cms=ignore_cms)
    log.info("Scheduling host %s add...", host.human_id())

    if host.type in [HostType.SERVER, HostType.VM]:
        task.assign_bot_project(host.get_project(["bot_project_id"]).bot_project_id)

    task.set_downtime()
    task.keep_downtime()

    if host.state != HostState.MAINTENANCE and host.type == HostType.SERVER:
        prepare_host_adding_stages(task, host, reason, deploy_configuration, dns, check)
        task.target_status(host.status)

    task.probe_cms()
    task.log_completed_operation(Operation.ADD)

    host.set_status(Operation.ADD.host_status, issuer, audit_entry.id, reason=reason)
    host.task = task.task(audit_entry)


def schedule_switch_to_maintenance(
    issuer,
    task_type,
    host,
    timeout_time,
    ticket_key,
    target_status=HostStatus.READY,
    power_off=False,
    disable_admin_requests=False,
    ignore_cms=False,
    reason=None,
    cms_task_action=CmsTaskAction.PROFILE,
    operation_state=HostOperationState.OPERATION,
):
    ready_to_off = host.type in HOST_TYPES_WITH_FULL_AUTOMATION
    if power_off:
        operation_restrictions = restrictions.check_restrictions(host, restrictions.REBOOT)
    else:
        operation_restrictions = []
    force_new_task = host.task is not None
    allowed_statuses = HostStatus.ALL_TASK if force_new_task else HostStatus.ALL_STEADY

    task_args = SwitchToMaintenanceTaskArgs(
        issuer=issuer,
        task_type=task_type,
        project=host.project,
        host_inv=host.inv,
        host_name=host.name,
        host_uuid=host.uuid,
        scenario_id=host.scenario_id,
        reason=reason,
        ignore_cms=ignore_cms,
        disable_admin_requests=disable_admin_requests,
        operation_state=operation_state,
        cms_action=cms_task_action,
        power_off=power_off and ready_to_off,
        timeout_status=target_status,
        ticket_key=ticket_key,
        timeout_time=timeout_time,
        operation_restrictions=operation_restrictions,
        force_new_task=force_new_task,
        allowed_statuses=allowed_statuses,
        monitor_on_completion=False,
        keep_downtime=True,
    )

    schedule_task(host, task_args, get_switch_to_maintenance_stages)


def schedule_setting_assigned_state(
    issuer,
    task_type,
    host,
    status=None,
    power_on=False,
    ignore_maintenance=False,
    disable_admin_requests=False,
    monitor_on_completion=True,
    with_auto_healing=None,
    reason=None,
):
    ready_to_on = host.type in HOST_TYPES_WITH_FULL_AUTOMATION
    task_args = SetAssignedTaskArgs(
        issuer=issuer,
        task_type=task_type,
        project=host.project,
        host_inv=host.inv,
        host_name=host.name,
        host_uuid=host.uuid,
        scenario_id=host.scenario_id,
        reason=reason,
        disable_admin_requests=disable_admin_requests,
        power_on=power_on and ready_to_on,
        monitor_on_completion=monitor_on_completion,
        ignore_maintenance=ignore_maintenance,
        with_auto_healing=with_auto_healing,
        target_status=status,
        checks_to_monitor=CheckGroup.NETWORK_AVAILABILITY,
        keep_task_id=True,
    )
    schedule_task(host, task_args, get_assign_stages)


def schedule_project_switching(
    issuer,
    task_type,
    host,
    project_id,
    host_restrictions,
    release,
    erase_disks=True,
    force=False,
    ignore_maintenance=False,
    ignore_cms=False,
    disable_admin_requests=False,
    reason=None,
    force_new_cms_task=False,
):
    try:
        target_project = walle.projects.get_by_id(project_id)
    except walle.projects.ProjectNotFoundError as e:
        raise BadRequestError(str(e))
    current_project = host.get_project()

    with_releasing = release or host.state == HostState.FREE
    _check_host_for_project_switching_support(host, current_project, target_project, with_releasing, force)

    allowed_statuses = HostStatus.ALL_STEADY[:]
    if not release and target_project.bot_project_id is None:
        allowed_statuses.append(HostStatus.INVALID)

    host_restrictions = strip_restrictions(host_restrictions, strip_to_none=True)

    if host.state != HostState.FREE:
        if release:
            force_new_cms_task = True
    else:
        ignore_cms = True

    task_args = ProjectSwitchingArgs(
        issuer=issuer,
        task_type=task_type,
        project=host.project,
        host_inv=host.inv,
        host_name=host.name,
        host_uuid=host.uuid,
        scenario_id=host.scenario_id,
        operation_restrictions=host_restrictions,
        release=release,
        force=force,
        ignore_maintenance=ignore_maintenance,
        ignore_cms=ignore_cms,
        disable_admin_requests=disable_admin_requests,
        reason=reason,
        erase_disks=erase_disks,
        current_project_id=current_project.id,
        target_project_id=target_project.id,
        allowed_statuses=allowed_statuses,
        target_project_bot_project_id=target_project.bot_project_id,
        host_state=host.state,
        monitor_on_completion=False,
        cms_task_id=host.cms_task_id,
        force_new_cms_task=force_new_cms_task,
    )
    schedule_task(host, task_args, get_switch_project_task_stages)


def schedule_host_release(
    issuer,
    task_type,
    host,
    erase_disks=True,
    ignore_maintenance=False,
    ignore_cms=False,
    disable_admin_requests=False,
    reason=None,
    force_new_cms_task=False,
):
    current_project = host.get_project()

    if host.state != HostState.FREE:
        force_new_cms_task = True
    else:
        ignore_cms = True

    task_args = HostReleaseArgs(
        issuer=issuer,
        task_type=task_type,
        project=host.project,
        host_inv=host.inv,
        host_name=host.name,
        host_uuid=host.uuid,
        scenario_id=host.scenario_id,
        ignore_maintenance=ignore_maintenance,
        ignore_cms=ignore_cms,
        disable_admin_requests=disable_admin_requests,
        reason=reason,
        erase_disks=erase_disks,
        current_project_id=current_project.id,
        force_new_cms_task=force_new_cms_task,
        host_state=host.state,
        monitor_on_completion=False,
        cms_task_id=host.cms_task_id,
    )
    schedule_task(host, task_args, get_release_host_stages)


def _check_host_for_project_switching_support(host, source_project, target_project, with_releasing, force):
    if host.state == HostState.MAINTENANCE and not with_releasing:
        if not force and not is_cms_equal_in_projects(source_project, target_project):
            raise BadRequestError(
                "Can't switch project for host on maintenance without releasing it: "
                "CMS on projects do not match. Use force to ignore this."
            )

        if force and is_default_cms_used_in_projects(source_project, target_project):
            raise BadRequestError(
                "Can't switch project for host on maintenance without releasing it: "
                "default CMS is used in one of the projects."
            )

    if with_releasing or force:
        return

    # TODO: A workaround until we decide how exactly we manage host naming and other project-specific things in
    # different projects.
    if (
        target_project.vlan_scheme not in (None, VLAN_SCHEME_STATIC)
        and target_project.vlan_scheme != source_project.vlan_scheme
    ):
        raise BadRequestError(
            "Can't switch projects for the host without releasing it: "
            "The source and target projects have different VLAN schemes."
        )

    if host.extra_vlans:
        no_access_vlans = set(host.extra_vlans) - set(target_project.owned_vlans)
        if no_access_vlans:
            raise BadRequestError(
                "Can't switch the host to '{}' project without releasing it: "
                "The host have extra VLANs that '{}' project doesn't have access to: {}.",
                target_project.id,
                target_project.id,
                ", ".join(str(vlan) for vlan in no_access_vlans),
            )


def schedule_prepare(
    issuer,
    task_type,
    host,
    profile=None,
    profile_tags=None,
    skip_profile=False,
    provisioner=None,
    config=None,
    deploy_config_policy=None,
    host_restrictions=None,
    extra_vlans=None,
    deploy_tags=None,
    deploy_network=None,
    ignore_cms=False,
    ignore_maintenance=False,
    disable_admin_requests=False,
    check=True,
    with_auto_healing=None,
    keep_name=None,
    reason=None,
    update_firmware=False,
    repair_request_severity=None,
):
    """Schedules preparing of the specified host.

    :raises InvalidProfileNameError, InvalidConfigNameError, InvalidHostStateError, RequestRejectedByCmsError
    """

    # Attention: If you change the logic here, don't forget to sync the changes with host adding function.

    if skip_profile and (profile is not None or profile_tags is not None):
        raise RequestValidationError("You must either set skip_profile or profile options - not both.")

    host_restrictions = strip_restrictions(host_restrictions, strip_to_none=True)

    checks_to_monitor = CheckGroup.NETWORK_AVAILABILITY + CheckGroup.OS_CONSISTENCY

    if keep_name:
        validate_hostname(host)

    network_update_args = host.deduce_profile_configuration(profile_mode=ProfileMode.SWP_UP)
    profile_configuration = host.deduce_profile_configuration(profile, profile_tags)
    (
        host_provisioner,
        host_config,
        host_deploy_tags,
        host_deploy_network,
        host_deploy_config_policy,
        deploy_configuration,
    ) = host.deduce_deploy_configuration(provisioner, config, deploy_tags, deploy_network, deploy_config_policy)
    bot_project_id = host.get_project(["bot_project_id"]).bot_project_id

    health_status_accuracy = None
    if deploy_configuration.provisioner == PROVISIONER_LUI:
        health_status_accuracy = deploy.COMPLETED_STATUS_TIME_ACCURACY

    update_firmware_configuration = host.deduce_profile_configuration(profile_mode=ProfileMode.FIRMWARE_UPDATE)

    task_args = PrepareTaskArgs(
        issuer=issuer,
        task_type=task_type,
        project=host.project,
        host_inv=host.inv,
        host_name=host.name,
        host_uuid=host.uuid,
        scenario_id=host.scenario_id,
        reason=reason,
        ignore_cms=ignore_cms,
        disable_admin_requests=disable_admin_requests,
        monitor_on_completion=check,
        with_auto_healing=with_auto_healing,
        profile_configuration=profile_configuration,
        checks_to_monitor=checks_to_monitor,
        operation_restrictions=host_restrictions,
        ignore_maintenance=ignore_maintenance,
        provisioner=provisioner,
        config=config,
        extra_vlans=extra_vlans,
        deploy_tags=deploy_tags,
        deploy_config_policy=deploy_config_policy,
        deploy_network=deploy_network,
        network_update_args=network_update_args,
        network_target=NetworkTarget.PROJECT,
        deploy_configuration=deploy_configuration,
        host_provisioner=host_provisioner,
        host_config=host_config,
        host_deploy_tags=host_deploy_tags,
        host_deploy_network=host_deploy_network,
        host_deploy_config_policy=host_deploy_config_policy,
        bot_project_id=bot_project_id,
        skip_profile=skip_profile,
        keep_fqdn=keep_name,
        health_status_accuracy=health_status_accuracy,
        update_firmware_configuration=update_firmware_configuration,
        firmware_update_needed=update_firmware,
        repair_request_severity=repair_request_severity,
    )
    schedule_task(host, task_args, get_prepare_stages)


def schedule_wait_for_bot_acquirement(issuer, task_type, host):
    task_args = EnsureHostPreorderAcquirementTaskArgs(
        issuer=issuer,
        task_type=task_type,
        project=host.project,
        host_inv=host.inv,
        host_name=host.name,
        host_uuid=host.uuid,
        scenario_id=host.scenario_id,
    )
    schedule_task(host, task_args, get_bot_acquirement_task_stages)


def _use_cloud_postprocess(project_id, task_type):
    enable_projects = config.get_value("projects_with_cloud_task_post_processor", [])
    return project_id in enable_projects and task_type == TaskType.AUTOMATED_HEALING


def schedule_reboot(
    issuer,
    task_type,
    host,
    from_current_task=False,
    ssh=None,
    ignore_maintenance=False,
    ignore_cms=False,
    disable_admin_requests=False,
    check=True,
    with_auto_healing=None,
    decision=None,
    reason=None,
):
    """Schedules reboot of the specified host."""
    if ssh is None:
        if host.type != HostType.SERVER:
            ssh = SshOperation.FORBID

    check_post_code, check_post_code_reason = check_post_code_allowed(host, task_type, with_auto_healing)
    decision_checks = decision.checks if decision and decision.checks else ()
    # WALLE-3450: experiment. extend to other tasks if successfull.
    # Check walle_meta to in addition to availability checks in all reboot tasks.
    # Next step: just drop CheckGroups and use either 'all' or availability + walle_meta
    checks_to_monitor = sorted(set(CheckGroup.NETWORK_AVAILABILITY) | {CheckType.W_META} | set(decision_checks))
    failure = decision.failures[0] if decision and decision.failures else None
    failure_type = decision.failure_type if decision and decision.failure_type else None
    operation_restrictions = (
        restrictions.AUTOMATED_REBOOT if issuer == authorization.ISSUER_WALLE else restrictions.REBOOT
    )

    oplog_params = None
    if decision and decision.params:
        oplog_params = decision.params

    reboot_task_args = RebootTaskArgs(
        issuer=issuer,
        task_type=task_type,
        project=host.project,
        host_inv=host.inv,
        host_name=host.name,
        host_uuid=host.uuid,
        scenario_id=host.scenario_id,
        ssh=ssh,
        ignore_cms=ignore_cms,
        disable_admin_requests=disable_admin_requests,
        monitor_on_completion=check,
        with_auto_healing=with_auto_healing,
        reason=reason,
        check_post_code=check_post_code,
        check_post_code_reason=check_post_code_reason,
        ignore_maintenance=ignore_maintenance,
        from_current_task=from_current_task,
        checks_to_monitor=checks_to_monitor,
        failure=failure,
        failure_type=failure_type,
        decision=decision,
        check_names=decision_checks or None,
        operation_restrictions=operation_restrictions,
        operation_log_params=oplog_params,
        without_ipmi=host.type != HostType.SERVER,
        use_cloud_post_processor=not ignore_cms and _use_cloud_postprocess(host.project, task_type),
        profile_after_task=True,
        redeploy_after_task=True,
    )
    schedule_task(host, reboot_task_args, get_reboot_task_stages)


def get_restrictions_for_profile(automated: bool, redeploy: bool) -> list[str]:
    if redeploy:
        if automated:
            check_restrictions = [
                restrictions.AUTOMATED_PROFILE,
                restrictions.AUTOMATED_REDEPLOY,
                restrictions.AUTOMATED_PROFILE_WITH_FULL_DISK_CLEANUP,
            ]
        else:
            check_restrictions = [restrictions.PROFILE, restrictions.REDEPLOY]
    else:
        if automated:
            check_restrictions = [restrictions.AUTOMATED_PROFILE, restrictions.AUTOMATED_PROFILE_WITH_FULL_DISK_CLEANUP]
        else:
            check_restrictions = [restrictions.PROFILE]
    return check_restrictions


def schedule_profile(
    issuer,
    task_type,
    host,
    profile=None,
    profile_tags=None,
    profile_mode=None,
    redeploy=None,
    provisioner=None,
    deploy_config=None,
    deploy_config_policy=None,
    deploy_tags=None,
    deploy_network=None,
    from_current_task=False,
    ignore_maintenance=False,
    repair_request_severity=None,
    ignore_cms=False,
    disable_admin_requests=False,
    force_update_network_location=False,
    monitor_on_completion=True,
    with_auto_healing=None,
    decision=None,
    reason=None,
    cms_task_id=None,
):
    """Schedules profiling of the specified host."""

    if not ignore_cms and _use_cloud_postprocess(host.project, task_type):
        redeploy = True

    if redeploy is False and deploy_config:
        raise LogicalError

    if with_auto_healing and not monitor_on_completion:
        raise RequestValidationError(
            "Auto healing can be enabled only when host checking after task completion is enabled."
        )

    profile, profile_tags, profile_modes = host.deduce_profile_configuration(profile, profile_tags, profile_mode)
    if redeploy is None:
        # redeploy requested by specifying deploy config, set flag accordingly
        # force redeploy after destructive profiles, unless explicitly forbidden.
        redeploy = (
            deploy_config
            or profile_modes is not None
            and any(mode in (ProfileMode.DISK_RW_TEST, ProfileMode.DANGEROUS_HIGHLOAD_TEST) for mode in profile_modes)
        )

    automated = issuer == authorization.ISSUER_WALLE
    check_restrictions = get_restrictions_for_profile(automated, redeploy)
    if redeploy:
        (
            host_provisioner,
            host_deploy_config,
            host_deploy_tags,
            host_deploy_network,
            host_deploy_config_policy,
            deploy_configuration,
        ) = host.deduce_deploy_configuration(
            provisioner, deploy_config, deploy_tags, deploy_network, deploy_config_policy
        )

        # TODO: CMS API supports only single action now. Use most destructive one.
        cms_action = CmsTaskAction.REDEPLOY
    else:
        (
            host_provisioner,
            host_deploy_config,
            host_deploy_tags,
            host_deploy_network,
            host_deploy_config_policy,
            deploy_configuration,
        ) = [None] * 6

        cms_action = CmsTaskAction.PROFILE

    with audit_log.on_profile_host(
        issuer,
        host.project,
        host.inv,
        host.name,
        host.uuid,
        profile,
        profile_tags,
        redeploy,
        deploy_configuration,
        ignore_cms,
        disable_admin_requests,
        monitor_on_completion,
        with_auto_healing=with_auto_healing,
        profile_modes=profile_modes,
        reason=reason,
        scenario_id=host.scenario_id,
    ) as audit_entry:
        allowed_states = HostState.ALL_ASSIGNED
        allowed_statuses = HostStatus.ALL_STEADY

        if from_current_task:
            query = host_query(host)
        else:
            query = check_state_and_get_query(
                host, issuer, task_type, ignore_maintenance, allowed_states, allowed_statuses
            )

        operation_restrictions = restrictions.check_restrictions(host, check_restrictions)
        query.update(restrictions__nin=operation_restrictions)

        reject_request_if_needed(issuer, task_type, host, cms_action, ignore_cms)

        log.info("%s: Scheduling profiling{}...".format(" with redeploy" if redeploy else ""), host.human_id())
        update_kwargs = Host.set_status_kwargs(
            None, Operation.PROFILE.host_status, issuer, audit_entry.id, reason=reason
        )

        if host_provisioner is not None:
            update_kwargs.update(
                fix_mongo_set_kwargs(
                    set__provisioner=host_provisioner,
                    set__config=host_deploy_config,
                    set__deploy_tags=host_deploy_tags,
                    set__deploy_network=host_deploy_network,
                    set__deploy_config_policy=host_deploy_config_policy,
                )
            )

        task = _new_profile_and_deploy_task(
            host,
            issuer,
            task_type,
            audit_entry,
            profile=profile,
            profile_tags=profile_tags,
            profile_modes=profile_modes,
            deploy_configuration=deploy_configuration,
            config_forced=deploy_config is not None,
            ignore_cms=ignore_cms,
            disable_admin_requests=disable_admin_requests,
            monitor_on_completion=monitor_on_completion,
            with_auto_healing=with_auto_healing,
            repair_request_severity=repair_request_severity,
            force_update_network_location=force_update_network_location,
            extra_checks=decision.checks if decision else None,
            failure=decision.failures[0] if decision and decision.failures else None,
            reason=reason,
            failure_type=decision.failure_type if decision else None,
            operation_type=decision.get_param("operation") if decision else None,
            cms_task_id=cms_task_id,
        )

        if not host.modify(query, set__task=task, **update_kwargs):
            raise InvalidHostStateError(host, allowed_states=allowed_states, allowed_statuses=allowed_statuses)

    notifications.on_task_enqueued(issuer, host, reason)


def _new_profile_and_deploy_task(
    host,
    issuer,
    task_type,
    audit_entry,
    profile,
    profile_tags,
    profile_modes,
    deploy_configuration,
    config_forced=False,
    ignore_cms=False,
    disable_admin_requests=False,
    monitor_on_completion=True,
    with_auto_healing=False,
    failure=None,
    extra_checks=None,
    reason=None,
    failure_type=None,
    operation_type=None,
    repair_request_severity=None,
    force_update_network_location=False,
    cms_task_id=None,
):

    if profile is None and deploy_configuration is None:
        raise LogicalError  # neither profile nor redeploy stages were generated

    cms_action = CmsTaskAction.REDEPLOY if deploy_configuration is not None else CmsTaskAction.PROFILE
    health_status_accuracy = None
    extra_checks = sorted(set(extra_checks or ()))

    builder = StageBuilder()
    builder.stage(
        Stages.ACQUIRE_PERMISSION,
        action=cms_action,
        comment=reason,
        failure=failure,
        check_names=extra_checks or None,
        failure_type=failure_type or None,
    )
    builder.stage(Stages.SET_DOWNTIME)

    if profile is not None:
        verify_network_location = False
        project = host.get_project()
        if project.vlan_scheme in VLAN_SCHEMES_WITH_VERIFYING:
            verify_network_location = True
        builder.add_stages(
            get_profile_stages(
                EineProfileOperation.PROFILE,
                profile,
                profile_tags,
                profile_modes,
                operation_type=operation_type,
                repair_request_severity=repair_request_severity,
                force_update_network_location=force_update_network_location,
                verify_network_location=verify_network_location,
            )
        )

    if deploy_configuration is not None:
        # WALLE-3511: temporarily disable escalation for automated tasks
        # only keep for manual tasks with special hackery
        allow_autohealing = with_auto_healing or (task_type == TaskType.AUTOMATED_HEALING)
        with_2nd_time_node = (
            allow_autohealing
            and not disable_admin_requests
            and (reason and "2nd_time_node" in reason.replace("-", "_"))
        )
        builder.add_stages(
            get_deploy_stages(
                deploy_configuration,
                config_forced=config_forced,
                # WALLE-3511: temporarily disable escalation for automated tasks
                # only keep for manual tasks with special hackery
                with_autohealing=allow_autohealing,
                with_2nd_time_node=with_2nd_time_node,
                profile_tags=profile_tags,
            )
        )

        health_status_accuracy = (
            deploy.COMPLETED_STATUS_TIME_ACCURACY if deploy_configuration.provisioner == PROVISIONER_LUI else None
        )

    builder.stage(Stages.SWITCH_VLANS, network=NetworkTarget.PROJECT)

    # Einstellung firmware profile stages power off the host at end of the stage - it's not that we want.
    # Even if we have our host redeployed, let's just ensure.
    builder.add_stages(get_power_on_stages())

    return new_task(
        issuer,
        task_type,
        audit_entry,
        stages=builder.get_stages(),
        host=host,
        ignore_cms=ignore_cms,
        disable_admin_requests=disable_admin_requests,
        monitor_on_completion=monitor_on_completion,
        checks_to_monitor=CheckGroup.NETWORK_AVAILABILITY + extra_checks,
        health_status_accuracy=health_status_accuracy,
        with_auto_healing=with_auto_healing,
        cms_task_id=cms_task_id,
    )


def schedule_redeploy(
    issuer,
    task_type,
    host,
    provisioner=None,
    deploy_config=None,
    deploy_tags=None,
    deploy_network=None,
    from_current_task=False,
    ignore_maintenance=False,
    ignore_cms=False,
    disable_admin_requests=False,
    monitor_on_completion=True,
    with_auto_healing=None,
    decision=None,
    reason=None,
    custom_profile_mode=None,
    deploy_config_policy=None,
    cms_task_id=None,
):
    """Schedules redeploying of the specified host."""

    (
        host_provisioner,
        host_deploy_config,
        host_deploy_tags,
        host_deploy_network,
        host_deploy_config_policy,
        deploy_configuration,
    ) = host.deduce_deploy_configuration(provisioner, deploy_config, deploy_tags, deploy_network, deploy_config_policy)

    if with_auto_healing and not monitor_on_completion:
        raise RequestValidationError(
            "Auto healing can be enabled only when host checking after task completion is enabled."
        )

    project = host.get_project()
    if custom_profile_mode is not None:
        profile, profile_tags, profile_modes = host.deduce_profile_configuration(profile_mode=custom_profile_mode)
        log.debug(
            (
                "Debug profile settings in scheduled redeploy: Custom profile: '%s', "
                "profile: '%s', profile_tags: '%s', profile_modes: '%s'"
            )
            % (custom_profile_mode, profile, profile_tags, profile_modes)
        )
    elif task_type in TaskType.ALL_AUTOMATED:
        profile, profile_tags, profile_modes = _get_profile_configuration_needed_for_redeploy(
            host, deploy_configuration.provisioner
        ) or (None, project.profile_tags, None)
        log.debug(
            (
                "Debug profile settings in scheduled redeploy: Task type: '%s', "
                "profile: '%s', profile_tags: '%s', profile_modes: '%s'"
            )
            % (task_type, profile, profile_tags, profile_modes)
        )
    else:
        profile, profile_tags, profile_modes = None, project.profile_tags, None
        log.debug("Debug profile settings in scheduled redeploy: Not used")

    with audit_log.on_redeploy_host(
        issuer,
        host.project,
        host.inv,
        host.name,
        host.uuid,
        deploy_configuration,
        ignore_cms,
        disable_admin_requests,
        monitor_on_completion,
        with_auto_healing,
        set_provisioner=host_provisioner,
        set_config=host_deploy_config,
        set_tags=host_deploy_tags,
        set_network=host_deploy_network,
        set_deploy_config_policy=host_deploy_config_policy,
        profile=profile,
        profile_tags=profile_tags,
        profile_modes=profile_modes,
        reason=reason,
        scenario_id=host.scenario_id,
    ) as audit_entry:
        # TODO: It would be fair to check also for profile restriction when we profile host, but sadly DMC can't
        # handle such unpredictability of expected restrictions at this time. But since redeploy is always more
        # destructive operation then profiling, we can live with it now.
        _restriction = (
            restrictions.AUTOMATED_REDEPLOY if issuer == authorization.ISSUER_WALLE else restrictions.REDEPLOY
        )
        operation_restrictions = restrictions.check_restrictions(host, _restriction)
        allowed_states = HostState.ALL_ASSIGNED
        allowed_statuses = HostStatus.ALL_STEADY

        if from_current_task:
            query = host_query(host)
        else:
            query = check_state_and_get_query(
                host, issuer, task_type, ignore_maintenance, allowed_states, allowed_statuses
            )

        query.update(restrictions__nin=operation_restrictions)

        reject_request_if_needed(issuer, task_type, host, CmsTaskAction.REDEPLOY, ignore_cms)

        log.info("%s: Scheduling redeploy...", host.human_id())

        update_kwargs = Host.set_status_kwargs(
            None, Operation.REDEPLOY.host_status, issuer, audit_entry.id, reason=reason
        )

        if host_provisioner is not None:
            update_kwargs.update(
                fix_mongo_set_kwargs(
                    set__provisioner=host_provisioner,
                    set__config=host_deploy_config,
                    set__deploy_config_policy=host_deploy_config_policy,
                    set__deploy_tags=host_deploy_tags,
                    set__deploy_network=host_deploy_network,
                )
            )

        task = _new_profile_and_deploy_task(
            host,
            issuer,
            task_type,
            audit_entry,
            deploy_configuration=deploy_configuration,
            config_forced=deploy_config is not None,
            profile=profile,
            profile_tags=profile_tags,
            profile_modes=profile_modes,
            ignore_cms=ignore_cms,
            disable_admin_requests=disable_admin_requests,
            monitor_on_completion=monitor_on_completion,
            with_auto_healing=with_auto_healing,
            extra_checks=decision.checks if decision else None,
            failure=decision.failures[0] if decision and decision.failures else None,
            reason=reason,
            failure_type=decision.failure_type if decision else None,
            cms_task_id=cms_task_id,
        )

        if not host.modify(query, set__task=task, **update_kwargs):
            raise InvalidHostStateError(host, allowed_states=allowed_states, allowed_statuses=allowed_statuses)

    notifications.on_task_enqueued(issuer, host, reason)


def schedule_dns_check(
    issuer,
    task_type,
    host,
    ignore_maintenance=False,
    disable_admin_requests=False,
    monitor_on_completion=True,
    with_auto_healing=None,
    reason=None,
):
    """Schedule DNS check and repair. Don't force DNS records to be pushed, just check and fix errors."""

    # NB: we don't ask CMS for permission for DNS fix operations

    # This is a handle for manual DNS repair (triggered by hand or scenario). It is not used from the healing process.
    # This operation doesn't check automation limits and restrictions.
    if task_type not in [TaskType.MANUAL, TaskType.AUTOMATED_ACTION]:
        raise LogicalError()

    project = host.get_project(fields=("vlan_scheme", "native_vlan", "dns_domain"))
    walle.network.assert_dns_assigning_supported(project, host.name)

    task = CheckDnsTaskHelper(
        issuer,
        task_type,
        host,
        reason=reason,
        ignore_maintenance=ignore_maintenance,
        disable_admin_requests=disable_admin_requests,
        with_auto_healing=with_auto_healing,
    )

    task.acquire_permission(ignore_cms=True)
    task.get_network_location(host, full=False)
    task.setup_dns()
    task.log_completed_operation(Operation.CHECK_DNS)

    if monitor_on_completion:
        task.monitor(CheckGroup.NETWORK_AVAILABILITY, timeout=DEFAULT_DNS_TTL)

    with task.audit(audit_log.on_check_dns_records, reason) as audit_entry:
        log.info("%s: Scheduling host DNS records check...", host.human_id())

        query = check_state_and_get_query(
            host, issuer, task_type, ignore_maintenance, HostState.ALL_ASSIGNED, HostStatus.ALL_STEADY
        )
        status_kwargs = Host.set_status_kwargs(
            None, Operation.CHECK_DNS.host_status, issuer, audit_entry.id, reason=reason
        )
        task.enqueue(query, status_kwargs)


def schedule_bmc_reset(issuer, task_type, host, decision, reason, from_current_task=False, ignore_maintenance=False):
    """Schedules resetting the BMC."""

    if task_type != TaskType.AUTOMATED_HEALING:
        raise LogicalError()

    task = TaskHelper(issuer, task_type, host, reason)
    cms_action = CmsTaskAction.REBOOT
    if _use_cloud_postprocess(host.project, task_type):
        cms_action = CmsTaskAction.REDEPLOY
    task.acquire_permission(action=cms_action, comment=reason, decision=decision)
    task.set_downtime()
    task.reset_bmc()
    task.log_completed_operation(Operation.RESET_BMC)
    task.monitor(decision.checks)
    if cms_action == CmsTaskAction.REDEPLOY:
        task.cloud_post_processor(need_profile=True, need_redeploy=True)

    task.add_audit(failure_type=decision.get_param("failure_type"))
    with task.audit(audit_log.on_bmc_reset, reason) as audit_entry:
        operation_restrictions = restrictions.check_restrictions(host, decision.get_restrictions())

        if from_current_task:
            query = host_query(host)
        else:
            query = check_state_and_get_query(
                host, issuer, task_type, ignore_maintenance, HostState.ALL_ASSIGNED, HostStatus.ALL_STEADY
            )

        query.update(restrictions__nin=operation_restrictions)
        status_kwargs = Host.set_status_kwargs(
            None, Operation.RESET_BMC.host_status, issuer, audit_entry.id, reason=reason
        )

        log.info("%s: Scheduling repair of problems with BMC...", host.human_id())
        task.enqueue(query, status_kwargs)


def schedule_hardware_repairing(
    issuer, task_type, host, decision, reason, from_current_task=False, ignore_maintenance=False
):
    """Schedules hardware repairing task."""

    if task_type != TaskType.AUTOMATED_HEALING:
        raise LogicalError()

    decision_params = decision.params.copy()
    operation = Operation(decision_params.pop("operation"))

    # NB: LINK_REPAIR is not profile (or is it?), probably need some more parametrization here
    reboot = decision_params.pop("reboot", False)
    redeploy = decision_params.pop("redeploy", False)
    power_on_before_repair = decision_params.pop("power_on_before_repair", False)
    cms_action = CmsTaskAction.PROFILE
    if redeploy or _use_cloud_postprocess(host.project, task_type):
        cms_action = CmsTaskAction.REDEPLOY

    task = TaskHelper(issuer, task_type, host, reason)
    task.acquire_permission(cms_action, comment=reason, extra=decision.params, decision=decision)
    task.set_downtime()

    with task.optional_power_off(reboot or redeploy):
        if power_on_before_repair:
            task.add_audit(power_on_before_repair=True)
            task.power_on()

        if operation.type == Operation.REPORT_SECOND_TIME_NODE.type:
            task.provide_diagnostic_host_access()

        task.repair_hardware(decision, decision_params, reason)

        if redeploy:
            deploy_configuration = host.get_deploy_configuration()
            profile_configuration = _get_profile_configuration_needed_for_redeploy(
                host, deploy_configuration.provisioner
            )

            if profile_configuration:
                task.add_audit(profile=True)
                task.profile(profile_configuration)

            task.add_audit(redeploy=True)
            task.redeploy(deploy_configuration)
            task.switch_vlans(NetworkTarget.PROJECT)

    # Currently assume that we want to limit hardware operations by type
    # even if different parts break, e.g. if we change x _different_ memory modules
    # we still want to hit the operation limit.
    params = drop_none(
        dict(request_type=decision_params.get("request_type", None), slot=decision_params.get("slot", None))
    )
    task.log_completed_operation(operation, **params)
    task.monitor(checks=CheckGroup.NETWORK_AVAILABILITY + decision.checks)

    if not redeploy and _use_cloud_postprocess(host.project, task_type):
        task.cloud_post_processor(need_profile=True, need_redeploy=True)

    with task.audit(audit_log.on_hardware_repair, reason) as audit_entry:
        operation_restrictions = restrictions.check_restrictions(host, decision.get_restrictions())

        if from_current_task:
            query = host_query(host)
        else:
            query = check_state_and_get_query(
                host, issuer, task_type, ignore_maintenance, HostState.ALL_ASSIGNED, HostStatus.ALL_STEADY
            )

        query.update(restrictions__nin=operation_restrictions)
        status_kwargs = Host.set_status_kwargs(None, operation.host_status, issuer, audit_entry.id, reason=reason)

        log.info("%s: Scheduling hardware repair task...", host.human_id())
        task.enqueue(query, status_kwargs)


def schedule_disk_change(
    issuer, task_type, host, decision, reason, from_current_task=False, ignore_maintenance=False, audit_log_type=None
):
    """Schedules change of a corrupted disk."""

    if task_type != TaskType.AUTOMATED_HEALING:
        raise LogicalError()

    details = decision.params.copy()
    profile_configuration, deploy_configuration = None, None

    if decision.params.get("redeploy", False):
        deploy_configuration = host.get_deploy_configuration()

    if "NVME_LINK_DEGRADED" not in decision.params.get("eine_code", []) and (
        "slot" in decision.params or "serial" in decision.params
    ):
        if decision.params.get("redeploy", False):
            profile_configuration = _get_profile_configuration_needed_for_redeploy(
                host, deploy_configuration.provisioner
            )
    else:
        profile_configuration = host.deduce_profile_configuration(profile_mode=ProfileMode.DISK_RW_TEST)

    if profile_configuration is not None:
        details.update(
            profile=profile_configuration.profile,
            profile_tags=profile_configuration.tags,
            profile_modes=profile_configuration.modes,
        )

    if not audit_log_type:
        audit_log_type = audit_log.on_change_disk

    with audit_log_type(issuer, host.project, host.inv, host.name, host.uuid, details, reason) as audit_entry:
        operation_restrictions = restrictions.check_restrictions(host, decision.get_restrictions())
        allowed_states = HostState.ALL_ASSIGNED
        allowed_statuses = HostStatus.ALL_STEADY

        if from_current_task:
            query = host_query(host, restrictions__nin=operation_restrictions)
        else:
            query = check_state_and_get_query(
                host,
                issuer,
                task_type,
                ignore_maintenance,
                allowed_states,
                allowed_statuses,
                restrictions__nin=operation_restrictions,
            )

        query.update(restrictions__nin=operation_restrictions)

        log.info("%s: Scheduling change of a corrupted disk...", host.human_id())

        if "NVME_LINK_DEGRADED" not in decision.params.get("eine_code", []) and (
            "slot" in decision.params or "serial" in decision.params
        ):
            cms_extra = drop_none(filter_dict_keys(decision.params, {"slot", "serial"}))
            task = _new_disk_change_task(
                host,
                issuer,
                task_type,
                audit_entry,
                decision,
                profile_configuration,
                deploy_configuration,
                cms_extra=cms_extra,
                reason=reason,
            )
        else:
            task = _new_profile_and_deploy_task(
                host,
                issuer,
                task_type,
                audit_entry,
                profile=profile_configuration.profile,
                profile_tags=profile_configuration.tags,
                profile_modes=profile_configuration.modes,
                deploy_configuration=deploy_configuration,
                extra_checks=[CheckType.DISK],
                failure=decision.failures[0] if decision.failures else None,
                reason=reason,
                failure_type=decision.failure_type if decision else None,
            )

        if not host.modify(
            query,
            set__task=task,
            **Host.set_status_kwargs(None, Operation.CHANGE_DISK.host_status, issuer, audit_entry.id, reason=reason),
        ):
            raise InvalidHostStateError(host, allowed_states=allowed_states, allowed_statuses=allowed_statuses)

    notifications.on_task_enqueued(issuer, host, reason)


def _new_disk_change_task(
    host,
    issuer,
    task_type,
    audit_entry,
    decision,
    profile_configuration=None,
    deploy_configuration=None,
    cms_extra=None,
    reason=None,
):
    needs_redeploy = deploy_configuration is not None

    # TODO: We need stages to be more dynamic here:
    # Stage params will be set by Stages.CHANGE_DISK stage. This hack is required, because operation params aren't known
    # at this code point and will be known only during task processing.
    change_disk_stage = Stage(
        name=Stages.CHANGE_DISK, params={"redeploy": needs_redeploy}, data={"orig_decision": decision.to_dict()}
    )
    log_operation_stage = Stage(name=Stages.LOG_COMPLETED_OPERATION, params={"operation": Operation.CHANGE_DISK.type})

    builder = StageBuilder()
    builder.stage(
        Stages.ACQUIRE_PERMISSION,
        # TODO: CMS API supports only single action now. Use most destructive one.
        action=CmsTaskAction.REDEPLOY if needs_redeploy else CmsTaskAction.CHANGE_DISK,
        failure=decision.failures[0] if decision and decision.failures else None,
        check_names=decision.checks if decision and decision.checks else None,
        failure_type=decision.failure_type if decision else None,
        comment=reason,
        extra=cms_extra,
    )
    builder.stage(Stages.SET_DOWNTIME)

    builder.stage(Stages.POWER_OFF, soft=True)

    with builder.nested(Stages.HEAL_DISK) as disk_stages:
        disk_stages.add_stages([change_disk_stage])

        if profile_configuration is not None:
            disk_stages.add_stages(
                get_profile_stages(
                    EineProfileOperation.PROFILE,
                    profile_configuration.profile,
                    profile_configuration.tags,
                    profile_configuration.modes,
                )
            )

        if needs_redeploy:
            disk_stages.add_stages(get_deploy_stages(deploy_configuration, with_autohealing=True))

        if profile_configuration is not None or needs_redeploy:
            disk_stages.stage(Stages.SWITCH_VLANS, network=NetworkTarget.PROJECT)

    builder.add_stages([log_operation_stage])

    task = new_task(
        issuer,
        task_type,
        audit_entry,
        stages=builder.get_stages(),
        host=host,
        monitor_on_completion=True,
        checks_to_monitor=CheckGroup.NETWORK_AVAILABILITY + [CheckType.DISK],
    )

    # Link the stages after all UIDs are set
    change_disk_stage.params["log_operation_stage_uid"] = log_operation_stage.uid

    return task


def schedule_report_task(issuer, task_type, host, decision, reason, from_current_task=False, ignore_maintenance=False):
    """Schedules task that reports host failure to startrek."""

    if task_type != TaskType.AUTOMATED_HEALING:
        raise LogicalError()

    task = ReportTaskHelper(issuer, task_type, host, reason=reason)
    task.acquire_permission(ignore_cms=True)
    task.report_failure(decision.checks, reason)
    task.log_completed_operation(Operation.REPORT_FAILURE)
    task.monitor(decision.checks)

    with task.audit(audit_log.on_report_host_failure, reason) as audit_entry:
        operation_restrictions = restrictions.check_restrictions(host, decision.get_restrictions())

        if from_current_task:
            query = host_query(host)
        else:
            query = check_state_and_get_query(
                host, issuer, task_type, ignore_maintenance, HostState.ALL_ASSIGNED, HostStatus.ALL_STEADY
            )

        query.update(restrictions__nin=operation_restrictions)
        status_kwargs = Host.set_status_kwargs(
            None, Operation.REPORT_FAILURE.host_status, issuer, audit_entry.id, reason=reason
        )

        log.info("%s: Scheduling report task for host failure...", host.human_id())
        task.enqueue(query, status_kwargs)


def schedule_rack_repair(issuer, task_type, host, decision, reason, from_current_task=False, ignore_maintenance=False):
    """Schedules task that reports rack failure to ITDC."""

    if task_type != TaskType.AUTOMATED_HEALING:
        raise LogicalError()

    task = ReportTaskHelper(issuer, task_type, host, reason=reason)
    task.acquire_permission(ignore_cms=True)
    task.repair_rack_failure(decision.checks, reason)
    task.log_completed_operation(Operation.REPAIR_RACK_FAILURE)
    task.monitor(CheckGroup.NETWORK_AVAILABILITY)

    location = host.location.get_path()
    task.add_audit(location_path=location)

    with task.audit(audit_log.on_repair_rack, reason) as audit_entry:
        operation_restrictions = restrictions.check_restrictions(host, decision.get_restrictions())

        if from_current_task:
            query = host_query(host)
        else:
            query = check_state_and_get_query(
                host, issuer, task_type, ignore_maintenance, HostState.ALL_ASSIGNED, HostStatus.ALL_STEADY
            )

        query.update(restrictions__nin=operation_restrictions)
        status_kwargs = Host.set_status_kwargs(
            None, Operation.REPAIR_RACK_FAILURE.host_status, issuer, audit_entry.id, reason=reason
        )

        log.info("%s: Scheduling rack repair for %s...", host.human_id(), location)
        task.enqueue(query, status_kwargs)


def schedule_rack_overheat_repair(
    issuer, task_type, host, decision, reason, from_current_task=False, ignore_maintenance=False
):
    """Schedules task that reports rack overheats to ITDC."""

    if task_type != TaskType.AUTOMATED_HEALING:
        raise LogicalError()

    task = ReportTaskHelper(issuer, task_type, host, reason=reason)
    task.acquire_permission(ignore_cms=True)
    task.repair_rack_overheat(decision.checks, reason)
    task.log_completed_operation(Operation.REPAIR_RACK_OVERHEAT)
    task.monitor(CheckGroup.NETWORK_AVAILABILITY)

    location = host.location.get_path()
    task.add_audit(location_path=location)

    with task.audit(audit_log.on_repair_rack, reason) as audit_entry:
        operation_restrictions = restrictions.check_restrictions(host, decision.get_restrictions())

        if from_current_task:
            query = host_query(host)
        else:
            query = check_state_and_get_query(
                host, issuer, task_type, ignore_maintenance, HostState.ALL_ASSIGNED, HostStatus.ALL_STEADY
            )

        query.update(restrictions__nin=operation_restrictions)
        status_kwargs = Host.set_status_kwargs(
            None, Operation.REPAIR_RACK_OVERHEAT.host_status, issuer, audit_entry.id, reason=reason
        )

        log.info("%s: Scheduling rack overheat repair for %s...", host.human_id(), location)
        task.enqueue(query, status_kwargs)


def schedule_deactivation(
    issuer, task_type, host, reason=None, from_current_task=False, ignore_maintenance=False, decision=None
):
    """Schedules deactivation of the specified host."""
    del decision  # accept decision for conformance with other task schedulers. This one does not need it.

    # TODO: notify project's CMS about this stuff somehow.
    with audit_log.on_deactivate_host(
        issuer, host.project, host.inv, host.name, host.uuid, reason=reason
    ) as audit_entry:
        allowed_states = HostState.ALL_ASSIGNED
        allowed_statuses = set(HostStatus.DEFAULT_STATUSES.values())
        if from_current_task:
            query = host_query(host)
        else:
            query = check_state_and_get_query(
                host, issuer, task_type, ignore_maintenance, allowed_states, allowed_statuses
            )

        log.info("%s: Scheduling deactivation...", host.human_id())

        task = _new_deactivate_task(host, issuer, task_type, audit_entry, reason=reason)
        if not host.modify(
            query,
            set__task=task,
            **Host.set_status_kwargs(None, Operation.DEACTIVATE.host_status, issuer, audit_entry.id, reason=reason),
        ):
            raise InvalidHostStateError(host, allowed_states=allowed_states, allowed_statuses=allowed_statuses)

    notifications.on_task_enqueued(issuer, host, reason)


def _new_deactivate_task(host, issuer, task_type, audit_entry, reason=None):
    builder = StageBuilder()
    builder.stage(Stages.DEACTIVATE, reason=reason)
    builder.stage(Stages.LOG_COMPLETED_OPERATION, operation=Operation.DEACTIVATE.type)
    return new_task(issuer, task_type, audit_entry, stages=builder.get_stages(), host=host, next_check=timestamp())


def host_query(host, revision=None, **kwargs):
    kwargs.setdefault("state", host.state)
    kwargs.setdefault("status", host.status)

    return dict(
        kwargs, task__task_id=host.task.task_id, task__revision=host.task.revision if revision is None else revision
    )


def _get_profile_configuration_needed_for_redeploy(host, provisioner):
    if provisioner != PROVISIONER_LUI:
        return

    try:
        profile_configuration = host.deduce_profile_configuration()
    except MissingDefaultProfileConfigurationError:
        return

    return profile_configuration


def schedule_bot_project_sync(issuer, host, bot_project_id, reason):
    task = TaskHelper(issuer, TaskType.AUTOMATED_ACTION, host, reason=reason, monitor_on_completion=False)
    task.assign_bot_project(bot_project_id)

    with task.audit(audit_log.on_bot_project_sync) as audit_entry:
        log.info("%s: Scheduling bot project sync %s...", host.human_id(), bot_project_id)

        query = check_state_and_get_query(
            host,
            issuer,
            TaskType.AUTOMATED_ACTION,
            ignore_maintenance=True,
            allowed_states=HostState.ALL_ASSIGNED,
            allowed_statuses=HostStatus.ALL_STEADY,
        )

        status_kwargs = Host.set_status_kwargs(
            None, Operation.BOT_PROJECT_SYNC.host_status, issuer, audit_entry.id, reason=reason
        )

        task.enqueue(query, status_kwargs)


def schedule_fqdn_deinvalidation(
    issuer,
    task_type,
    host,
    ignore_cms=False,
    disable_admin_requests=True,
    release=False,
    ignore_maintenance=False,
    clear_old_fqdn_records=False,
    reason=None,
):
    """Schedule FQDN deinvalidation. Get data from bot and replace fqdn. Set host's status as free if released"""

    # This is a handle for manual FQDN deinvalidation. It is not used from the healing process.
    # This operation doesn't check automation limits and restrictions.
    if task_type != TaskType.MANUAL:
        raise LogicalError()

    current_project = host.get_project()

    task_args = FqdnDeinvalidationArgs(
        issuer=issuer,
        task_type=task_type,
        project=host.project,
        host_inv=host.inv,
        host_name=host.name,
        host_uuid=host.uuid,
        scenario_id=host.scenario_id,
        ignore_cms=ignore_cms,
        disable_admin_requests=disable_admin_requests,
        reason=reason,
        bot_project_id=current_project.bot_project_id,
        release=release,
        ignore_maintenance=ignore_maintenance,
        cms_task_id=host.cms_task_id,
        clear_old_fqdn_records=clear_old_fqdn_records,
    )
    schedule_task_from_api(host, task_args, get_fqdn_deinvalidation_stages)
