"""API utilities"""

import functools
import http.client
import inspect
import itertools
import logging
import re
import sys
import typing as tp
from collections import namedtuple, OrderedDict

import fastjsonschema
import pymongo
import six
from flask import Flask, request

from sepelib.core import config
from sepelib.core.exceptions import Error, LogicalError
from sepelib.flask.h import prep_response
from sepelib.flask.h.error_handlers import exception_handler
from walle.application import app
from walle.authorization import blackbox, iam, has_iam, csrf
from walle.clients.tvm import check_ticket, TvmSourceIsNotAllowed, TvmUnknownAppId
from walle.constants import ROBOT_WALLE_OWNER, TESTING_ENV_NAME, HostType
from walle.errors import (
    BadRequestError,
    RequestValidationError,
    ApiError,
    ResourceNotFoundError,
    UnauthenticatedError,
    MethodNotAllowedError,
)
from walle.models import get_requested_fields
from walle.projects import Project
from walle.statbox.contexts import exception_context
from walle.statbox.loggers import api_logger
from walle.util import patterns
from walle.util.content_encoding import ContentEncoder
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.misc import ellipsis_string, InvOrUUIDOrName
from walle.util.mongo import MongoDocument, SECONDARY_LOCAL_DC_PREFERRED

log = logging.getLogger(__name__)

_DEFAULT_PAGE_SIZE = 100
_FALLBACK_MAX_PAGE_SIZE = 1000
_DEFAULT_STRICT_API = False
_DEFAULT_MIME_TYPE = "application/json"

_API_REFERENCE = OrderedDict()
_DATA_METHODS = {"PUT", "POST", "PATCH", "DELETE"}

_ApiMethod = namedtuple("ApiMethod", ["uri", "methods", "query_schema", "json_schema", "description"])


class ExtendedFlask(Flask):
    def __call__(self, environ, start_response):
        environ["SCRIPT_NAME"] = config.get_value("web.http.url_prefix") + environ["SCRIPT_NAME"]
        return super().__call__(environ, start_response)


def get_api_reference():
    return _API_REFERENCE


def api_handler(*args, **kwargs):
    include_to_api_reference = kwargs.pop("include_to_api_reference", True)
    blueprint = kwargs.pop("blueprint", app.api_blueprint)
    return generic_api_handler(
        blueprint,
        *args,
        include_to_api_reference=include_to_api_reference,
        oauth_client_id=config.get_value("oauth.client_id"),
        **kwargs,
    )


def admin_request(func):
    @functools.wraps(func)
    def decorated_function(*args, **kwargs):
        if not has_iam():
            blackbox.authorize(
                kwargs["issuer"],
                "You must have admin privileges in Wall-E to perform this request.",
                authorize_admins=True,
            )
        return func(*args, **kwargs)

    return decorated_function


def production_only(func):
    @functools.wraps(func)
    def decorated_function(*args, **kwargs):
        if not has_iam():
            env_name = config.get_value("environment.name", TESTING_ENV_NAME)
            if env_name == TESTING_ENV_NAME:
                blackbox.authorize(
                    kwargs["issuer"],
                    "This method is available only for Wall-E admins on {} environment.".format(env_name),
                    authorize_admins=True,
                )

        return func(*args, **kwargs)

    return decorated_function


def walle_only(func):
    # TODO(rocco66): move to api_handler
    @functools.wraps(func)
    def decorated_function(*args, **kwargs):
        if not has_iam():
            blackbox.authorize(
                kwargs["issuer"],
                "You must be Wall-E, but you are not.",
                [ROBOT_WALLE_OWNER],
                authorize_admins=False,
                authorize_by_group=False,
            )
        return func(*args, **kwargs)

    return decorated_function


def _does_func_have_issuer_arg(func):
    spec = inspect.signature(func)
    return "issuer" in spec.parameters


def _get_iam_permissions(iam_permissions: tp.Optional[iam.BaseApiIamPermission]) -> list[iam.BaseApiIamPermission]:
    if iam_permissions is None:
        return []
    elif isinstance(iam_permissions, iam.BaseApiIamPermission):
        return [iam_permissions]
    elif isinstance(iam_permissions, list):
        return iam_permissions
    else:
        raise RuntimeError("wrong IamPermission usage, use BaseApiIamPermission or list[BaseApiIamPermission]")


OneOrManyApiIamPermissions = tp.Union[iam.BaseApiIamPermission, list[iam.BaseApiIamPermission]]


def generic_api_handler(
    blueprint,
    path,
    methods,
    schema=None,
    params=None,
    authenticate=False,
    with_sudo=False,
    with_ignore_maintenance=False,
    with_fields=None,
    with_paging=None,
    with_sort=None,
    with_reason=False,
    include_to_api_reference=False,
    oauth_client_id=None,
    allowed_tvm_sources_aliases_fn=None,
    rps=None,
    max_concurrent_requests=None,
    concurrent_requests_timeout=30,
    allowed_mime_type="application/json",
    form_processor=None,
    params_validator=None,
    body_validator=None,
    iam_permissions: tp.Optional[OneOrManyApiIamPermissions] = None,
    allowed_host_types: tp.Optional[list[HostType]] = None,
):
    # Yes, there is a lot of magic here. But it really saves us from a lot of copy-paste.

    if isinstance(methods, str):
        methods = (methods,)

    params, schema = _generate_api_schema(
        methods, params, schema, with_sudo, with_ignore_maintenance, with_fields, with_paging, with_reason, with_sort
    )

    if params_validator is None and params is not None:
        params_validator = fastjsonschema.compile(
            {
                "type": "object",
                "properties": params,
                "additionalProperties": False,
            }
        )

    if body_validator is None and schema is not None:
        body_validator = fastjsonschema.compile(schema)

    def decorator(func):
        if include_to_api_reference:
            uri = blueprint.url_prefix + path
            category = sys.modules[func.__module__].__doc__
            _API_REFERENCE.setdefault(category, []).append(
                _ApiMethod(uri, tuple(methods), params, schema, func.__doc__)
            )

        func_args_names = list(inspect.signature(func).parameters)

        @blueprint.route(path, methods=methods)
        @functools.wraps(func)
        def decorated_function(*args, **kwargs):
            with app.flask.limit_manager.check_limit(
                func.__name__,
                rps=rps,
                max_concurrent=max_concurrent_requests,
                concurrent_timeout=concurrent_requests_timeout,
            ):
                request_url = request.path
                if request.query_string:
                    request_url += "?" + six.ensure_str(request.query_string)

                issuer_log_prefix = ""
                handler_kwargs = {}
                try:
                    # TODO(rocco66): return parsed query and request, not kwargs
                    handler_kwargs = _parse_request(
                        params,
                        schema,
                        params_validator,
                        body_validator,
                        request_url,
                        with_reason,
                        allowed_tvm_sources_aliases_fn,
                        allowed_mime_type,
                        form_processor,
                    )
                    query_args = handler_kwargs.get("query_args", {})
                    request_obj = handler_kwargs.get("request", {})
                    if issuer := _authenticate(authenticate, oauth_client_id, iam_permissions, query_args, request_obj):
                        issuer_log_prefix = f"[{issuer}] "
                        handler_kwargs["issuer"] = issuer
                    if allowed_host_types:
                        handler_kwargs["allowed_host_types"] = allowed_host_types
                finally:
                    _log_request(issuer_log_prefix, request_url, request_json=handler_kwargs.get("request"))

                conflicting_kwargs = set(kwargs) & set(handler_kwargs)
                if conflicting_kwargs:
                    raise Error(
                        "Logical error: Request handler got conflicting arguments: {}.", ", ".join(conflicting_kwargs)
                    )

                kwargs.update({k: v for k, v in handler_kwargs.items() if k in func_args_names})
                return func(*args, **kwargs)

        return decorated_function

    return decorator


def configure_api_blueprint(blueprint):
    """Configures the blueprint for API requests handling."""

    def unknown_uri_handler(path):
        raise ResourceNotFoundError("Unknown resource '{}'", path)

    # Set up a "catch-all" handler that returns 404 for unknown URIs
    blueprint.add_url_rule("/", view_func=unknown_uri_handler, defaults={"path": ""})
    blueprint.add_url_rule("/<path:path>", view_func=unknown_uri_handler)

    def set_cache_control(response):
        response.cache_control.no_cache = True
        return response

    # API responses mustn't be cached
    blueprint.after_request(set_cache_control)

    from walle.request import log_request, log_response

    blueprint.before_request(log_request)
    blueprint.after_request(log_response)

    blueprint.errorhandler(ApiError)(api_error_handler)
    blueprint.errorhandler(Exception)(internal_error_handler)

    # Enable response compression
    encoder = ContentEncoder()
    encoder.init_app(blueprint)


def host_id_handler(*args, **kwargs):
    def decorator(func):
        @functools.wraps(func)
        def decorated_function(*args, **kwargs):
            if "host_id" not in kwargs or "host_id_query" in kwargs:
                raise RequestValidationError("Invalid host ID handler.")

            kwargs["host_id_query"] = InvOrUUIDOrName(kwargs.pop("host_id"))
            return func(*args, **kwargs)

        return api_handler(*args, **kwargs)(decorated_function)

    return decorator


def scenario_id_handler(*args, **kwargs):
    def decorator(func):
        @functools.wraps(func)
        def decorated_function(*args, **kwargs):
            if "scenario_id" not in kwargs or not kwargs['scenario_id'].isdigit():
                raise RequestValidationError("Invalid scenario ID")
            return func(*args, **kwargs)

        return api_handler(*args, **kwargs)(decorated_function)

    return decorator


def api_response(obj, **kwargs):
    kwargs.setdefault("fmt", "json")
    return prep_response(obj, **kwargs)


def check_client_version():
    """Checks API client version and raises an error if it's not supported."""

    user_agent = request.headers.get("User-Agent")
    if user_agent is None:
        return

    min_versions = {
        "Wall-E.CLI": (4,),
        "Wall-E.Client": (4,),
    }

    for word in user_agent.split(" "):
        tokens = word.split("/")
        if len(tokens) != 2:
            continue

        name, version = tokens

        min_version = min_versions.get(name)
        if min_version is None:
            continue

        try:
            version = _parse_version_string(version)
        except ValueError:
            continue

        if version < min_version:
            raise BadRequestError(
                "Sorry, your {} version is not supported by the server anymore. Please update your client.", name
            )


class FilterQueryParser:
    def __init__(
        self,
        model,
        enum_fields=tuple(),
        substring_fields=tuple(),
        prefix_fields=(),
        or_fields=(),
        field_aliases=None,
        allowed_modifiers=None,
    ):
        self.pymongo_object = MongoDocument.for_model(model)
        self.enum_fields = set(enum_fields)
        self.substring_fields = set(substring_fields)
        self.prefix_fields = prefix_fields
        self.or_fields = set(or_fields)
        self.field_aliases = field_aliases or {}
        self.allowed_modifiers = allowed_modifiers or {"in", "nin"}

    def parse_query_filter(self, query_args):
        """Creates a list of mongodb query filters from values specified in HTTP query string."""

        query = {}
        listed_or_fields = self.or_fields.intersection(query_args)

        if len(listed_or_fields) > 1:  # at least 2 OR fields must present
            or_field_values = []

            for or_field in listed_or_fields:
                # pop or_field from query_args
                prep_field, prep_value = self._prepare_field_and_value(or_field, query_args.pop(or_field))
                or_field_values.append({self._resolve_field(prep_field): prep_value})

            query["$or"] = or_field_values

        for field, value in query_args.items():
            prep_field, prep_value = self._prepare_field_and_value(field, value)
            if prep_field:
                query[self._resolve_field(prep_field)] = prep_value

        return query

    def _resolve_field(self, field):
        """convert only when we know that parameter is indeed a field"""
        return self.pymongo_object.resolve_field(field)

    def _prepare_field_and_value(self, field, value):
        """return field alias of field and prepared value"""

        field, _, modifier = field.partition("__")
        field_alias = self.field_aliases.get(field, field)

        if modifier and modifier in self.allowed_modifiers and field in self.enum_fields:
            modifier = "$" + modifier

            return field_alias, {modifier: value}

        elif not modifier:
            if field in self.substring_fields:
                return field_alias, {"$in": pattern_values(value, by_prefix=False)}

            elif field in self.prefix_fields:
                return field_alias, {"$in": pattern_values(value, by_prefix=True)}

            elif field in self.enum_fields:
                return field_alias, {"$in": value}

        return None, None


class SortQueryParser:
    def __init__(self, model, aliases=None):
        self.pymongo_object = MongoDocument.for_model(model)
        self.aliases = aliases or {}

    def parse_sort_query(self, fields):
        sort_params = []
        exist_fields = set()

        for field_with_param in fields:
            field, order = self._parse_field(field_with_param)
            for alias in self._find_aliases(field):
                pymongo_field = self.pymongo_object.resolve_field(alias)
                if pymongo_field not in exist_fields:
                    exist_fields.add(pymongo_field)
                    sort_params.append((pymongo_field, order))

        return sort_params

    def _find_aliases(self, field):
        alias = self.aliases.get(field, field)
        if isinstance(alias, str):
            return [alias]
        return alias

    @staticmethod
    def _parse_field(field):
        parsed = field.split(":")
        pymongo_order = pymongo.ASCENDING

        if len(parsed) > 1 and parsed[1] == "desc":
            pymongo_order = pymongo.DESCENDING

        return parsed[0], pymongo_order


def expand_query_params(params, allowed_modifiers=None):
    allowed_modifiers = allowed_modifiers or ("in", "nin")

    def expand_field(field):
        return itertools.chain([field], ("{}__{}".format(field, modifier) for modifier in allowed_modifiers))

    return {expanded_field: schema for field, schema in params.items() for expanded_field in expand_field(field)}


def get_object_result(model, query, query_args, postprocessor=None):
    """Returns result of querying a single object.
    `query` must be a pymongo (mongodb) query, not MongoEngine.
    """

    document = MongoDocument.for_model(model)

    query, requested_fields, query_fields, postprocessor = _get_query_params(
        model, document, query, query_args, postprocessor
    )

    obj = model.get_collection().find_one(query, query_fields)
    if obj is None:
        return

    if postprocessor is not None:
        return postprocessor.process_one(document(obj), requested_fields)
    else:
        return document(obj).to_api_obj(requested_fields)


def get_simple_query_result(obj_class, query_args, filter_kwargs=None, postprocessor=None):
    """Returns simple query result (without paging)."""

    fields = query_args.get("fields")
    cursor = obj_class.objects(**(filter_kwargs or {})).only(*obj_class.api_query_fields(fields)).order_by("id")

    if postprocessor is not None:
        objects = postprocessor.process(cursor, fields)
    else:
        objects = [obj.to_api_obj(fields) for obj in gevent_idle_iter(cursor)]

    return api_response({"result": objects})


def get_query_result(
    model, queries, cursor_field, query_args, order_by=None, cursor_only=False, reverse=False, postprocessor=None
):
    """Returns query result according to specified paging mode.
    `queries` is a list of search criteria which must all be met for the document to appear in the result.
    `queries` may be None to express a query that returns an empty result.
    `queries` are the pymongo (mongodb) queries, not MongoEngine.
    """

    if "cursor" in query_args and "offset" in query_args:
        raise RequestValidationError("'offset' parameter is no allowed with 'cursor' parameter.")

    cursor = query_args.get("cursor")
    offset = query_args.get("offset")
    limit = query_args.get("limit", _DEFAULT_PAGE_SIZE)
    cursor_mode = cursor_only or cursor is not None or (offset is None and not order_by)

    if cursor_mode and order_by:
        raise RequestValidationError("'order by' parameter is no allowed in cursor mode.")

    response = {}

    if queries is None:
        response["total"] = 0
        response["result"] = []
        return response

    document = MongoDocument.for_model(model)
    query, requested_fields, query_fields, postprocessor = _get_query_params(
        model, document, queries, query_args, postprocessor, cursor_mode, cursor_field
    )

    if cursor_mode and cursor is not None:
        query[cursor_field.db_field] = {("$lte" if reverse else "$gte"): cursor}

    if not order_by:
        order_by = [(cursor_field.db_field, pymongo.DESCENDING if reverse else pymongo.ASCENDING)]
    collection = model.get_collection(read_preference=SECONDARY_LOCAL_DC_PREFERRED)
    db_cursor = collection.find(query, query_fields, sort=order_by)

    if not cursor_mode or cursor is None or cursor == 0:
        response["total"] = collection.find(query).count()

    if cursor_mode:
        result, next_cursor = get_result_next_from_cursor(document, db_cursor, cursor_field, limit)
    else:
        offset = offset or 0
        limit = min(limit, response["total"] - offset)

        if offset > 0:
            db_cursor = db_cursor.skip(offset)

        if limit == 0:
            result, next_cursor = (), None
        else:
            result, next_cursor = (document(row) for row in db_cursor.limit(limit)), None

    if postprocessor is not None:
        response["result"] = postprocessor.process(result, requested_fields)
    else:
        response["result"] = [obj.to_api_object_shallow(requested_fields) for obj in gevent_idle_iter(result)]

    if cursor_mode and next_cursor is not None:
        response["next_cursor"] = next_cursor

    return response


def get_result_next_from_cursor(document, db_cursor, cursor_field, limit):
    if limit == 0:
        # mongo treats limit=0 as 'no limit'. But we receive user input for this value and we do not want
        # 'no limit' at all. We also have been using 0 for "empty list" since day one,
        # as this was the MongoEngine's behaviour.
        return [], None
    counter = 0
    result = []
    for row in db_cursor.limit(limit + 1):
        obj = document(row)
        if counter >= limit:
            return result, obj.to_api_obj([cursor_field.name]).get(cursor_field.name, None)
        result.append(obj)
        counter += 1
    return result, None


def _get_query_params(model, obj_class, query, query_args, postprocessor=None, cursor_mode=False, cursor_field=None):
    """Return:
    pymongo query object,
    list of requested fields,
    list of document fields to fetch,
    nullify postprocessor if it's not needed.
    """
    if isinstance(query, list):
        # filter out empty clauses
        query = list(filter(None, query))
        if not query:
            # match-all query
            query = {}
        elif len(query) == 1:
            # do not "$and" one single clause
            query = query[0]
        else:
            query = {"$and": query}

    fields = query_args.get("fields")
    if fields and cursor_mode and cursor_field is not None:
        # make sure cursor fields is always returned
        fields.append(cursor_field.name)

    requested_fields = get_requested_fields(model, fields)
    query_fields = obj_class.fields_to_query_fields(model.api_query_fields(requested_fields))

    if postprocessor is not None:
        if postprocessor.is_needed(requested_fields):
            query_fields.extend(postprocessor.extra_db_fields)
        else:
            postprocessor = None

    requested_fields = obj_class.hash_fields(tuple(sorted(requested_fields)))
    return query, requested_fields, query_fields, postprocessor


def validate_project_filter(query_args):
    project_filter = query_args.get("project")
    if not project_filter:
        return

    existing_projects = {project.id for project in Project.objects(id__in=project_filter).only("id")}
    invalid_projects = set(project_filter) - existing_projects

    if invalid_projects:
        raise BadRequestError("Invalid project ID: {}.", ", ".join(invalid_projects))


def validate_reason(reason):
    if reason:
        reason = reason.strip()

    if not reason:
        return None

    max_len = 1000
    if len(reason) > max_len:
        raise RequestValidationError("The reason string exceeds the maximum length limit ({}).", max_len)

    return reason


def validate_user_string(user_string, name):
    for char in user_string:
        if ord(char) < 32:
            raise RequestValidationError(
                "The {} string contains invalid characters (no new line or other control characters are allowed).", name
            )

    return re.sub(r"\s+", " ", user_string).strip()


def get_api_error_headers(error):
    """Returns headers that must be set for the specified ApiError."""

    headers = error.headers

    # Don't know how to do it more correctly without specifying host name in config or adding Flask dependency to the
    # exception.
    if error.http_code == http.client.UNAUTHORIZED:
        headers = headers.copy()
        headers.setdefault("WWW-Authenticate", 'OAuth realm="{}"'.format(request.host_url))

    return headers


@exception_handler
def api_error_handler(error):
    log.warning("API error: %s", error)
    api_logger().log(
        error_type="api", error_class=error.__class__.__name__, error=error, walle_action="api_error_handler"
    )
    return str(error), error.http_code, get_api_error_headers(error)


@exception_handler
def internal_error_handler(error):
    log.exception("Internal error occurred")
    api_logger().log(
        api_error="exception", error_type="internal", walle_action="api_internal_error_handler", **exception_context()
    )
    return "Internal error occurred: {}".format(error), http.client.INTERNAL_SERVER_ERROR


def _generate_api_schema(
    methods, params, schema, with_sudo, with_ignore_maintenance, with_fields, with_paging, with_reason, with_sort
):

    if params is None and (
        with_sudo
        or with_ignore_maintenance
        or with_fields is not None
        or with_paging is not None
        or with_sort is not None
    ):
        params = {}

    if schema is None and with_reason:
        schema = {
            "type": "object",
            "properties": {},
            "additionalProperties": False,
        }

    if schema is not None and not set(methods).intersection(_DATA_METHODS):
        raise LogicalError()

    if with_sudo:
        assert "sudo" not in params
        params["sudo"] = {"type": "boolean", "description": "Use admin privileges to process the request"}

    if with_ignore_maintenance:
        if "ignore_maintenance" in params:
            raise LogicalError()

        params["ignore_maintenance"] = {
            "type": "boolean",
            "description": "Forcefully submit operation ignoring host maintenance status, default is false",
        }

    if with_fields is not None:
        assert "fields" not in params
        params["fields"] = {
            "type": "array",
            "items": {
                "type": "string",
                "description": "Object fields to return. Available fields: {}.".format(
                    ", ".join(sorted(with_fields.api_fields))
                ),
            },
        }

    if with_paging is not None:
        assert "cursor" not in params and "limit" not in params and "offset" not in params

        if with_paging.get("cursor"):
            params.update({"cursor": with_paging["cursor"]})

        if not with_paging.get("cursor_only"):
            params.update(
                {"offset": {"type": "integer", "minimum": 0, "description": "An offset to return $limit entries from"}}
            )

        params.update(
            {
                "limit": {
                    "type": "integer",
                    "minimum": 0,
                    "maximum": with_paging.get("max_limit", _FALLBACK_MAX_PAGE_SIZE),
                    "default": with_paging.get("default_limit", _DEFAULT_PAGE_SIZE),
                    "description": "Maximum number of returned entries per page (page size)",
                },
            }
        )

    if with_sort is not None:
        params.update(
            {
                "sort-by": {
                    "type": "array",
                    "description": with_sort["description"],
                    "items": {
                        "type": "string",
                        "pattern": r"^({})(\:(asc|desc))?$".format("|".join(with_sort["fields"])),
                    },
                }
            }
        )

    if with_reason:
        if "reason" in schema["properties"]:
            raise LogicalError()

        schema["properties"]["reason"] = {"type": "string", "description": "An optional reason string"}

    return params, schema


def _authenticate(
    authenticate, oauth_client_id, iam_permissions: tp.Optional[iam.BaseApiIamPermission], query_args, request_obj
):
    if has_iam():
        return iam.check(_get_iam_permissions(iam_permissions), query_args, request_obj)
    elif authenticate:
        issuer, session_id = blackbox.authenticate(oauth_client_id)

        if session_id is not None and request.method in ("POST", "PUT", "DELETE"):
            csrf.check_csrf_token(session_id, request)
        return issuer


def _log_request(issuer_log_prefix, request_url, request_json):
    log.info(
        "%sAPI request: %s %s %s",
        issuer_log_prefix,
        request.method,
        ellipsis_string(request_url, 10000),
        "" if request_json is None else ellipsis_string(request.get_data(as_text=True), 10000),
    )


def _parse_request(
    params,
    schema,
    params_validator,
    body_validator,
    request_url,
    with_reason,
    allowed_tvm_sources_aliases_fn,
    allowed_mime_type,
    form_processor,
):
    handler_kwargs = {}
    if allowed_tvm_sources_aliases_fn is not None:
        _check_tvm_service_ticket(allowed_tvm_sources_aliases_fn())

    data_method = request.method in _DATA_METHODS
    if request.mimetype == "application/json":
        request_json = request.get_json(silent=True) if data_method else None
    elif form_processor is not None:
        request_json = form_processor(request.form)
    else:
        request_json = request.form

    query_args = _validate_query_params(params, params_validator)
    if params is not None:
        handler_kwargs["query_args"] = query_args

    if data_method:
        if request.method == "DELETE" and not request.mimetype:
            request_json = None
        elif request.mimetype == allowed_mime_type:
            if request_json is None:
                raise RequestValidationError("Request data is empty")

            if schema is not None:
                try:
                    body_validator(request_json)
                except fastjsonschema.JsonSchemaValueException as e:
                    _handle_specific_schema_validation_error(e, request_json, request_url)
                except fastjsonschema.JsonSchemaException as e:
                    _handle_unknown_schema_validation_error(e, request_json, request_url)
        else:
            raise BadRequestError("Got wrong request MIME type {} (must be {})", request.mimetype, allowed_mime_type)

        if schema is not None:
            if request_json is None:
                request_obj, options_dict = None, {}
            else:
                request_obj = options_dict = request_json.copy()

            if with_reason:
                handler_kwargs["reason"] = validate_reason(options_dict.pop("reason", None))

            handler_kwargs["request"] = request_obj

    return handler_kwargs


def _check_tvm_service_ticket(allowed_tvm_sources_aliases):
    tvm_ticket = request.headers.get("X-Ya-Service-Ticket")
    if tvm_ticket is None:
        raise BadRequestError("TVM ticket is missing in request")

    try:
        check_ticket(tvm_ticket, allowed_tvm_sources_aliases)
    except (TvmSourceIsNotAllowed, TvmUnknownAppId) as e:
        raise UnauthenticatedError(str(e))


def _validate_query_params(schema, validator):
    if "strict" in request.args:
        strict = _parse_query_param("strict", request.args.getlist("strict"), {"type": "boolean"})
    else:
        strict = _DEFAULT_STRICT_API

    if strict:
        unknown_args = set(request.args) - set(schema or {}) - {"strict"}
        if unknown_args:
            raise RequestValidationError("Invalid parameter: {}.", unknown_args.pop())

    if schema is None or not request.args:
        return {}

    query_args = {}

    for arg_name, arg_scheme in schema.items():
        if arg_name not in request.args:
            continue

        query_args[arg_name] = _parse_query_param(arg_name, request.args.getlist(arg_name), arg_scheme)

    try:
        validator(query_args)
    except fastjsonschema.JsonSchemaException as e:
        raise RequestValidationError(str(e).split("\n")[0])

    return query_args


def _parse_query_param(name, list_value, scheme):
    value = list_value[0]

    if "type" not in scheme:
        return value

    if scheme["type"] == "integer":
        try:
            return int(value)
        except ValueError:
            raise RequestValidationError("Invalid {} parameter value: {}.", name, value)
    elif scheme["type"] == "number":
        try:
            return float(value)
        except ValueError:
            raise RequestValidationError("Invalid {} parameter value: {}.", name, value)
    elif scheme["type"] == "boolean":
        stripped_value = value.strip().lower()

        if stripped_value in ("0", "no", "false"):
            return False
        elif stripped_value in ("1", "yes", "true"):
            return True
        else:
            raise RequestValidationError("Invalid {} parameter value: {}.", name, value)
    elif scheme["type"] == "array":
        array_value = []

        for value in list_value:
            for item in value.split(","):
                item = item.strip()
                if item:
                    array_value.append(_parse_query_param(name, (item,), scheme["items"]))

        return array_value
    else:
        return value


def pattern_values(values, by_prefix=False):
    if isinstance(values, str):
        values = [values]
    return [patterns.parse_pattern(v, by_prefix=by_prefix) for v in values]


def _parse_version_string(value):
    try:
        return tuple(int(version) for version in value.split("."))
    except ValueError:
        raise ValueError("Invalid version string.")


def _handle_specific_schema_validation_error(exc, request_json, url):
    error_message = "error: {}; definition: {}, value: {}".format(str(exc).split("\n")[0], exc.definition, exc.value)
    log.error(
        "%s: Request json doesn't match schema: %s, json: %s, definition: %s",
        url,
        error_message,
        request_json,
        exc.definition,
    )
    raise RequestValidationError(error_message)


def _handle_unknown_schema_validation_error(exc, request_json, url):
    error = str(exc).split("\n")[0]
    log.error("%s: Request json doesn't match schema: %s, json: %s", url, error, request_json)
    raise RequestValidationError(error)


def read_only():
    allowed_methods = ["OPTIONS", "GET"]
    if request.method not in allowed_methods:
        raise MethodNotAllowedError("Method {} is temporarily disabled for this endpoint.".format(request.method))
