import dataclasses
import typing

import cachetools

from walle.clients import ok, startrek
from walle.models import timestamp


class ApprovalTicketResolution:
    REFUSAL = "refusal"
    SUCCESSFUL = "successful"


# WALLE-4057 Dataclasses below must be hashable to cache StarTrek and Ok responses.


@dataclasses.dataclass(eq=True, frozen=True)
class CreateStartrekTicketRequest:
    """https://wiki.yandex-team.ru/tracker/api/issues/create/"""

    queue: str
    type: str
    summary: str
    description: str
    tags: typing.Tuple[str, ...] = dataclasses.field(default_factory=tuple)
    parent: str = None


@dataclasses.dataclass(eq=True, frozen=True)
class CreateOkApprovementRequest:
    """https://wiki.yandex-team.ru/Intranet/OK/private-api/"""

    ticket_key: str
    text: str
    author: str
    groups: typing.Tuple[str, ...] = dataclasses.field(default_factory=tuple)
    approvers: typing.Tuple[str, ...] = dataclasses.field(default_factory=tuple)


@dataclasses.dataclass(eq=True, frozen=True)
class CreateStartrekCommentRequest:
    """https://wiki.yandex-team.ru/tracker/api/issues/comments/create/"""

    issue_id: str
    text: str
    summonees: typing.Tuple[str, ...] = dataclasses.field(default_factory=tuple)


class ApproveClient:
    """
    A collection of wrappers around Startrek and Ok clients that simplifies
    creating and managing Ok approvements.
    """

    TStartrekTicketKey = str
    TOkUuid = str
    TStartrekCommentId = str
    TOkApprovementStatus = str
    TOkApprovementResolution = str

    # WALLE-4057 Cache StarTrek and Ok responses.
    DEFAULT_RATELIMIT_CACHE_MAXSIZE: int = 64
    DEFAULT_RATELIMIT_CACHE_TTL: int = 900

    def __init__(self, startrek_client: startrek.StartrekClient = None, ok_client: ok.OkClient = None):
        self._startrek_client = startrek_client
        self._ok_client = ok_client

    @cachetools.cached(
        cachetools.TTLCache(maxsize=DEFAULT_RATELIMIT_CACHE_MAXSIZE, ttl=DEFAULT_RATELIMIT_CACHE_TTL, timer=timestamp)
    )
    def create_startrek_ticket(self, request: CreateStartrekTicketRequest) -> TStartrekTicketKey:
        return self._startrek_client.create_issue(issue_params=dataclasses.asdict(request))["key"]

    @cachetools.cached(
        cachetools.TTLCache(maxsize=DEFAULT_RATELIMIT_CACHE_MAXSIZE, ttl=DEFAULT_RATELIMIT_CACHE_TTL, timer=timestamp)
    )
    def create_ok_approvement(self, request: CreateOkApprovementRequest) -> TOkUuid:
        return self._ok_client.create_approvement(
            ticket_key=request.ticket_key,
            text=request.text,
            author=request.author,
            approvers=request.approvers,
            groups=request.groups,
        ).uuid

    def edit_approvers(self, uuid, approvers):
        return self._ok_client.edit_approvers(uuid=uuid, approvers=approvers)

    @cachetools.cached(
        cachetools.TTLCache(maxsize=DEFAULT_RATELIMIT_CACHE_MAXSIZE, ttl=DEFAULT_RATELIMIT_CACHE_TTL, timer=timestamp)
    )
    def create_startrek_comment(self, request: CreateStartrekCommentRequest) -> TStartrekCommentId:
        return self._startrek_client.add_comment(
            issue_id=request.issue_id, text=request.text, summonees=list(request.summonees)
        )["id"]

    def get_ok_approvement_status_resolution(
        self, ok_uuid: TOkUuid
    ) -> (ok.ApprovementStatus, ok.ApprovementResolution):
        approvement = self._ok_client.get_approvement(uuid=ok_uuid)
        return approvement.status, approvement.resolution

    def close_startrek_ticket(
        self,
        startrek_ticket_key: TStartrekTicketKey,
        resolution: ApprovalTicketResolution = ApprovalTicketResolution.SUCCESSFUL,
    ):
        ticket_status = self._startrek_client.get_issue(issue_id=startrek_ticket_key)["status"]["key"]
        if ticket_status != startrek.TicketStatus.CLOSED:
            self._startrek_client.close_issue(issue_id=startrek_ticket_key, transition="closed", resolution=resolution)

    def close_ok_approvement(self, ok_uuid: TOkUuid):
        approvement = self._ok_client.get_approvement(uuid=ok_uuid)
        if approvement.status != ok.ApprovementStatus.CLOSED:
            self._ok_client.close_approvement(uuid=ok_uuid)
