"""Simple in-memory cache."""

import sys
import time
from functools import wraps
from threading import RLock

import cachetools


def cached_with_exceptions(
    cache_values=None,
    cache_errors=None,
    value_ttl=None,
    error_ttl=None,
    value_key=None,
    error_key=None,
    lock=None,
    timer=time.time,
):
    """A decorator that caches returned function value with the specified timeout."""

    if lock is None:
        if cache_values is not None or cache_errors is not None:
            raise TypeError("The 'cached_with_exceptions' decorator needs lock if used custom cache instances.")
        lock = RLock()

    if cache_values is None:
        if value_ttl is None:
            raise TypeError("The 'cached_with_exceptions' decorator needs either cache instance or cache ttl.")
        cache_values = cachetools.TTLCache(maxsize=1, ttl=value_ttl, timer=timer)

    if cache_errors is None and error_ttl is not None:
        cache_errors = cachetools.TTLCache(maxsize=1, ttl=error_ttl, timer=timer)

    value_key = 'data' if value_key is None else value_key
    error_key = 'error' if error_key is None else error_key

    def cache_clear():
        with lock:
            cache_values.pop(value_key, None)
            if cache_errors is not None:
                cache_errors.pop(error_key, None)

    def decorator(func):
        @wraps(func)
        def decorated():
            with lock:
                data = cache_values.get(value_key, None)
                if data is not None:
                    return data[0]

                if cache_errors is not None:
                    error_type, error_value = cache_errors.get(error_key, (None, None))
                    if error_type is not None:
                        raise error_value.with_traceback(None)

                try:
                    value = func()
                except Exception:
                    if cache_errors is not None:
                        error_type, error_value, traceback = sys.exc_info()
                        cache_errors[error_key] = (error_type, error_value)
                    raise
                else:
                    cache_values[value_key] = (value,)
                    return value

        # useful for tests
        decorated.cache_clear = cache_clear

        return decorated

    return decorator
