import logging
from datetime import datetime

import six
from OpenSSL import crypto

from sepelib.core import config
from walle.clients import juggler
from walle.models import timestamp

log = logging.getLogger(__name__)


def notify_if_cert_expires_soon(cert_name, cert_path_option, event_name, days_to_warn):
    log.info("Checking %s certificate...", cert_name)
    days_left = _days_till_expiration(config.get_value(cert_path_option))

    event_name = "wall-e.{}".format(event_name)
    message = "{} certificate expires in {} days".format(cert_name, days_left)

    if days_left < days_to_warn:
        log.error("%s certificate expires in %d days", cert_name, days_left)
        status = juggler.JugglerCheckStatus.CRIT
    else:
        log.debug("%s certificate expires in %d days", cert_name, days_left)
        status = juggler.JugglerCheckStatus.OK

    juggler.send_event(event_name, status=status, message=message)


def _days_till_expiration(cert_path):
    expiration_dt = datetime.strptime(six.ensure_str(_get_cert_expiration(cert_path)), "%Y%m%d%H%M%SZ")
    cur_dt = datetime.fromtimestamp(timestamp())
    return (expiration_dt - cur_dt).days


def _get_cert_expiration(filename):
    with open(filename) as cert:
        cert = crypto.load_certificate(crypto.FILETYPE_PEM, cert.read())
    return cert.get_notAfter()
