"""Some utils for runtime cloud."""

import os
import socket

from sepelib.core import config
from sepelib.core.exceptions import Error

RTC_PROD_SYSTEM_NAME = "walle"


def get_bsconfig_itag(tag_name):
    env_name = "BSCONFIG_ITAGS"
    tag_prefix = "a_{}_".format(tag_name)

    for itag in os.environ.get(env_name, "").split(" "):
        if itag.startswith(tag_prefix):
            return itag[len(tag_prefix) :]

    raise Error("{} environment variable doesn't contains {}* tag.", env_name, tag_prefix)


def get_instance_role():
    """Return instance role based on itype."""
    try:
        return get_bsconfig_itag("itype")
    except Error as e:
        raise Error("Unable to determine instance role: {}", e)


def get_ctype():
    """Return instance ctype (prod/test)."""
    try:
        return get_bsconfig_itag("ctype")
    except Error as e:
        raise Error("Unable to determine instance ctype: {}", e)


def get_tier():
    """Return instance tier id (1, 2, 3, etc)."""
    try:
        return get_bsconfig_itag("tier")
    except Error as e:
        raise Error("Unable to determine instance tier: {}", e)


def get_process_identifier():
    """Return unique process identifier based on instance port."""
    env_name = "BSCONFIG_IPORT"
    if os.environ.get(env_name):
        return '{}-{}-{}'.format(socket.getfqdn(), os.environ[env_name], os.getpid())
    else:
        return '{}-{}'.format(socket.getfqdn(), os.getpid())


def get_instance_iport():
    env_name = "BSCONFIG_IPORT"
    return os.environ.get(env_name)


def get_config_path():
    try:
        conf_name = get_bsconfig_itag("ctype")
    except Error:
        conf_name = "conf"

    instance_path = os.environ.get("BSCONFIG_IDIR", "")
    return os.path.join(instance_path, "conf", "walle.{name}.yaml".format(name=conf_name))


def get_system_suffix():
    system_name = config.get_value("tier_configuration.system_name")
    if system_name == RTC_PROD_SYSTEM_NAME:
        return ""
    else:
        return f".{system_name}"


def get_juggler_event_key(juggler_service_name, host_name=None):
    event_key = {"service": f"{juggler_service_name}{get_system_suffix()}"}

    if host_name:
        event_key["host"] = host_name
    else:
        event_key["host"] = socket.getfqdn()

        instance_port = get_instance_iport()
        if instance_port:
            event_key["instance"] = instance_port

    return event_key
