from flask import request


class BaseEncoder:
    content_encoding = None
    skip_content_type = [
        'image/*',
        'video/*',
        'application/pdf',
    ]

    def __init__(self, minimum_size=4096, skip_content_type=None):
        self.minimum_size = minimum_size
        if skip_content_type is not None:
            self.skip_content_type = skip_content_type

    def after_request(self, response):
        if self.should_compress(response):
            # If we are streaming file, we need to freeze it (read in memory)
            # Better way would be to compress stream without loading everything into
            # memory, but seems like it would require some dirty fiddling
            # into Response object to make it not streamed.
            # Should work for now
            if response.is_streamed:
                response.freeze()
            response.data = self.compress(response.data, content_type=response.headers.get('content-type'))
            response.headers['Content-Encoding'] = self.content_encoding
            response.headers['Content-Length'] = len(response.data)
        return response

    def should_compress(self, response):
        content_type = response.headers.get('content-type', None)
        content_len = response.headers.get('content-length', None)
        if content_len is None:
            return False
        return all(
            [
                self.content_encoding.lower() in request.headers.get('accept-encoding', '').lower(),
                int(content_len) >= self.minimum_size,
                not any(
                    [
                        content_type in self.skip_content_type,
                        content_type.split('/')[0] + '/*' in self.skip_content_type,
                    ]
                ),
            ]
        )

    def compress(self, data, content_type=None):
        raise NotImplementedError
