import itertools
import json
import logging
from collections import defaultdict
from copy import deepcopy

import attr

from sepelib.core.exceptions import Error
from walle.clients import bot, deploy

log = logging.getLogger(__name__)

# These sections of config are prohibited to change in our configs
# They are inherited, so by some very strange design we can delete them from our config
DENY_SECTIONS = ["ipxe", "pxe", "infrapackages", "repositories", "base"]


def convert_gb_to_gib(x):
    return int(x * 1000 * 1000 * 1000 / 1024 / 1024 / 1024)


class DeployConfigStrategy:
    """:returns dict of modified stage parameters"""

    def generate(self, host, deploy_config_name):
        raise NotImplementedError()


class NoSystemDiskError(Error):
    def __init__(self, disk_conf, msg):
        super().__init__("Cannot determine system disk for disk conf {}: {}".format(disk_conf, msg))


class UnsupportedDiskConfError(Error):
    def __init__(self, disk_conf, msg):
        super().__init__("Cannot generate setup config for disk conf {}: {}".format(disk_conf, msg))


class DeployConfigPolicyNotFound(Error):
    def __init__(self, policy_name):
        msg = "Could not find deploy config policy {} (available policies are {})".format(
            policy_name, DeployConfigPolicies.get_all_names()
        )
        super().__init__(msg)


class PassthroughConfigStrategy(DeployConfigStrategy):
    description = """Default deploy config policy, doesn't change any config params in any way"""

    def generate(self, host, deploy_config_name):
        return {}


@attr.s
class ModifierContext:
    disk_conf = attr.ib()
    sys_disk_kind = attr.ib()
    partitions = attr.ib()
    volume_groups = attr.ib()


@attr.s
class PartitionDescr:
    name = attr.ib()
    disk_info = attr.ib()
    system_disk = attr.ib()
    block_kind = attr.ib()
    block_name = attr.ib()


@attr.s
class VolumeGroupDescr:
    name = attr.ib()
    system_disk = attr.ib()
    block_kind = attr.ib()
    shared_size = attr.ib()
    stripes = attr.ib(default=None)


@attr.s
class FilesystemDescr:
    name = attr.ib()
    mountpoint = attr.ib(default=None)
    size = attr.ib(default="*")
    fs = attr.ib(default="ext4")
    fs_opts = attr.ib(default="-b 4096")
    # for mount options: None means "default" (whatever Setup thinks default is), empty string means no options
    mount_options = attr.ib(default=None)


class SetupConfigFactory:
    def __init__(self, host, deploy_config_name, destroy_all_partition_tables=False):
        self._host = host
        deploy_provider = deploy.get_deploy_provider(self._host.get_eine_box())
        self._original_deploy_config = deploy.get_deploy_config(deploy_provider, deploy_config_name)
        self._updated_deploy_config = deepcopy(self._original_deploy_config)
        self._disk_conf = bot.get_host_disk_configuration(self._host.inv)
        self._sys_disk_kind = self._determine_system_disk(self._disk_conf)
        self._destroy_all_partition_tables = destroy_all_partition_tables

    def _determine_system_disk(self, disk_conf):
        if disk_conf.hdds:
            return BlockKind.HDD
        elif disk_conf.ssds:
            return BlockKind.SSD
        elif disk_conf.nvmes:
            return BlockKind.NVME
        else:
            log.error("%s: There are no disks in host (disk conf %s)", self._host, disk_conf)
            raise NoSystemDiskError(disk_conf, "there are no disks in host")

    def _remove_deny_sections(self):
        for section_name in DENY_SECTIONS:
            if section_name in self._updated_deploy_config:
                del self._updated_deploy_config[section_name]

    def _move_repositories_to_extra(self):
        """We cannot send data in repositories section (it is in DENY_SECTIONS), so move the contents to user section"""
        repos = self._updated_deploy_config.pop("repositories", None)
        if repos:
            self._updated_deploy_config["repositories_extras"] = repos

    def _add_erase_partitions_option(self):
        self._updated_deploy_config.setdefault("system", {})["destroy_all_partition_tables"] = True

    def modify_config(self, modifiers):
        context = ModifierContext(
            disk_conf=self._disk_conf, sys_disk_kind=self._sys_disk_kind, partitions=[], volume_groups=[]
        )
        for gen in modifiers:
            self._updated_deploy_config.update(gen(context))

        if self._destroy_all_partition_tables:
            self._add_erase_partitions_option()
        self._move_repositories_to_extra()
        self._remove_deny_sections()

    def get_stage_params(self):
        return {
            "config_name": deploy.DEPLOY_CONFIG_EXTERNAL,
            "config_content_json": json.dumps(self._updated_deploy_config, indent=2),
        }


class BlockKind:
    HDD = "hdd"
    SSD = "ssd"
    NVME = "nvme"

    ALL = (HDD, SSD, NVME)
    ROTATIONAL = (HDD,)
    NON_ROTATIONAL = (SSD, NVME)

    disk_kind_map = {
        bot.DiskKind.HDD: HDD,
        bot.DiskKind.SSD: SSD,
        bot.DiskKind.NVME: NVME,
    }

    @classmethod
    def from_disk_kind(cls, disk_kind):
        return cls.disk_kind_map[disk_kind]


def _iterate_over_disk_conf(disk_conf):
    return itertools.chain(enumerate(disk_conf.hdds), enumerate(disk_conf.ssds), enumerate(disk_conf.nvmes))


def _get_stripes(stripe_partitions, block_kind):
    return len(stripe_partitions) if block_kind in (BlockKind.HDD, BlockKind.SSD) else None


class DiskManagerConfigStrategy(DeployConfigStrategy):
    description = """This policy dynamically modifies LUI config to support Diskmanager
    Destroys all existing partitions
    First partition is for boot loader, everything else is used by LVM
    """

    fs_descrs = [
        FilesystemDescr(name="root", mountpoint="/", size="40G", mount_options="barrier=1,noatime,lazytime"),
        FilesystemDescr(
            name="home", mountpoint="/home", size="6G", mount_options="barrier=1,noatime,lazytime,nosuid,nodev"
        ),
        FilesystemDescr(
            name="place", mountpoint="/place", size="500G", mount_options="barrier=1,noatime,lazytime,nosuid,nodev"
        ),
    ]

    def generate(self, host, deploy_config_name):
        factory = SetupConfigFactory(host, deploy_config_name)
        factory.modify_config(
            [
                self._gen_parted_section,
                self._gen_md_section,
                self._gen_lvm_section,
                self._gen_fs_section,
            ]
        )
        return factory.get_stage_params()

    def _gen_parted_section(self, context):
        parted = {}

        for idx, disk_info in _iterate_over_disk_conf(context.disk_conf):
            block_kind = BlockKind.from_disk_kind(disk_info.kind)
            block_name = "disk_{}".format(disk_info.serial_number)
            parted[block_name] = ["grub", "*"]
            system_disk = idx == 0 and block_kind == context.sys_disk_kind
            if system_disk and disk_info.from_node_storage:
                raise NoSystemDiskError(context.disk_conf, "system disk located on node storage")
            context.partitions.append(
                PartitionDescr(
                    name="{}_2".format(block_name),
                    disk_info=disk_info,
                    system_disk=system_disk,
                    block_kind=block_kind,
                    block_name=block_name,
                )
            )

        return {"parted": parted}

    def _gen_md_section(self, context):
        # we don't need md if LVM is used
        return {"md": {}}

    def _gen_lvm_section(self, context):
        lvm_groups = {}
        for gen in [self._gen_volume_group, self._gen_logical_volumes]:
            lvm_groups.update(gen(context))

        return {"lvm": lvm_groups}

    def _gen_volume_group(self, context):
        vgs = {}
        volume_group_iter = ("vg{}".format(idx) for idx in itertools.count())
        for idx, partition in enumerate(context.partitions):
            volume_group_name = next(volume_group_iter)
            vgs[volume_group_name] = [
                # NB: setup doesn't support "-" in volume group name because they become "--" in /dev/mapper
                "vg_{}_{}".format(partition.block_kind, partition.disk_info.instance_number),
                partition.name,
                "--addtag diskman=true",
            ]
            if volume_group_name != "vg0" and partition.system_disk:
                raise UnsupportedDiskConfError(context.disk_conf, "only one system disk can exists")
        return vgs

    def _gen_logical_volumes(self, context):
        lvs = {}

        for lv_idx, fs_descr in enumerate(self.fs_descrs):
            lvs["lv{}".format(lv_idx)] = [fs_descr.name, fs_descr.size, "vg0", "--addtag diskman.sys=true"]

        return lvs

    def _gen_fs_section(self, context):
        partitions = {}
        for part_idx, fs_descr in enumerate(self.fs_descrs):
            partitions["lv{}".format(part_idx)] = [
                fs_descr.fs,
                fs_descr.fs_opts,
                fs_descr.mountpoint,
                fs_descr.mount_options,
            ]
        return {"fs": partitions}


class BaseSharedConfigStrategy(DeployConfigStrategy):
    sys_fs_descrs = [
        FilesystemDescr(name="root", mountpoint="/", size="40G", mount_options="barrier=1,noatime,lazytime"),
        FilesystemDescr(
            name="home", mountpoint="/home", size="6G", mount_options="barrier=1,noatime,lazytime,nosuid,nodev"
        ),
    ]

    place_fs_descr = FilesystemDescr(
        name="place", mountpoint="/place", size="*", mount_options="barrier=1,noatime,lazytime,nosuid,nodev"
    )
    ssd_fs_descr = FilesystemDescr(
        name="ssd", mountpoint="/ssd", size="*", mount_options="barrier=1,noatime,lazytime,nosuid,nodev"
    )

    def generate(self, host, deploy_config_name):
        factory = SetupConfigFactory(host, deploy_config_name)
        factory.modify_config(
            [
                self._gen_parted_section,
                self._gen_md_section,
                self._gen_lvm_section,
            ]
        )
        return factory.get_stage_params()

    def _gen_parted_section(self, context):
        parted = {}

        for idx, disk_info in _iterate_over_disk_conf(context.disk_conf):
            block_kind = BlockKind.from_disk_kind(disk_info.kind)
            block_name = "disk_{}".format(disk_info.serial_number)
            parted[block_name] = ["grub", "*"]
            context.partitions.append(
                PartitionDescr(
                    name="{}_2".format(block_name),
                    disk_info=disk_info,
                    system_disk=False,
                    block_kind=block_kind,
                    block_name=block_name,
                )
            )

        return {"parted": parted}

    def _gen_md_section(self, context):
        # we don't need md if LVM is used
        return {"md": {}}

    def _gen_volume_group(self, context):
        raise NotImplementedError()

    def _gen_logical_volumes(self, context):
        lvs = {}
        filesystems = {}
        mount_points = set()
        logical_volume_iter = ("lv{}".format(idx) for idx in itertools.count())

        def create_logical_volume(fs_descr, volume_group):
            if fs_descr.mountpoint in mount_points:
                return
            mount_points.add(fs_descr.mountpoint)

            logical_volume_name = next(logical_volume_iter)
            options = "--addtag diskman.sys=true"
            if volume_group.stripes and volume_group.stripes >= 2:
                options = "{} --stripes {}".format(options, volume_group.stripes)
            lvs[logical_volume_name] = [fs_descr.name, fs_descr.size, volume_group.name, options]
            filesystems[logical_volume_name] = [
                fs_descr.fs,
                fs_descr.fs_opts,
                fs_descr.mountpoint,
                fs_descr.mount_options,
            ]

        for volume_group in context.volume_groups:
            if volume_group.system_disk:
                for fs_descr in self.sys_fs_descrs:
                    create_logical_volume(fs_descr, volume_group)
            if volume_group.shared_size is not None:
                if volume_group.block_kind == BlockKind.HDD or volume_group.system_disk:
                    fs_descr = self.place_fs_descr
                else:
                    fs_descr = self.ssd_fs_descr
                fs_descr = attr.evolve(fs_descr, size=volume_group.shared_size)
                create_logical_volume(fs_descr, volume_group)

        for fs_descr in self.sys_fs_descrs + [self.place_fs_descr]:
            if fs_descr.mountpoint not in mount_points:
                raise UnsupportedDiskConfError(context.disk_conf, "fs {} not created".format(fs_descr.mountpoint))

        return lvs, filesystems

    def _gen_lvm_section(self, context):
        vgs = self._gen_volume_group(context)
        lvs, fs = self._gen_logical_volumes(context)

        lvm = {}
        lvm.update(vgs)
        lvm.update(lvs)

        return {"lvm": lvm, "fs": fs}


class SharedConfigStrategy(BaseSharedConfigStrategy):
    description = (
        """This policy dynamically modifies LUI config to create shared FS over all disks of same storage class."""
    )

    def _gen_volume_group(self, context):
        partitions_by_block_kind = defaultdict(list)
        for partition in context.partitions:
            partitions_by_block_kind[partition.block_kind].append(partition)

        vgs = {}
        volume_group_iter = ("vg{}".format(idx) for idx in itertools.count())
        for block_kind in BlockKind.ALL:
            if block_kind not in partitions_by_block_kind:
                continue

            partitions = partitions_by_block_kind[block_kind]
            partitions.sort(key=lambda x: x.name)

            diskman_partitions = []
            stripe_partitions = []
            for partition in partitions:
                if partition.disk_info.from_node_storage:
                    # disk from separate disk storage, let's give it away to diskman
                    diskman_partitions.append(partition)
                elif block_kind == context.sys_disk_kind:
                    stripe_partitions.append(partition)
                elif block_kind == BlockKind.SSD and BlockKind.NVME in partitions_by_block_kind:
                    # only one /ssd can exists, let's give disk away to diskman
                    diskman_partitions.append(partition)
                elif block_kind == BlockKind.NVME and context.sys_disk_kind == BlockKind.SSD:
                    # only one /ssd can exists, let's give disk away to diskman
                    diskman_partitions.append(partition)
                else:
                    stripe_partitions.append(partition)

            if stripe_partitions:
                volume_group_name = next(volume_group_iter)
                vgs[volume_group_name] = [block_kind, ",".join([partition.name for partition in stripe_partitions])]
                stripes = _get_stripes(stripe_partitions, block_kind)
                context.volume_groups.append(
                    VolumeGroupDescr(
                        name=volume_group_name,
                        block_kind=block_kind,
                        system_disk=block_kind == context.sys_disk_kind,
                        shared_size="*",
                        stripes=stripes,
                    )
                )
            elif block_kind == context.sys_disk_kind:
                raise NoSystemDiskError(context.disk_conf, "no system disk found")

            for partition in diskman_partitions:
                volume_group_name = next(volume_group_iter)
                vgs[volume_group_name] = [
                    # NB: setup doesn't support "-" in volume group name because they become "--" in /dev/mapper
                    "vg_{}_{}".format(block_kind, partition.disk_info.instance_number),
                    partition.name,
                    "--addtag diskman=true",
                ]

        return vgs


class SharedLvmConfigStrategy(BaseSharedConfigStrategy):
    description = """This policy dynamically modifies LUI config to support LVM at the same time with shared FS
    Destroys all existing partitions
    First partition is for boot loader, first disk of HDD storage class given to the shared FS, everything else is used by LVM
    """

    # NB: this heuristic is needed to keep reasonable amount of space on /place
    capacity_threshold_gb = 600

    def _gen_volume_group(self, context):
        partitions_by_block_kind = defaultdict(list)
        for partition in context.partitions:
            partitions_by_block_kind[partition.block_kind].append(partition)

        vgs = {}
        volume_group_iter = ("vg{}".format(idx) for idx in itertools.count())
        for block_kind in BlockKind.ALL:
            if block_kind not in partitions_by_block_kind:
                continue

            partitions = partitions_by_block_kind[block_kind]
            partitions.sort(key=lambda x: (x.disk_info.capacity_gb, x.name))

            small_partitions = []
            normal_partitions = []
            stripe_partitions = []
            if block_kind == context.sys_disk_kind:
                for partition in partitions:
                    if partition.disk_info.capacity_gb <= self.capacity_threshold_gb:
                        small_partitions.append(partition)
                    else:
                        normal_partitions.append(partition)

                total_capacity_gb = 0
                for partition in small_partitions:
                    if total_capacity_gb <= self.capacity_threshold_gb:
                        stripe_partitions.append(partition)
                    else:
                        normal_partitions.append(partition)
                    total_capacity_gb += partition.disk_info.capacity_gb

                for idx, partition in enumerate(normal_partitions):
                    if idx == 0 and not stripe_partitions:
                        partition.system_disk = True
            else:
                normal_partitions.extend(partitions)

            if stripe_partitions:
                volume_group_name = next(volume_group_iter)
                vgs[volume_group_name] = [block_kind, ",".join([partition.name for partition in stripe_partitions])]
                stripes = _get_stripes(stripe_partitions, block_kind)
                context.volume_groups.append(
                    VolumeGroupDescr(
                        name=volume_group_name,
                        block_kind=block_kind,
                        system_disk=True,
                        shared_size="*",
                        stripes=stripes,
                    )
                )

            for partition in normal_partitions:
                volume_group_name = next(volume_group_iter)
                setup_volume_group = [
                    # NB: setup doesn't support "-" in volume group name because they become "--" in /dev/mapper
                    "vg_{}_{}".format(block_kind, partition.disk_info.instance_number),
                    partition.name,
                    "--addtag diskman=true",
                ]
                vgs[volume_group_name] = setup_volume_group

                shared_size = None
                if partition.system_disk:
                    if partition.disk_info.from_node_storage:
                        raise NoSystemDiskError(context.disk_conf, "system disk located on node storage")
                    else:
                        shared_size = "*"

                context.volume_groups.append(
                    VolumeGroupDescr(
                        name=volume_group_name,
                        block_kind=block_kind,
                        system_disk=partition.system_disk,
                        shared_size=shared_size,
                    )
                )

        return vgs


def _get_yt_mountpoint(block_kind, idx):
    if block_kind == BlockKind.HDD:
        return "/yt/disk{}".format(idx)
    else:
        return "/yt/{}{}".format(block_kind, idx)


class _YTFSDescriptions:
    # Increase YT disks journal size for different kind of media. See HOSTMAN-1233 for details.
    rotational = FilesystemDescr(
        name="yt", fs_opts="-b 4096 -J size=32768", mount_options="nodiscard,barrier=1,noatime,lazytime,nosuid,nodev"
    )
    nonrotational = FilesystemDescr(
        name="yt", fs_opts="-b 4096 -J size=4096", mount_options="nodiscard,barrier=1,noatime,lazytime,nosuid,nodev"
    )

    @classmethod
    def get_desc(cls, block_kind):
        if block_kind in BlockKind.ROTATIONAL:
            return cls.rotational
        else:
            return cls.nonrotational


class YtDedicatedConfigStrategy(DeployConfigStrategy):
    description = """This policy dynamically modifies LUI config to give all disks to YT"""

    fs_descrs = [
        FilesystemDescr(name="ESP", fs="vfat", fs_opts="-S 512 -s 8", mountpoint="/boot/efi", mount_options="noatime"),
        FilesystemDescr(name="boot", mountpoint="/boot", mount_options="barrier=1,noatime,lazytime"),
        FilesystemDescr(name="rootA", mountpoint="/", mount_options="barrier=1,noatime,lazytime"),
        FilesystemDescr(name="rootB", mountpoint="/rootB", mount_options="barrier=1,noatime,lazytime"),
        FilesystemDescr(name="home", mountpoint="/home", mount_options="barrier=1,noatime,lazytime,nosuid,nodev"),
        FilesystemDescr(name="place", mountpoint="/place", mount_options="barrier=1,noatime,lazytime,nosuid,nodev"),
    ]

    def generate(self, host, deploy_config_name):
        factory = SetupConfigFactory(host, deploy_config_name)
        factory.modify_config([self._gen_parted_and_fs_sections, self._gen_md_section, self._gen_lvm_section])
        return factory.get_stage_params()

    def _gen_parted_and_fs_sections(self, context):
        parted = {}
        fs = {}

        for idx, disk_info in _iterate_over_disk_conf(context.disk_conf):
            block_kind = BlockKind.from_disk_kind(disk_info.kind)
            block_name = "disk_{}".format(disk_info.serial_number)

            # system disk
            if idx == 0 and block_kind == context.sys_disk_kind:
                if disk_info.from_node_storage:
                    raise NoSystemDiskError(context.disk_conf, "system disk located on node storage")
                # partition order (mandatory, should be in sync with self.fs_descrs):
                # biosboot, ESP, boot, rootA, rootB, home, place, yt
                parted[block_name] = ["grub", "2g", "2g", "40g", "40g", "5g", "500g", "*"]

                partition_index = 2
                for fs_descr in self.fs_descrs:
                    fs["{}_{}".format(block_name, partition_index)] = [
                        fs_descr.fs,
                        fs_descr.fs_opts,
                        fs_descr.mountpoint,
                        fs_descr.mount_options,
                    ]
                    partition_index += 1

                partition_name = "{}_{}".format(block_name, partition_index)
            else:
                parted[block_name] = ["grub", "*"]
                partition_name = "{}_2".format(block_name)

            mountpoint = _get_yt_mountpoint(block_kind, idx + 1)
            yt_fs_descr = _YTFSDescriptions.get_desc(block_kind)

            fs[partition_name] = [yt_fs_descr.fs, yt_fs_descr.fs_opts, mountpoint, yt_fs_descr.mount_options]

        return {"parted": parted, "fs": fs}

    def _gen_md_section(self, context):
        # we don't need md
        return {"md": {}}

    def _gen_lvm_section(self, context):
        # we don't need lvm
        return {"lvm": {}}


class YtStorageConfigStrategy(YtDedicatedConfigStrategy):
    description = """This policy dynamically modifies LUI config to give all disks from storage node to YT"""

    def _gen_parted_and_fs_sections(self, context):
        parted = {}
        fs = {}
        system_disk_present = False
        for idx, disk_info in _iterate_over_disk_conf(context.disk_conf):
            block_kind = BlockKind.from_disk_kind(disk_info.kind)
            block_name = "disk_{}".format(disk_info.serial_number)

            # system disk
            # using first available disk not from node storage
            # assuming _iterate_over_disk_conf() order: hdds -> ssds -> nvmes
            if not (disk_info.from_node_storage or system_disk_present):
                # partition order (mandatory, should be in sync with self.fs_descrs):
                # biosboot, ESP, boot, rootA, rootB, home, place, yt
                parted[block_name] = ["grub", "2g", "2g", "40g", "40g", "5g", "500g", "*"]

                partition_index = 2
                for fs_descr in self.fs_descrs:
                    fs["{}_{}".format(block_name, partition_index)] = [
                        fs_descr.fs,
                        fs_descr.fs_opts,
                        fs_descr.mountpoint,
                        fs_descr.mount_options,
                    ]
                    partition_index += 1

                partition_name = "{}_{}".format(block_name, partition_index)
                system_disk_present = True
            else:
                parted[block_name] = ["grub", "*"]
                partition_name = "{}_2".format(block_name)

            mountpoint = _get_yt_mountpoint(block_kind, idx + 1)
            yt_fs_descr = _YTFSDescriptions.get_desc(block_kind)

            fs[partition_name] = [yt_fs_descr.fs, yt_fs_descr.fs_opts, mountpoint, yt_fs_descr.mount_options]

        if not system_disk_present:
            raise NoSystemDiskError(
                context.disk_conf,
                "unsupported configuration: node must have at least one internal drive for system use",
            )

        return {"parted": parted, "fs": fs}


class YtSharedConfigStrategy(DeployConfigStrategy):
    description = (
        """This policy dynamically modifies LUI config to give first HDD / NVME to YP and all other disks to YT"""
    )

    fs_descrs = [
        FilesystemDescr(name="root", mountpoint="/", mount_options="barrier=1,noatime,lazytime"),
        FilesystemDescr(name="home", mountpoint="/home", mount_options="barrier=1,noatime,lazytime,nosuid,nodev"),
        FilesystemDescr(name="place", mountpoint="/place", mount_options="barrier=1,noatime,lazytime,nosuid,nodev"),
    ]

    ssd_fs_descr = FilesystemDescr(
        name="ssd", mountpoint="/ssd", mount_options="barrier=1,noatime,lazytime,nosuid,nodev"
    )

    def generate(self, host, deploy_config_name):
        factory = SetupConfigFactory(host, deploy_config_name)
        factory.modify_config([self._gen_parted_and_fs_sections, self._gen_md_section, self._gen_lvm_section])
        return factory.get_stage_params()

    def _gen_parted_and_fs_sections(self, context):
        parted = {}
        fs = {}

        skipped = False
        for idx, disk_info in _iterate_over_disk_conf(context.disk_conf):
            block_kind = BlockKind.from_disk_kind(disk_info.kind)
            block_name = "disk_{}".format(disk_info.serial_number)

            if idx == 0:
                skipped = False

            if idx == 0 and block_kind == context.sys_disk_kind:
                # system disk
                if disk_info.from_node_storage:
                    raise NoSystemDiskError(context.disk_conf, "system disk located on node storage")

                parted[block_name] = ["grub", "40g", "5g", "*"]

                partition_index = 2
                for fs_descr in self.fs_descrs:
                    fs["{}_{}".format(block_name, partition_index)] = [
                        fs_descr.fs,
                        fs_descr.fs_opts,
                        fs_descr.mountpoint,
                        fs_descr.mount_options,
                    ]
                    partition_index += 1

                skipped = True
                continue
            elif idx == 0 and block_kind == BlockKind.NVME:
                # disk for /ssd
                parted[block_name] = ["grub", "*"]

                fs["{}_2".format(block_name)] = [
                    self.ssd_fs_descr.fs,
                    self.ssd_fs_descr.fs_opts,
                    self.ssd_fs_descr.mountpoint,
                    self.ssd_fs_descr.mount_options,
                ]

                skipped = True
                continue
            else:
                parted[block_name] = ["grub", "*"]
                partition_name = "{}_2".format(block_name)

            if not skipped:
                idx += 1
            mountpoint = _get_yt_mountpoint(block_kind, idx)
            yt_fs_descr = _YTFSDescriptions.get_desc(block_kind)

            fs[partition_name] = [yt_fs_descr.fs, yt_fs_descr.fs_opts, mountpoint, yt_fs_descr.mount_options]

        return {"parted": parted, "fs": fs}

    def _gen_md_section(self, context):
        # we don't need md
        return {"md": {}}

    def _gen_lvm_section(self, context):
        # we don't need lvm
        return {"lvm": {}}


class YtMastersConfigStrategy(DeployConfigStrategy):
    description = """This policy dynamically modifies LUI config to give system disk to YP and all other disks to YT"""

    fs_descrs = [
        FilesystemDescr(name="root", mountpoint="/", mount_options="barrier=1,noatime,lazytime"),
        FilesystemDescr(name="home", mountpoint="/home", mount_options="barrier=1,noatime,lazytime,nosuid,nodev"),
        FilesystemDescr(name="place", mountpoint="/place", mount_options="barrier=1,noatime,lazytime,nosuid,nodev"),
    ]

    def generate(self, host, deploy_config_name):
        factory = SetupConfigFactory(host, deploy_config_name)
        factory.modify_config([self._gen_parted_and_fs_sections, self._gen_md_section, self._gen_lvm_section])
        return factory.get_stage_params()

    def _gen_parted_and_fs_sections(self, context):
        parted = {}
        fs = {}

        skipped = False
        for idx, disk_info in _iterate_over_disk_conf(context.disk_conf):
            block_kind = BlockKind.from_disk_kind(disk_info.kind)
            block_name = "disk_{}".format(disk_info.serial_number)

            if idx == 0:
                skipped = False

            if idx == 0 and block_kind == context.sys_disk_kind:
                # system disk
                if disk_info.from_node_storage:
                    raise NoSystemDiskError(context.disk_conf, "system disk located on node storage")

                parted[block_name] = ["grub", "40g", "5g", "*"]

                partition_index = 2
                for fs_descr in self.fs_descrs:
                    fs["{}_{}".format(block_name, partition_index)] = [
                        fs_descr.fs,
                        fs_descr.fs_opts,
                        fs_descr.mountpoint,
                        fs_descr.mount_options,
                    ]
                    partition_index += 1

                skipped = True
            else:
                parted[block_name] = ["grub", "*"]
                partition_name = "{}_2".format(block_name)

                if not skipped:
                    idx += 1
                mountpoint = _get_yt_mountpoint(block_kind, idx)
                yt_fs_descr = _YTFSDescriptions.get_desc(block_kind)

                fs[partition_name] = [yt_fs_descr.fs, yt_fs_descr.fs_opts, mountpoint, yt_fs_descr.mount_options]

        return {"parted": parted, "fs": fs}

    def _gen_md_section(self, context):
        # we don't need md
        return {"md": {}}

    def _gen_lvm_section(self, context):
        # we don't need lvm
        return {"lvm": {}}


class MdsDedicatedConfigStrategy(DeployConfigStrategy):
    description = """This policy dynamically modifies LUI config to give all hdd but two to MDS"""

    sys_fs_descrs = [
        FilesystemDescr(name="ESP", fs="vfat", fs_opts="-S 512 -s 8", mountpoint="/boot/efi", mount_options="noatime"),
        FilesystemDescr(name="boot", mountpoint="/boot", mount_options="barrier=1,noatime,lazytime"),
        FilesystemDescr(name="rootA", mountpoint="/", mount_options="barrier=1,noatime,lazytime"),
        FilesystemDescr(name="rootB", mountpoint="/rootB", mount_options="barrier=1,noatime,lazytime"),
        FilesystemDescr(name="home", mountpoint="/home", mount_options="barrier=1,noatime,lazytime,nosuid,nodev"),
    ]

    place_fs_descr = FilesystemDescr(
        name="place", mountpoint="/place", mount_options="barrier=1,noatime,lazytime,nosuid,nodev"
    )

    ssd_fs_descr = FilesystemDescr(
        name="ssd", mountpoint="/ssd", mount_options="barrier=1,noatime,lazytime,nosuid,nodev"
    )

    mds_fs_descr = FilesystemDescr(
        name="mds",
        fs_opts="-b 4096 -J size=4096 -m 0",
        mount_options="nodiscard,noatime,lazytime,nosuid,nodev,errors=remount-ro",
    )

    def generate(self, host, deploy_config_name):
        factory = SetupConfigFactory(host, deploy_config_name)
        factory.modify_config([self._gen_config])
        return factory.get_stage_params()

    def _gen_config(self, context):
        parted, lvm, fs = self._gen_parted_lvm_fs_info(context)

        return {'parted': parted, 'md': {}, 'lvm': lvm, 'fs': fs}

    def _gen_parted_lvm_fs_info(self, context):
        parted = {}
        fs = {}
        lvm = {}

        place_partitions = []
        ssd_partitions = defaultdict(list)

        for idx, disk_info in _iterate_over_disk_conf(context.disk_conf):
            block_kind = BlockKind.from_disk_kind(disk_info.kind)
            block_name = "disk_{}".format(disk_info.serial_number)

            if idx == 0 and block_kind == context.sys_disk_kind:
                # system disk
                if disk_info.from_node_storage:
                    raise NoSystemDiskError(context.disk_conf, "system disk located on node storage")
                # partition order (mandatory, should be in sync with self.sys_fs_descrs):
                # biosboot, ESP, boot, rootA, rootB, home, place
                parted[block_name] = ["grub", "2g", "2g", "40g", "40g", "5g", "*"]

                partition_index = 2
                for fs_descr in self.sys_fs_descrs:
                    partition_name = "{}_{}".format(block_name, partition_index)
                    fs[partition_name] = [
                        fs_descr.fs,
                        fs_descr.fs_opts,
                        fs_descr.mountpoint,
                        fs_descr.mount_options,
                    ]
                    partition_index += 1
                place_partitions.append("{}_{}".format(block_name, partition_index))

            elif idx >= 2 and block_kind == BlockKind.HDD:
                # mds hdds
                parted[block_name] = ["grub", "*"]
                partition_name = "{}_2".format(block_name)
                fs[partition_name] = [
                    self.mds_fs_descr.fs,
                    self.mds_fs_descr.fs_opts,
                    "/srv/storage/{}".format(idx - 1),
                    self.mds_fs_descr.mount_options,
                ]
            else:
                # non-system /place and /ssd disks
                parted[block_name] = ["grub", "*"]
                partition_name = "{}_2".format(block_name)
                if block_kind == BlockKind.HDD or context.sys_disk_kind != BlockKind.HDD:
                    place_partitions.append(partition_name)
                else:
                    ssd_partitions[block_kind].append(partition_name)

            place_options = "--addtag diskman.sys=true"
            stripes = _get_stripes(place_partitions, context.sys_disk_kind)
            if stripes is not None and stripes >= 2:
                place_options = "{} --stripes {}".format(place_options, stripes)
            lvm["vg0"] = [context.sys_disk_kind, ",".join(place_partitions)]
            lvm["lv0"] = ["place", "*", "vg0", place_options]
            fs["lv0"] = [
                self.place_fs_descr.fs,
                self.place_fs_descr.fs_opts,
                self.place_fs_descr.mountpoint,
                self.place_fs_descr.mount_options,
            ]

            if ssd_partitions:
                ssd_partitions_type = BlockKind.NVME if BlockKind.NVME in ssd_partitions else BlockKind.SSD
                ssd_options = "--addtag diskman.sys=true"
                stripes = _get_stripes(ssd_partitions[ssd_partitions_type], ssd_partitions_type)
                if stripes is not None and stripes >= 2:
                    ssd_options = "{} --stripes {}".format(ssd_options, stripes)
                lvm["vg1"] = [ssd_partitions_type, ",".join(ssd_partitions[ssd_partitions_type])]
                lvm["lv1"] = ["ssd", "*", "vg1", ssd_options]
                fs["lv1"] = [
                    self.ssd_fs_descr.fs,
                    self.ssd_fs_descr.fs_opts,
                    self.ssd_fs_descr.mountpoint,
                    self.ssd_fs_descr.mount_options,
                ]
        return parted, lvm, fs


class DeployConfigPolicies:
    PASSTHROUGH = "passthrough"
    DISKMANAGER = "diskmanager"
    SHARED = "shared"
    SHAREDLVM = "sharedlvm"
    YT_DEDICATED = "yt_dedicated"
    YT_SHARED = "yt_shared"
    YT_MASTERS = "yt_masters"
    YT_STORAGE = "yt_storage"
    MDS_DEDICATED = "mds_dedicated"

    _name_to_class = {
        PASSTHROUGH: PassthroughConfigStrategy,
        SHARED: SharedConfigStrategy,
        SHAREDLVM: SharedLvmConfigStrategy,
        DISKMANAGER: DiskManagerConfigStrategy,
        YT_DEDICATED: YtDedicatedConfigStrategy,
        YT_SHARED: YtSharedConfigStrategy,
        YT_MASTERS: YtMastersConfigStrategy,
        YT_STORAGE: YtStorageConfigStrategy,
        MDS_DEDICATED: MdsDedicatedConfigStrategy,
    }

    @classmethod
    def get_all_names(cls):
        return sorted(cls._name_to_class.keys())

    @classmethod
    def get_policy_class(cls, name):
        if name is None:
            return cls.get_default_policy_class()

        try:
            return cls._name_to_class[name]
        except KeyError:
            raise DeployConfigPolicyNotFound(name)

    @classmethod
    def get_default_policy_class(cls):
        return cls.get_policy_class(cls.PASSTHROUGH)
