import json
import logging
import uuid

import sentry_sdk
from errorboosterclient.sentry import ErrorBoosterTransport

from sepelib.core.exceptions import Error

DEFAULT_PROJECT = "Wall-E"


class ErrorBoosterSender:
    def __init__(self, env, path_format, when, backup_count):
        self.logger_name = "{}.{}".format(__name__, str(uuid.uuid1()))

        logger = logging.getLogger(self.logger_name)
        logger.level = logging.DEBUG
        logger.propagate = False

        path = path_format.format(env=env)

        handler = logging.handlers.TimedRotatingFileHandler(
            filename=path,
            when=when,
            backupCount=backup_count,
        )
        handler.setFormatter(logging.Formatter("%(message)s"))
        logger.addHandler(handler)

    def __call__(self, event):
        logging.getLogger(self.logger_name).debug(json.dumps(event))


def init(env, config):
    enabled = config.get("enabled", False)
    if not enabled:
        return

    project = config.get("project", DEFAULT_PROJECT)
    params = dict(env=env)
    for field in ["path_format", "when", "backup_count"]:
        params[field] = config.get(field, None)
        if params[field] is None:
            raise Error("ErrorBooster: field {} are required", field)

    sentry_sdk.init(
        environment=env,
        with_locals=False,
        send_default_pii=True,
        transport=ErrorBoosterTransport(project=project, sender=ErrorBoosterSender(**params)),
        default_integrations=False,
    )
