"""Gevent utilities."""

import functools
import json
import logging
import time

import gevent
import simplejson

log = logging.getLogger(__name__)


_JSON_MONKEYPATCHED, _JSON_ORIG_DECODERS = False, None


def monkeypatch_json(idle_every=1000):
    """
    Monkeypatch JSON modules to not block gevent when parsing huge JSON objects.

    Don't monkeypatch JSONEncoder: the only way to monkeypatch it is `item_sort_key`, but it's called for each object
    key, which is too expensive. Believe that we don't produce very large JSONs - only consume.
    """

    global _JSON_MONKEYPATCHED, _JSON_ORIG_DECODERS
    if _JSON_MONKEYPATCHED:
        return

    class IdleObjectHook:
        def __init__(self):
            self.counter = 0

        def __call__(self, value):
            self.counter += 1

            if self.counter >= idle_every:
                self.counter = 0
                gevent.idle()

            return value

    def wrap_object_hook(object_hook, idle_object_hook=IdleObjectHook()):
        if object_hook is None:
            return idle_object_hook

        @functools.wraps(object_hook)
        def wrapper(value):
            return idle_object_hook(object_hook(value))

        return wrapper

    def get_decoder(base_class):
        class Decoder(base_class):
            def __init__(self, **kwargs):
                kwargs["object_hook"] = wrap_object_hook(kwargs.get("object_hook"))
                super().__init__(**kwargs)

            def decode(self, *args, **kwargs):
                start_time = time.time()
                result = super().decode(*args, **kwargs)

                decode_time = time.time() - start_time
                if decode_time > 1:
                    log.debug("A big JSON object decoded: it took %.1f seconds.", decode_time)

                return result

        return Decoder

    _JSON_MONKEYPATCHED, _JSON_ORIG_DECODERS = True, []

    for json_module in (json, simplejson):
        Decoder = get_decoder(json_module.JSONDecoder)
        _JSON_ORIG_DECODERS.append((json_module, json_module.JSONDecoder, json_module._default_decoder))

        # json uses default decoder when called without custom parameters
        json_module._default_decoder = Decoder()
        json_module.JSONDecoder = json_module.decoder.JSONDecoder = Decoder


def demonkeypatch_json():
    global _JSON_MONKEYPATCHED, _JSON_ORIG_DECODERS
    if not _JSON_MONKEYPATCHED:
        return

    for json_module, decoder, default_decoder in _JSON_ORIG_DECODERS:
        json_module._default_decoder = default_decoder
        json_module.JSONDecoder = json_module.decoder.JSONDecoder = decoder

    _JSON_MONKEYPATCHED, _JSON_ORIG_DECODERS = False, None


def gevent_idle_iter(objs, idle_on_start=False, cpu_intensive=False):
    idle_period = 10 if cpu_intensive else 100

    for obj_id, obj in enumerate(objs, 0 if idle_on_start else 1):
        if obj_id % idle_period == 0:
            gevent.idle()
        yield obj


def gevent_idle_generator(func=None, idle_on_start=False, cpu_intensive=False):
    """Decorator that can make any generator function gevent friendly."""
    if func is None:
        return functools.partial(gevent_idle_generator, idle_on_start=idle_on_start, cpu_intensive=cpu_intensive)

    @functools.wraps(func)
    def _generator(*args, **kwargs):
        return gevent_idle_iter(func(*args, **kwargs), idle_on_start=idle_on_start, cpu_intensive=cpu_intensive)

    return _generator
