from typing import Any, Optional, Union

from walle.util.misc import drop_none


class _Base:
    def get_schema(self) -> dict[str, Any]:
        raise NotImplementedError()

    @staticmethod
    def get_allowed_params() -> frozenset[str]:
        return frozenset()


class AnyOf(_Base):
    def __init__(self, schemes: list[_Base]):
        self.schemes = schemes

    def get_schema(self) -> dict[str, Any]:
        return {"anyOf": [schema.get_schema() for schema in self.schemes]}


class OneOf(_Base):
    def __init__(self, schemes: list[_Base]):
        self.schemes = schemes

    def get_schema(self) -> dict[str, Any]:
        return {"oneOf": [schema.get_schema() for schema in self.schemes]}


class OneOfPartial(_Base):
    def __init__(self, parent: _Base, params: list[dict[str, Any]]):
        self.parent = parent
        self.params = params

    def get_schema(self) -> dict[str, Any]:
        schema = self.parent.get_schema()
        allowed_params = self.parent.get_allowed_params()

        for variant in self.params:
            if not all(param in allowed_params for param in variant):
                raise ValueError("Params {} is not allowed for schema".format(tuple(variant.keys())))
            if any(param in schema for param in variant):
                raise ValueError("One of params {} is defined unambiguously in schema".format(tuple(variant.keys())))

        return {**schema, "oneOf": self.params}


class Enum(_Base):
    def __init__(self, elements: list[Any]):
        self.elements = elements

    def get_schema(self) -> dict[str, Any]:
        return {"enum": self.elements}


class Null(_Base):
    def __init__(self):
        pass

    def get_schema(self) -> dict[str, Any]:
        return {"type": "null"}


class Boolean(_Base):
    def __init__(self):
        pass

    def get_schema(self) -> dict[str, Any]:
        return {"type": "boolean"}


class Number(_Base):
    def __init__(self, minimum: Optional[Union[int, float]] = None, maximum: Optional[Union[int, float]] = None):
        self.minimum = minimum
        self.maximum = maximum

    def get_schema(self) -> dict[str, Any]:
        return drop_none({"type": "number", "minimum": self.minimum, "maximum": self.maximum})

    @staticmethod
    def get_allowed_params() -> frozenset[str]:
        return frozenset(("minimum", "maximum"))


class Integer(_Base):
    def __init__(self, minimum: Optional[int] = None, maximum: Optional[int] = None):
        self.minimum = minimum
        self.maximum = maximum

    def get_schema(self) -> dict[str, Any]:
        return drop_none({"type": "integer", "minimum": self.minimum, "maximum": self.maximum})

    @staticmethod
    def get_allowed_params() -> frozenset[str]:
        return frozenset(("minimum", "maximum"))


class String(_Base):
    def __init__(
        self, min_length: Optional[int] = None, max_length: Optional[int] = None, pattern: Optional[str] = None
    ):
        self.min_length = min_length
        self.max_length = max_length
        self.pattern = pattern

    def get_schema(self) -> dict[str, Any]:
        return drop_none(
            {"type": "string", "minLength": self.min_length, "maxLength": self.max_length, "pattern": self.pattern}
        )

    @staticmethod
    def get_allowed_params() -> frozenset[str]:
        return frozenset(("minLength", "maxLength"))


class Array(_Base):
    def __init__(self, items: Optional[_Base], min_items: Optional[int] = None, max_items: Optional[int] = None):
        self.items = items
        self.min_items = min_items
        self.max_items = max_items

    def get_schema(self) -> dict[str, Any]:
        return drop_none(
            {
                "type": "array",
                "items": self.items.get_schema() if self.items is not None else None,
                "minItems": self.min_items,
                "maxItems": self.max_items,
            }
        )

    @staticmethod
    def get_allowed_params() -> frozenset[str]:
        return frozenset(("minItems", "maxItems"))


class Object(_Base):
    def __init__(
        self, properties: dict[str, _Base], required: Optional[list[str]] = None, additional_properties: bool = True
    ):
        self.properties = properties
        self.required = required or []
        self.additional_properties = additional_properties

    def get_schema(self) -> dict[str, Any]:
        return {
            "type": "object",
            "properties": {key: value.get_schema() for key, value in self.properties.items()},
            "required": self.required,
            "additionalProperties": self.additional_properties,
        }


class WithDescription(_Base):
    def __init__(self, parent: _Base, description: str):
        self.parent = parent
        self.description = description

    def get_schema(self) -> dict[str, Any]:
        schema = self.parent.get_schema()
        if "description" in schema:
            raise ValueError("Description is defined unambiguously in schema")

        return {**schema, "description": self.description}


class WithDefault(_Base):
    def __init__(self, parent: _Base, value: Optional[Union[str, int, float, bool]]):
        self.parent = parent
        self.value = value

    def get_schema(self) -> dict[str, Any]:
        schema = self.parent.get_schema()
        if "default" in schema:
            raise ValueError("Default value is defined unambiguously in schema")

        return {**schema, "default": self.value}
