"""Some fun hacks around logging."""

import inspect
import logging
import time
from functools import wraps

from contextlib2 import contextmanager


def wrap_logging_helper(func):
    @wraps(func)
    def wrapped_helper(*args, **kwargs):
        frame = inspect.currentframe()
        with tmp_monkeypatch(logging, "currentframe", lambda: frame):
            return func(*args, **kwargs)

    return wrapped_helper


@contextmanager
def tmp_monkeypatch(obj, attr, stub_obj, _undef=object()):
    orig = getattr(obj, attr, _undef)
    try:
        setattr(obj, attr, stub_obj)
        yield
    finally:
        if orig is _undef:
            delattr(obj, attr)
        else:
            setattr(obj, attr, orig)


class RateLimitFilter:
    """
    Allows approximately one record with particular fingerprint during interval seconds
    """

    def __init__(self, interval=5):
        self.interval = interval
        self._seen_fingerprints = set()
        self._last_interval_start = 0

    def filter(self, record):
        now = time.time()
        if self._last_interval_start < (now - self.interval):
            self._seen_fingerprints.clear()
            self._last_interval_start = now

        if record.msg in self._seen_fingerprints:
            return False

        self._seen_fingerprints.add(record.msg)
        return True
