import typing as tp
from logging import Logger

import gevent
from gevent.event import Event
from pymongo.errors import ConnectionFailure, OperationFailure


class MongoRetry:
    """Helper for retrying method which can raise pymongo errors"""

    RETRY_EXCEPTIONS = (ConnectionFailure, OperationFailure)

    def __init__(
        self,
        logger: Logger,
        retry_attempts: int = 1,
        delay: float = 0.5,
        backoff: float = 1.5,
        interrupt_event: tp.Optional[Event] = None,
    ) -> None:
        self.retry_attempts = retry_attempts
        self.delay = delay
        self._cur_delay = delay
        self._attempts = 0
        self.interrupt_event = interrupt_event
        self.backoff = backoff
        self.logger = logger

    def reset(self) -> None:
        self._attempts = 0
        self._cur_delay = self.delay

    def __call__(self, func, *args: tp.Any, **kwargs: tp.Any) -> tp.Any:
        self.reset()
        error = None
        while True:
            try:
                if error is not None:
                    self.logger.error("Retry %s call due to error: %s", func, error)
                return func(*args, **kwargs)
            except self.RETRY_EXCEPTIONS as e:
                error = e
                self._attempts += 1
                if self._attempts == self.retry_attempts:
                    raise e

                self.logger.error(
                    "%s call will be retried after %s seconds due to error %s", func, self._cur_delay, error
                )
                if self.interrupt_event:
                    if self.interrupt_event.wait(timeout=self._cur_delay):
                        return
                else:
                    gevent.sleep(self._cur_delay)

                self._cur_delay *= self.backoff
