"""Various network utilities."""

import hashlib
import re
import socket
from ipaddress import IPv6Address, IPv6Network
from ipaddress import ip_address as parse_ip_address

import six

from sepelib.core.exceptions import Error
from walle.errors import InvalidHostNameError


class InvalidNetworkError(Error):
    pass


class InvalidEui64AddressError(Error):
    def __init__(self, ip):
        super().__init__("{} is not a EUI-64 address.", ip)


class InvalidHbfIPv6AddressError(Error):
    def __init__(self, ip):
        super().__init__("{} is not an HBF-enabled IPv6 address.", ip)


def is_valid_fqdn(fqdn):
    return re.search(r"^(?:[a-z0-9]+(?:-[a-z0-9]+)*\.)+[a-z]{2,}$", fqdn, re.IGNORECASE) is not None


def mac_to_int(mac):
    return int(mac.replace(".", "").replace(":", ""), 16)


def mac_from_int(mac):
    return format_mac("{:012x}".format(mac))


def format_mac(mac):
    mac = mac.strip().replace(":", "").replace(".", "").lower()
    if re.search(r"^[0-9a-f]{12}$", mac) is None:
        raise Error("Got an invalid MAC address: {}.", mac)

    formatted = ""
    for byte in range(len(mac) // 2):
        if byte:
            formatted += ":"
        formatted += mac[byte * 2 : byte * 2 + 2]

    return formatted


def ip_to_int(ip, family=None):
    # Don't use any IP library here - all of them are too slow

    if family is None:
        family = socket.AF_INET6 if ":" in ip else socket.AF_INET
    raw_ip = socket.inet_pton(family, ip)
    hex_ip = raw_ip.encode("hex") if six.PY2 else raw_ip.hex()
    int_ip = int(hex_ip, 16)

    # Transform IPv4 addresses to IPv4-mapped IPv6 addresses
    if family == socket.AF_INET:
        int_ip |= 0xFFFF00000000

    return int_ip


def is_local_ip(int_ip):
    """Attention: Assumes that the specified IP is IPv6 or IPv4-mapped IPv6 address."""

    # Don't use any IP library here - all of them are too slow

    if (
        # IPv4-mapped IPv6 address
        int_ip >> 32 == 0xFFFF
        and (
            # 10.0.0.0/8
            int_ip & 0xFF000000 == 0x0A000000
            or
            # 169.254.0.0/16
            int_ip & 0xFFFF0000 == 0xA9FE0000
            or
            # 172.16.0.0/12
            int_ip & 0xFFF00000 == 0xAC100000
            or
            # 192.168.0.0/16
            int_ip & 0xFFFF0000 == 0xC0A80000
        )
    ):
        return True

    return False


def gethostbyaddr(ip):
    # Python doesn't have these error code constants

    # The specified host is unknown.
    HOST_NOT_FOUND = 1

    # A temporary error occurred on an authoritative name server. Try again later.
    # TRY_AGAIN = 2

    # A nonrecoverable name server error occurred.
    # NO_RECOVERY = 3

    # The requested name is valid but does not have an IP address.
    NO_ADDRESS = NO_DATA = 4

    try:
        return str(socket.gethostbyaddr(ip)[0])
    except socket.herror as e:
        if e.errno in (HOST_NOT_FOUND, NO_ADDRESS, NO_DATA):
            return None

        raise Error("Error during lookup of {}: {}.", ip, str(e))


def explode_ip(ip_address):
    return str(parse_ip_address(str(ip_address)).exploded)


def get_host_ips(host):
    v4, v6 = set(), set()

    try:
        addresses = socket.getaddrinfo(host, None)
    except socket.gaierror as e:
        if e.errno in (socket.EAI_NODATA, socket.EAI_NONAME):
            raise InvalidHostNameError(host)

        raise Error("Error during lookup of {} host: {}.", host, e)

    for address in addresses:
        family = address[0]

        if family == socket.AF_INET:
            v4.add(str(address[4][0]))
        elif family == socket.AF_INET6:
            v6.add(str(address[4][0]))

    if not v4 and not v6:
        raise InvalidHostNameError(host)

    return v4, v6


def get_host_int_ips(host):
    return {
        ip_to_int(ip, family)
        for family, ips in zip((socket.AF_INET, socket.AF_INET6), get_host_ips(host))
        for ip in ips
    }


_64_NETWORK_MASK = 0xFFFFFFFFFFFFFFFF0000000000000000
_64_HOST_MASK = 0x0000000000000000FFFFFFFFFFFFFFFF
_EUI_64_MASK = 0x0000000000000000000000FFFE000000
_IPV4_UPSCL_MASK = 0x000000000000000000000000FFFFFFFF


def get_eui_64_address(network, mac):
    """Generates an IPv6 EUI-64 address."""

    network_obj = IPv6Network(network)
    if network_obj.prefixlen != 64:
        raise InvalidNetworkError("The network must be /64 to form an IPv6 EUI-64 address.")

    int_network = int(network_obj.network_address)

    int_mac = mac_to_int(mac)
    eui_64_mac = int_mac << (2 * 8) & 0xFFFFFF0000000000 | _EUI_64_MASK | int_mac & 0x000000FFFFFF

    invert_bit = 1 << (8 * 8 - 7)
    modified_eui_64_mac = eui_64_mac ^ invert_bit

    return str(IPv6Address(int_network | modified_eui_64_mac))


def get_ipv4_embedded_ipv6_address(network, ipv4):
    network_obj = IPv6Network(network)

    if network_obj.prefixlen > 80:
        raise InvalidNetworkError(
            "The network prefix length must be maximum /80 to form a proper IPv4-embedding IPv6 address."
        )

    int_network = int(network_obj.network_address)
    int_ipv4_address = ip_to_int(ipv4) & _IPV4_UPSCL_MASK

    return str(IPv6Address(int_network | int_ipv4_address))


def get_hbf_ipv6_address_with_mac_hostid(network, project_id, mac):
    """Generate IPv6 address for hbf-enabled networks.
    https://wiki.yandex-team.ru/NOC/newnetwork/hbf/project_id/#chtotakoeprojectid
    Generate host part from the last 32 bits of MAC-address.
    """

    hostid = mac_to_int(mac) & 0x0000FFFFFFFF
    return _get_hbf_ipv6_address(network, project_id, hostid)


def get_hbf_ipv6_address_with_hostname_hostid(network, project_id, hostname):
    """Generate IPv6 address for hbf-enabled networks.
    https://wiki.yandex-team.ru/NOC/newnetwork/hbf/project_id/#chtotakoeprojectid
    Generate host part from the first 32 bits of the hash from the hostname.

    see https://st.yandex-team.ru/WALLESUPPORT-334
    """

    hostid = int(hashlib.md5(six.ensure_binary(hostname, "utf-8")).hexdigest()[:8], 16)
    return _get_hbf_ipv6_address(network, project_id, hostid)


def _get_hbf_ipv6_address(network, project_id, hostid):
    """Generate IPv6 address for hbf-enabled networks.
    https://wiki.yandex-team.ru/NOC/newnetwork/hbf/project_id/#chtotakoeprojectid
    Generate host part from the last 32 bits of MAC-address.
    """

    network_obj = IPv6Network(network)
    if network_obj.prefixlen != 64:
        raise InvalidNetworkError("The network must be /64 for HBF-enabled projects.")

    int_network = int(network_obj.network_address)

    return str(IPv6Address(int_network | (project_id << 32) | hostid))


def split_eui_64_address(ip):
    """Splits an IPv6 EUI-64 address into IPv6 network and host MAC address."""

    int_ip = int(IPv6Address(ip))
    if int_ip & _EUI_64_MASK != _EUI_64_MASK:
        raise InvalidEui64AddressError(ip)

    network = str(IPv6Network(int_ip & _64_NETWORK_MASK).supernet(64))

    int_host_part = int_ip & _64_HOST_MASK
    modified_eui_64_mac = int_host_part >> (2 * 8) & 0xFFFFFFFFFFFFFFFFFFFFFF000000 | int_host_part & 0xFFFFFF

    invert_bit = 1 << (6 * 8 - 7)
    int_mac = modified_eui_64_mac ^ invert_bit

    return network, mac_from_int(int_mac)


def split_hbf_ipv6_address(ip):
    """Split a HBF-enabled host's IPv6 address into network, project is and mac-part"""
    int_ip = int(IPv6Address(ip))

    network = str(IPv6Network(int_ip & _64_NETWORK_MASK).supernet(64))
    int_host_part = int_ip & _64_HOST_MASK
    mac_part = int_host_part & 0x0000FFFFFFFF
    project_id = (int_host_part - mac_part) >> 32

    if not project_id:
        raise InvalidHbfIPv6AddressError(ip)
    # mac part will always start with 00:00
    return network, project_id, mac_from_int(mac_part)
