"""Email notifications."""

import logging
import textwrap

import cachetools
import yaml
from cachetools import cached
from cachetools.keys import hashkey

import walle.util.misc
from sepelib.core import config
from sepelib.util.net.mail import Mail as Mailer
from walle import authorization, util
from walle.authorization import get_issuer_name, ISSUER_WALLE
from walle.models import timestamp

log = logging.getLogger(__name__)

SEVERITY_AUDIT = "audit"
SEVERITY_INFO = "info"
SEVERITY_WARNING = "warning"
SEVERITY_BOT = "bot"
SEVERITY_ERROR = "error"
SEVERITY_GLOBAL = "global"  # recipients = all owners of all projects linked to some automation plot
SEVERITY_CRITICAL = "critical"  # recipients = all owners of all projects

_SEVERITY_ID_MAPPING = {
    SEVERITY_AUDIT: 0,
    SEVERITY_INFO: 1,
    SEVERITY_WARNING: 2,
    SEVERITY_BOT: None,  # This is a special severity which is separated from all other severities
    SEVERITY_ERROR: 3,
    SEVERITY_GLOBAL: 4,
    SEVERITY_CRITICAL: 5,
}

SEVERITIES = sorted(_SEVERITY_ID_MAPPING)

RATELIMIT_CACHE_MAXSIZE = 128
RATELIMIT_CACHE_TTL = 600


def send_email(recipients, subject, body, **extra_headers):
    if not recipients:
        return

    log.debug("Sending notification to %s: %s.", ", ".join(recipients), subject)

    mail_config = config.get_value("mail")
    headers = _collect_headers(mail_config, extra_headers)

    try:
        mailer = Mailer.from_config(mail_config)
        mailer.send_message(
            subject=subject,
            body=body,
            sender=config.get_value("notifications.sender"),
            recipients=list(recipients),
            extra_headers=headers,
        )
    except Exception as e:
        log.exception("Failed to send notification '%s': %s", subject, e)


def on_event(entry):
    from walle.audit_log import (
        STATUS_UNKNOWN,
        STATUS_ACCEPTED,
        STATUS_REJECTED,
        STATUS_COMPLETED,
        STATUS_CANCELLED,
        STATUS_FAILED,
    )

    status = entry.status

    if status in (STATUS_UNKNOWN, STATUS_REJECTED):
        return

    if status not in (STATUS_ACCEPTED, STATUS_COMPLETED, STATUS_CANCELLED, STATUS_FAILED):
        log.critical("Got an audit log entry with an unknown status %s.", status)
        return

    subject = ""

    if entry.host_inv is not None:
        if entry.host_name is not None:
            target = entry.host_name
        else:
            target = "#{}".format(entry.host_inv)

        if entry.project is not None:
            target += " ({})".format(entry.project)
    elif entry.project is not None:
        target = entry.project
    else:
        target = None

    if target:
        subject += target + ": "

    issuer_name = get_issuer_name(entry.issuer)
    subject += "{event} by {issuer} -> {status}".format(event=entry.type, issuer=issuer_name, status=status)
    message = (
        textwrap.dedent(
            """
        Event: {entry.type}
        Issuer: {issuer}
        Status: {entry.status}

        Event Time: {time}
        Status time: {status_time}
    """
        )
        .strip("\n")
        .format(
            entry=entry,
            issuer=issuer_name,
            time=util.misc.format_time(entry.time),
            status_time=util.misc.format_time(entry.status_time),
        )
    )

    if entry.project is not None or entry.host_inv is not None:
        message += "\n"

        if entry.project is not None:
            message += "\nProject: " + entry.project

        if entry.host_inv is not None:
            message += "\nHost inventory number: #{}".format(entry.host_inv)

        if entry.host_name is not None:
            message += "\nHost name: " + entry.host_name

        if entry.host_uuid is not None:
            message += "\nHost UUID: " + entry.host_uuid

    if entry.reason:
        message += "\n\nEvent reason: " + entry.reason

    if entry.payload:
        details = yaml.safe_dump(entry.to_mongo()["payload"], default_flow_style=False).rstrip("\n")
        message += "\n\nDetails:\n" + _indent(details, 2)

    if entry.error:
        message += "\n\nError: " + entry.error

    _send(SEVERITY_AUDIT, subject, message, project_id=entry.project, event=entry.type, status=entry.status)


def on_task_enqueued(issuer, host, reason=None):
    from walle.hosts import TaskType

    task_type = host.task.type

    subject = "{host}: {task_type} task enqueued ({task})".format(
        host=host.human_name(), task=host.status, task_type=task_type.capitalize()
    )

    message = "{issuer} has enqueued '{task}' task for host {host} from project {project}.".format(
        host=host.human_name(), project=host.project, task=host.status, issuer=get_issuer_name(issuer)
    )
    body = _add_host_link(_add_reason(message, reason), host)
    _send(
        SEVERITY_INFO if task_type == TaskType.MANUAL else SEVERITY_WARNING,
        subject,
        body,
        project_id=host.project,
        event="task-enqueued",
    )


def on_task_completed(host):
    subject = "{host}: Task completed ({task})".format(host=host.human_name(), task=host.status)
    message = "{host} has successfully completed '{task}' task.".format(host=host.human_name(), task=host.status)
    _send(SEVERITY_INFO, subject, _add_host_link(message, host), project_id=host.project, event="task-completed")


def on_task_cancelled(issuer, host, reason=None):
    subject = "{host}: Task cancelled ({task})".format(host=host.human_name(), task=host.status)
    message = "Task '{task}' for {host} has been cancelled by {issuer}.".format(
        host=host.human_name(), task=host.status, issuer=get_issuer_name(issuer)
    )
    body = _add_host_link(_add_reason(message, reason), host)

    severity = SEVERITY_WARNING if issuer == ISSUER_WALLE else SEVERITY_INFO
    _send(
        severity,
        subject,
        body,
        project_id=host.project,
        event="task-cancelled",
        cc_user=None if issuer == host.task.owner else host.task.owner,
    )


def on_deleted_host(host, issuer, reason=None):
    subject = "Host {host} was successfully deleted from Wall-E.".format(host=host.human_name())
    body = "Host {host} deletion was requested by {issuer}{reason}.".format(
        host=host.human_name(), issuer=issuer, reason=": {}".format(reason) if reason else ""
    )
    _send(SEVERITY_INFO, subject, body, project_id=host.project, event="deleted-host", rate_limited=False)


def on_host_status_changed(host, status, reason, cc_user=None):
    if status == walle.hosts.HostStatus.DEAD:
        on_dead_host(host, reason, cc_user=cc_user)
    else:
        subject = "Host {host} status has been changed to '{status}'.".format(host=host.human_name(), status=status)
        _send(
            SEVERITY_WARNING,
            subject,
            _add_host_link(reason, host),
            project_id=host.project,
            event="host-status-changed",
            cc_user=cc_user,
        )


def on_dead_host(host, reason, cc_user=None):
    _send(
        SEVERITY_ERROR,
        "Host {host} considered dead".format(host=host.human_name()),
        _add_host_link(reason, host),
        project_id=host.project,
        event="dead-host",
        cc_user=cc_user,
    )


def on_invalid_host(host, reason):
    _send(
        SEVERITY_ERROR,
        "Host {host} is invalid".format(host=host.human_name()),
        _add_host_link(reason, host),
        project_id=host.project,
        event="invalid-host",
        rate_limited=False,
    )


def on_failed_to_create_report(project_id, reason):
    subject = "Failed to create failure report for project {project_id}.".format(project_id=project_id)
    message = "Failed to create failure report: {reason}".format(reason=reason)

    _send(SEVERITY_ERROR, subject, message, project_id=project_id, event="failed-to-create-report")


def on_healing_automation_disabled(issuer, project_id=None, reason=None):
    issuer_name = get_issuer_name(issuer)

    if project_id is None:
        subject = "Healing automation disabled globally"
        message = "Healing automation has been globally disabled by {}.".format(issuer_name)
    else:
        subject = "{} project: Healing automation disabled".format(project_id)
        message = "Healing automation has been disabled by {} for '{}' project.".format(issuer_name, project_id)

    message += " Wall-E won't undertake any self-healing actions until someone enables it back."
    _send(
        SEVERITY_CRITICAL,
        subject,
        _add_reason(message, reason),
        project_id=project_id,
        event="healing-automation-disabled",
        rate_limited=False,
    )


def on_healing_automation_enabled(issuer, project_id=None, reason=None):
    issuer_name = get_issuer_name(issuer)

    if project_id is None:
        subject = "Healing automation enabled globally"
        message = "Healing automation has been globally enabled by {}.".format(issuer_name)
    else:
        subject = "{} project: Healing automation enabled".format(project_id)
        message = "Healing automation has been enabled by {} for '{}' project.".format(issuer_name, project_id)

    message += " Wall-E will automatically heal hosts with failures until someone disables it back."
    _send(
        SEVERITY_CRITICAL,
        subject,
        _add_reason(message, reason),
        project_id=project_id,
        event="healing-automation-enabled",
        rate_limited=False,
    )


def on_dns_automation_disabled(issuer, project_id=None, reason=None):
    issuer_name = get_issuer_name(issuer)

    if project_id is None:
        subject = "DNS automation disabled globally"
        message = "DNS automation has been globally disabled by {}.".format(issuer_name)
    else:
        subject = "{} project: DNS automation disabled".format(project_id)
        message = "DNS automation has been disabled by {} for '{}' project.".format(issuer_name, project_id)

    message += " Wall-E won't try to make any DNS recovery until someone enables it back."
    _send(
        SEVERITY_CRITICAL,
        subject,
        _add_reason(message, reason),
        project_id=project_id,
        event="dns-automation-disabled",
        rate_limited=False,
    )


def on_dns_automation_enabled(issuer, project_id=None, reason=None):
    issuer_name = get_issuer_name(issuer)

    if project_id is None:
        subject = "DNS automation enabled globally"
        message = "DNS automation has been globally enabled by {}.".format(issuer_name)
    else:
        subject = "{} project: DNS automation enabled".format(project_id)
        message = "DNS automation has been enabled by {} for '{}' project.".format(issuer_name, project_id)

    message += " Wall-E will automatically recover DNS records until someone disables it back."
    _send(
        SEVERITY_CRITICAL, subject, _add_reason(message, reason), project_id=project_id, event="dns-automation-enabled"
    )


def on_automation_plot_check_disabled(issuer, plot_id, check_name, reason=None):
    subject = "Check {} disabled for automation plot {}.".format(check_name, plot_id)
    message = (
        "Check {check_name} in automation plot {plot_id} was disabled by {issuer}."
        " Wall-E will not process this check's failures until someone enables it back.".format(
            check_name=check_name, plot_id=plot_id, issuer=get_issuer_name(issuer)
        )
    )

    _send(
        SEVERITY_GLOBAL, subject, _add_reason(message, reason), plot_id=plot_id, event="automation-plot-check-disabled"
    )


def on_automation_plot_check_enabled(issuer, plot_id, check_name, reason=None):
    subject = "Check {} enabled for automation plot {}.".format(check_name, plot_id)
    message = (
        "Check {check_name} in automation plot {plot_id} was enabled by {issuer}."
        " Wall-E will automatically heal hosts with this check's failures until someone disables it back.".format(
            check_name=check_name, plot_id=plot_id, issuer=get_issuer_name(issuer)
        )
    )

    _send(
        SEVERITY_GLOBAL, subject, _add_reason(message, reason), plot_id=plot_id, event="automation-plot-check-enabled"
    )


def on_cms_api_error(error, url, project_id, response):
    """Send info about broken CMS API to projects' notification recipients.
    NB: this should not be done instead of logging,
    because if cms-api is a default cms then it's not up to projects' owners fixing it.
    """

    subject = "CMS api for your project {} is misbehaving.".format(project_id)
    message = (
        "CMS api for your project {} is misbehaving.\n"
        "Wall-E's got this error:\n{}.\n"
        "Requested url was {}.".format(project_id, error, url)
    )

    message += _dump_response(response)

    _send(SEVERITY_ERROR, subject, message, project_id=project_id, event="cms-api-error", rate_limited=True)


def get_recipients_config(project_id=None, suppress_errors=False):
    recipients_configs = [config.get_value("notifications.recipients_by_severity")]

    if project_id is not None:
        recipients_configs.append(_get_recipients_for_project(project_id, suppress_errors))

    return _merge_recipients_configs(recipients_configs)


def get_recipients_by_severity(message_severity, recipients_config):
    recipients = set()
    message_severity_id = _SEVERITY_ID_MAPPING[message_severity]

    for severity, severity_recipients in recipients_config.items():
        try:
            severity_id = _SEVERITY_ID_MAPPING[severity]
        except KeyError:
            log.error("Recipients config contains an invalid severity: %s.", severity)
            continue

        if severity_id is not None and message_severity_id >= severity_id:
            recipients.update(severity_recipients)

    return recipients


def _add_reason(message, reason=None):
    if reason is not None:
        message += "\n\nReason: " + reason

    return message


def _add_host_link(message, host):
    ui_url = config.get_value("stand.ui_url", None)

    if ui_url is None:
        log.error("Incomplete configuration, value for stand.ui_url is not set")
        return message

    message += "\n\nView host in Wall-E: {url}?hosts={uuid}".format(url=ui_url, uuid=host.uuid)
    return message


def _dump_response(response):
    result = ["=" * 20]
    if getattr(response, "request", None) is not None:
        request = response.request
        result.append("Request: {method} {url}".format(method=request.method, url=request.url))
        result.extend(_format_http_headers(request.headers))

        if request.body is not None:
            result.extend(["", request.body])

        result.extend(["", "=" * 20])

    result.append("Response: {code} {reason}".format(code=response.status_code, reason=response.reason))
    result.extend(_format_http_headers(response.headers))
    if response.content:
        result.extend(["", response.content])

    result.extend(["", "=" * 20])

    return "\n".join(map(str, result))


def _format_http_headers(headers):
    scramble_headers = {"authorization", "x-ya-service-ticket"}
    headers_list = []
    for header, value in headers.items():
        if header.lower() in scramble_headers:
            value = "*" * len(value)
        headers_list.append("{header}: {value}".format(header=header, value=value))

    return headers_list


def _send(severity, subject, body, project_id=None, plot_id=None, event=None, cc_user=None, rate_limited=True, **extra):
    recipients = tuple(sorted(_get_notification_recipients(cc_user, project_id, plot_id, severity)))
    subject = "[Wall-E] {severity}: {subject}".format(severity=severity.upper(), subject=subject)

    extra.setdefault("event", event)
    extra.setdefault("project", project_id)
    extra.setdefault("severity", severity)

    if rate_limited:
        _send_rate_limited(recipients, subject, body, **extra)
    else:
        send_email(recipients, subject, body, **extra)


def _get_notification_recipients(cc_user, project_id, plot_id, severity):
    recipients_config = get_recipients_config(project_id, suppress_errors=True)

    if severity == SEVERITY_CRITICAL and project_id is None:
        # Global critical notifications are exceptional: they must be delivered to all projects' subscribers.
        recipients_config = _merge_recipients_configs([recipients_config] + _get_recipients_configs_by_query())
    elif severity == SEVERITY_GLOBAL and plot_id and project_id is None:
        recipients_config = _merge_recipients_configs(
            [recipients_config] + _get_recipients_configs_by_query(plot_id=plot_id)
        )

    recipients = get_recipients_by_severity(severity, recipients_config)

    if cc_user is not None:
        cc_email = authorization.get_user_email(cc_user)
        if cc_email is not None:
            recipients.add(cc_email)

    return recipients


def _merge_recipients_configs(recipients_configs):
    """Merges several recipients configs into one."""

    result_config = {}

    for recipients_config in recipients_configs:
        for severity, recipients in recipients_config.items():
            if recipients:
                result_config.setdefault(severity, set()).update(recipients)

    return {severity: list(recipients) for severity, recipients in result_config.items()}


def _get_recipients_for_project(project_id, suppress_errors):
    import walle.projects

    recipients = {}

    try:
        project = walle.projects.get_by_id(project_id, fields=("notifications.recipients",))
    except walle.projects.ProjectNotFoundError:
        pass
    except Exception as e:
        if suppress_errors:
            log.error("Failed to get notification recipients for '%s' project: %s", project_id, e)
        else:
            raise
    else:
        if project.notifications:
            recipients = project.notifications.recipients.to_mongo().to_dict()

    return recipients


def _get_recipients_configs_by_query(plot_id=None):
    from walle.projects import Project

    query = walle.util.misc.drop_none(dict(automation_plot_id=plot_id))

    try:
        return [
            project.notifications.recipients.to_mongo().to_dict()
            for project in Project.objects(query).only("notifications.recipients")
        ]
    except Exception as e:
        log.error("Failed to get notification recipients for projects: %s", e)

    return []


def _get_recipients_configs_for_all_projects():
    from walle.projects import Project

    try:
        return [
            project.notifications.recipients.to_mongo().to_dict()
            for project in Project.objects.only("notifications.recipients")
        ]
    except Exception as e:
        log.error("Failed to get notification recipients for all projects: %s", e)

    return []


def _collect_headers(mail_config, extra_headers):
    headers = mail_config.get("default_headers", {}).copy()
    prefix = mail_config.get("headers_prefix", "")

    extra_headers.setdefault("stand", _get_stand_name())
    headers.update(_prefixed_headers(prefix, extra_headers))

    return walle.util.misc.drop_none(headers)


def _get_stand_name():
    try:
        return config.get_value("stand.name")
    except config.MissingConfigOptionError:
        return config.get_value("stand.ui_url").rstrip("/")


def _prefixed_headers(prefix, extra_headers):
    for header, value in extra_headers.items():
        yield prefix + header.capitalize(), value


def _indent(string, size):
    if string:
        padding = " " * size
        string = padding + ("\n" + padding).join(string.split("\n"))

    return string


def email_hashkey(recipients, subject, *args, **kwargs):
    return hashkey(recipients, subject)


@cached(
    cachetools.TTLCache(maxsize=RATELIMIT_CACHE_MAXSIZE, ttl=RATELIMIT_CACHE_TTL, timer=timestamp), key=email_hashkey
)
def _send_rate_limited(recipients, subject, body, **extra_headers):
    send_email(recipients, subject, body, **extra_headers)
