import logging
import logging.config
import re

from walle.constants import TESTING_ENV_NAME
from walle.util.error_booster import init as error_booster_init


def setup_logging(config, level=None):
    log_config = config.get_value("log")
    debug = config.get_value("run.debug")

    if level is None:
        level = "DEBUG" if debug else "INFO"
    else:
        level = logging.getLevelName(level)

    for handler in log_config.get("handlers", {}).values():
        formatter = handler.setdefault("formatter", "default")
        if level == "DEBUG":
            debug_formatter = formatter + ".debug"
            if debug_formatter in log_config.get("formatters", {}):
                handler["formatter"] = debug_formatter

    for logger_config in [log_config.get("root", {})] + list(log_config.get("loggers", {}).values()):
        logger_config.setdefault("level", level)

    error_booster_init(config.get_value("environment.name", TESTING_ENV_NAME), config.get_value("error_booster"))

    logging.addLevelName(logging.DEBUG, "D")
    logging.addLevelName(logging.INFO, "I")
    logging.addLevelName(logging.WARNING, "W")
    logging.addLevelName(logging.ERROR, "E")
    logging.addLevelName(logging.CRITICAL, "C")

    logging.config.dictConfig(log_config)


def make_origin_regex(origin):
    if origin.startswith("."):  # make a loophole for "any sub-domain"
        return re.compile(r"^(?:https?://)?(:?[\w_-]+.)+" + re.escape(origin.lstrip(".")) + r"/?$")
    if origin.endswith(":"):  # make a loophole for "any port"
        return re.compile(r"^(?:https?://)?" + re.escape(origin) + r"\d+" + r"/?$")
    else:
        return re.compile(r"^(?:https?://)?" + re.escape(origin) + r"/?$")
