"""Validation utilities."""

import abc
import copy
import logging
import re
from ipaddress import ip_address, ip_network

import object_validator
from object_validator import Object, String, Integer, DictScheme, List
from object_validator import ValidationError, InvalidTypeError, InvalidValueError
from sepelib.core.exceptions import LogicalError
from walle.util.gevent_tools import gevent_idle_iter

log = logging.getLogger(__name__)


class _MutuallyExclusiveFieldsError(ValidationError):
    def __init__(self, fields):
        self.__fields = fields
        super().__init__("", "Mutually exclusive fields present.")

    def get_message(self):
        return "{} contains mutually exclusive fields: {}.".format(self.object_name, ", ".join(self.__fields))


class _MissingRequiredMutuallyExclusiveFieldsError(ValidationError):
    def __init__(self, fields):
        self.__fields = fields
        super().__init__("", "Required mutually exclusive fields are missing.")

    def get_message(self):
        return "{} must have one of the following fields: {}.".format(self.object_name, ", ".join(set(self.__fields)))


class ApiDictScheme(DictScheme):
    __mutually_exclusive_required = None

    def __init__(self, scheme, ignore_unknown=True, drop_none=False, mutually_exclusive_required=None, **kwargs):
        self.__drop_none = drop_none

        if mutually_exclusive_required:
            mutually_exclusive_required = set(mutually_exclusive_required)

            if mutually_exclusive_required - set(scheme):
                raise LogicalError()

            self.__mutually_exclusive_required = mutually_exclusive_required

        super().__init__(scheme, ignore_unknown=ignore_unknown, **kwargs)

    def validate(self, obj):
        if self.__drop_none:
            if type(obj) is not dict:
                raise InvalidTypeError(obj)

            for key, value in list(obj.items()):
                if value is None:
                    del obj[key]

        obj = super().validate(obj)

        if self.__mutually_exclusive_required:
            present = set(obj) & self.__mutually_exclusive_required
            if not present:
                raise _MissingRequiredMutuallyExclusiveFieldsError(self.__mutually_exclusive_required)

            if len(present) != 1:
                raise _MutuallyExclusiveFieldsError(present)

        return obj


class Number(Object):
    """Validate value that may be of any numeric type: int, long or float."""

    def __init__(self, min_value=None, max_value=None, *args, **kwargs):
        self.__min_value = min_value
        self.__max_value = max_value
        super().__init__(*args, **kwargs)

    def validate(self, obj):
        if not isinstance(obj, (int, float)):
            raise InvalidTypeError(obj)

        if (
            self.__min_value is not None
            and obj < self.__min_value
            or self.__max_value is not None
            and obj > self.__max_value
        ):
            raise InvalidValueError(obj)

        return obj


class NoneValue(Object):
    def validate(self, obj):
        if not isinstance(obj, type(None)):
            raise InvalidTypeError(obj)

        return obj


class NumberToInteger(Number):
    """Validate value that may be of any numeric type (int, long or float) and cast it to integer."""

    def validate(self, obj):
        try:
            return int(super().validate(obj))
        except OverflowError:
            raise InvalidValueError(obj)


class StringToInteger(Integer):
    __string_validator = String()

    def __init__(self, nullable=False, *args, **kwargs):
        self._nullable = nullable
        super().__init__(*args, **kwargs)

    def validate(self, obj):
        if obj is None and self._nullable:
            return None

        obj = self.__string_validator.validate(obj)

        try:
            obj = int(obj)
        except ValueError:
            raise InvalidValueError(obj)

        return super().validate(obj)


class Nullable(Object):
    __scheme = None

    def __init__(self, scheme=None, **kwargs):
        super().__init__(**kwargs)
        if scheme:
            self.__scheme = scheme

    def validate(self, obj):
        if obj is None:
            return None

        return object_validator.validate_object(obj, self.__scheme)


class FlatableList(List):
    """Validate a list that may be just an object when contain only one value.

    Always return a list.
    """

    def __init__(self, scheme, max_length=None, **kwargs):
        # make scheme required parameter. And make it accessible.
        self._scheme = scheme
        super().__init__(scheme, min_length=None, max_length=max_length, **kwargs)

    def validate(self, obj):
        try:
            return [self._scheme.validate(obj)]
        except ValidationError:
            pass

        return super().validate(obj)


class PossiblyEmptyList(List):
    def validate(self, obj):
        if isinstance(obj, list) and len(obj) == 0:
            return obj

        return super().validate(obj)


class AnyScheme(Object):
    def __init__(self, schemes, deepcopy=True, **kwargs):
        super().__init__(**kwargs)
        self.__schemes = schemes
        self.__deepcopy = deepcopy

    def validate(self, obj):
        for scheme_id, scheme in enumerate(self.__schemes):
            try:
                return scheme.validate(copy.deepcopy(obj) if self.__deepcopy else obj)
            except ValidationError:
                if scheme_id == len(self.__schemes) - 1:
                    raise

        raise InvalidValueError(obj)


class GeventIdleList(Object):
    """A clone of object_validator.List witch uses gevent_idle_iter() for iteration over objects.

    Used for validation of very large lists.
    """

    __scheme = None

    def __init__(self, scheme=None, **kwargs):
        super().__init__(**kwargs)

        if scheme is not None:
            self.__scheme = scheme

    def validate(self, obj):
        if type(obj) is not list:
            raise InvalidTypeError(obj)

        if self.__scheme is not None:
            for index, value in enumerate(gevent_idle_iter(obj)):
                try:
                    obj[index] = object_validator.validate_object(value, self.__scheme)
                except ValidationError as e:
                    e.prefix_object_name("[{}]".format(index))
                    raise

        return obj


class IPAddress(String):
    _types = (str,)

    def validate(self, obj):
        super().validate(obj)

        try:
            ip_address(obj)
        except ValueError:
            raise InvalidValueError(obj)

        return obj


class NetworkValidator(String, metaclass=abc.ABCMeta):

    _types = (str,)  # ipaddress consumes only unicode strings

    @abc.abstractproperty
    def _network_regex(self):
        raise NotImplementedError()

    def validate(self, obj):
        super().validate(obj)

        match = self._parse(obj)
        network = match.group("network")
        try:
            ip_network(network)
        except ValueError:
            raise InvalidValueError(obj)

        return obj

    def _parse(self, obj):
        match = re.search(self._network_regex, obj)
        if not match:
            raise InvalidValueError(obj)
        return match


class Network(NetworkValidator):
    _network_regex = r"^(?P<network>.+/\d+)$"


class TrypoNetwork(NetworkValidator):
    """Network with embedded project id"""

    _network_regex = r"^[0-9a-f]+@(?P<network>.+?/\d+)$"


class TrypoNetworkRange(NetworkValidator):
    """Network with project id range"""

    _network_regex = r"^[0-9a-f]+/\d+@(?P<network>.+?/\d+)$"
