"""Audit log"""  # Will be used as category name in API reference

import logging

import mongoengine

from walle import audit_log
from walle.audit_log import LogEntry
from walle.authorization import iam
from walle.errors import ResourceNotFoundError
from walle.util.api import api_handler, api_response, get_query_result, FilterQueryParser

log = logging.getLogger(__name__)


class AuditLogEntryNotFoundError(ResourceNotFoundError):
    def __init__(self):
        super().__init__("The specified audit log entry doesn't exist.")


@api_handler(
    "/audit-log",
    "GET",
    params={
        "issuer": {"type": "array", "items": {"type": "string", "minLength": 1, "description": "Filter by issuer"}},
        "type": {"type": "array", "items": {"enum": audit_log.TYPES, "description": "Filter by event type"}},
        "project": {"type": "array", "items": {"type": "string", "minLength": 1, "description": "Filter by project"}},
        "preorder": {
            "type": "array",
            "items": {"type": "integer", "minLength": 1, "description": "Filter by preorder"},
        },
        "automation_plot": {
            "type": "array",
            "items": {"type": "string", "minLength": 1, "description": "Filter by automation plot id"},
        },
        "host_inv": {
            "type": "array",
            "items": {"type": "integer", "minimum": 0, "description": "Filter by inventory number"},
        },
        "host_name": {
            "type": "array",
            "items": {"type": "string", "minLength": 1, "description": "Filter by host name"},
        },
        "host_uuid": {
            "type": "array",
            "items": {"type": "string", "minLength": 1, "description": "Filter by host UUID"},
        },
        "status": {
            "type": "array",
            "items": {"type": "string", "enum": audit_log.STATUSES, "description": "Filter by task status"},
        },
        "scenario_id": {"type": "array", "items": {"type": "integer", "description": "Filter by scenario id"}},
        "maintenance_plot": {
            "type": "array",
            "items": {"type": "string", "minLength": 1, "description": "Filter by maintenance plot id"},
        },
        "start_time": {"type": "number", "minimum": 0, "description": "Time lower bound"},
        "end_time": {"type": "number", "minimum": 0, "description": "Time upper bound"},
        "reverse": {"type": "boolean", "description": "Log entries output order. Default is true"},
    },
    with_fields=LogEntry,
    with_paging={
        "cursor": {"type": "number", "minimum": 0, "description": "Event time to start the search from"},
        "cursor_only": True,
    },
    iam_permissions=iam.GetHostsApiIamPermission(
        projects_arg_name="project",
        hosts_arg_name="host_name",
        hosts_arg_inv="host_inv",
        hosts_arg_uuid="host_uuid",
    ),
)
def get_audit_log(query_args):
    """Returns audit log entries."""
    query_parser = FilterQueryParser(
        LogEntry,
        enum_fields=(
            "issuer",
            "type",
            "project",
            "preorder",
            "automation_plot",
            "host_inv",
            "host_uuid",
            "status",
            "scenario_id",
            "host_name",
            "maintenance_plot",
        ),
        or_fields=("host_name", "host_inv", "host_uuid"),
    )

    query = query_parser.parse_query_filter(query_args)

    time_db_field = LogEntry.time.db_field
    if "start_time" in query_args:
        query.setdefault(time_db_field, {})["$gte"] = query_args["start_time"]

    if "end_time" in query_args:
        query.setdefault(time_db_field, {})["$lt"] = query_args["end_time"]

    return api_response(
        get_query_result(
            LogEntry, query, LogEntry.time, query_args, cursor_only=True, reverse=query_args.get("reverse", True)
        )
    )


@api_handler(
    "/audit-log/<entry_id>",
    "GET",
    with_fields=LogEntry,
    iam_permissions=iam.LogEntryApiIamPermission("entry_id"),
)
def get_audit_log_entry(entry_id, query_args):
    """Returns the specified audit log entry."""

    fields = query_args.get("fields")

    try:
        log_entry = LogEntry.objects.only(*LogEntry.api_query_fields(fields)).get(id=entry_id)
    except mongoengine.DoesNotExist:
        raise AuditLogEntryNotFoundError()

    return api_response(log_entry.to_api_obj(fields))
