"""Methods to add, read and remove automation plots."""
import http.client

import mongoengine

from walle import audit_log
from walle.authorization import iam, has_iam
from walle.errors import ResourceAlreadyExistsError, ResourceNotFoundError, ResourceConflictError, BadRequestError
from walle.expert import automation_plot
from walle.expert.automation_plot import AutomationPlot
from walle.locks import AutomationPlotInterruptableLock
from walle.projects import Project
from walle.util.api import api_response, api_handler
from walle.util.validate_checks import validate_checks_list
from .request_schemas import get_automation_plot_schema


@api_handler(
    "/automation-plot/<plot_id>",
    "GET",
    with_fields=AutomationPlot,
    iam_permissions=iam.GetAutomationPlotApiIamPermission("plot_id"),
)
def get_plot(plot_id, query_args):
    """Return automation plot."""
    fields = query_args.get("fields")
    plot = automation_plot.get_automation_plot(plot_id, fields=fields)

    return api_response(plot.to_api_obj(fields))


@api_handler(
    "/automation-plot/",
    "GET",
    with_fields=AutomationPlot,
)
def list_plots(query_args):
    """List of all automation plots."""
    fields = query_args.get("fields")
    plots = automation_plot.get_all_automation_plots(fields)

    return api_response([p.to_api_obj(fields) for p in plots])


@api_handler(
    "/automation-plot/",
    "POST",
    get_automation_plot_schema(),
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.BindAutomationPlotFolderApiIamPermission("yc_iam_folder_id"),
)
def new_plot(issuer, request, reason):
    """Create new automation plot."""
    if not has_iam() and not request.get("owners"):
        raise BadRequestError("You should specify automation plot owners")

    if "checks" in request:
        validate_checks_list(request["checks"])

    plot = AutomationPlot(**request)
    with audit_log.on_add_automation_plot(issuer, plot.id, request, reason):
        try:
            plot.save(force_insert=True)
        except mongoengine.NotUniqueError:
            raise ResourceAlreadyExistsError("Automation plot with specified ID or name already exists.")

    return api_response(plot.to_api_obj(), code=http.client.CREATED)


@api_handler(
    "/automation-plot/<plot_id>",
    ("PATCH", "POST"),
    get_automation_plot_schema(full=False),
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateAutomationPlotApiIamPermission("plot_id"),
)
def modify_plot(issuer, plot_id, request, reason):
    """Update specified fields of automation plot. Checks are not "updated", whole checks list is replaced."""
    plot = automation_plot.authorize(issuer, plot_id)

    if "checks" in request:
        validate_checks_list(request["checks"])

    with audit_log.on_update_automation_plot(issuer, plot_id, request, reason):
        try:
            if not plot.modify(dict(id=plot_id), **request):
                raise ResourceNotFoundError("Automation plot with specified ID not found.")
        except mongoengine.NotUniqueError:
            raise ResourceAlreadyExistsError("Automation plot with specified name already exists.")

    updated_plot = automation_plot.get_automation_plot(plot_id, fields=None)
    return api_response(updated_plot.to_api_obj(), code=http.client.OK)


@api_handler(
    "/automation-plot/<plot_id>/<check_name>/enable",
    "POST",
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateAutomationPlotApiIamPermission("plot_id"),
)
def enable_check(issuer, plot_id, check_name, request, reason):
    """Enable specified check in automation plot."""
    plot = automation_plot.authorize(issuer, plot_id, fields=["checks"])

    automation_plot.enable_check_in_automation_plot(issuer, plot, check_name, reason=reason)
    return api_response("", code=http.client.OK)


@api_handler(
    "/automation-plot/<plot_id>/<check_name>/disable",
    "POST",
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.UpdateAutomationPlotApiIamPermission("plot_id"),
)
def disable_check(issuer, plot_id, check_name, request, reason):
    """Disable specified check in automation plot."""
    plot = automation_plot.authorize(issuer, plot_id, fields=["checks"])

    automation_plot.disable_check_in_plot(issuer, plot, check_name, reason=reason)
    return api_response("", code=http.client.OK)


@api_handler(
    "/automation-plot/<plot_id>",
    "DELETE",
    authenticate=True,
    with_reason=True,
    iam_permissions=iam.DeleteAutomationPlotApiIamPermission("plot_id"),
)
def delete_plot(issuer, plot_id, request, reason):
    """Delete specified automation plot."""

    with AutomationPlotInterruptableLock(plot_id):
        plot = automation_plot.authorize(issuer, plot_id)

        if Project.objects(automation_plot_id=plot_id).count() > 0:
            raise ResourceConflictError("There are projects that use this automation plot.")

        with audit_log.on_delete_automation_plot(issuer, plot_id, reason):
            plot.delete()

    return "", http.client.NO_CONTENT
