import functools
import http.client
import logging
from collections import defaultdict
from itertools import groupby

from walle.application import app
from walle.authorization import iam
from walle.clients import staff
from walle.clients.juggler import JugglerCheckStatus, send_event
from walle.clients.staff import is_group
from walle.hosts import Host, HostState
from walle.idm.project_role_managers import iter_project_roles_memberships, ProjectRole
from walle.models import timestamp
from walle.projects import Project
from walle.util.api import api_handler, api_response
from walle.util.misc import replace_group_with_its_members

logger = logging.getLogger(__name__)

PROJECTS_TO_HOSTS_TIMEOUT = 30
PROJECTS_TO_RESPONSIBLES_TIMEOUT = 30
PROJECTS_CAUTH_SETTINGS_TIMEOUT = 30


def _iter_non_free_hosts_to_project():
    col = Host.get_collection()
    cursor = col.find({"state": {"$in": HostState.ALL_ASSIGNED}}, ["name", "project"]).sort("project")
    for obj in cursor:
        yield obj["project"], obj["name"]


def _iter_projects_to_hosts():
    for project, hosts_group in groupby(_iter_non_free_hosts_to_project(), key=lambda p: p[0]):
        yield project, (p[1] for p in hosts_group)


def _get_projects_to_owners():
    project_to_owners = defaultdict(list)
    group_owners = set()
    for project_id, role, member in iter_project_roles_memberships():
        if role == ProjectRole.OWNER:
            project_to_owners[project_id].append(member)
            if is_group(member):
                group_owners.add(member)

    group_to_members = staff.batch_get_groups_members(tuple(sorted(group_owners)), allow_robots=False)

    project_to_owners_expanded = {
        project_id: replace_group_with_its_members(owners, group_to_members)
        for project_id, owners in project_to_owners.items()
    }
    return project_to_owners_expanded


def _iter_projects_without_hosts(projects_with_hosts):
    col = Project.get_collection()
    cursor = col.find({"_id": {"$nin": projects_with_hosts}}, ["_id"])
    for project in cursor:
        yield project["_id"]


def _get_projects_cauth_settings():
    result = {}
    for project in Project.objects.only("id", "cauth_settings"):
        project_settings = {}
        if project.cauth_settings:
            project_settings = project.cauth_settings.to_request_params()
        result[project.id] = project_settings
    return result


def cauth_response_handler(handle_name, timeout=None):
    def decorator(func):
        @functools.wraps(func)
        def decorated_function(*args, **kwargs):
            juggler_status = JugglerCheckStatus.OK
            juggler_message = "OK"

            start_time = timestamp()
            try:
                return func(*args, **kwargs)
            except Exception as e:  # usual exception handling doesn't work with @api_handler
                logger.exception("Got exception while rendering {}".format(handle_name))
                juggler_status = JugglerCheckStatus.CRIT
                juggler_message = "Got exception while rendering {}: {}".format(handle_name, repr(e))
                return api_response(juggler_message, code=http.client.INTERNAL_SERVER_ERROR)
            finally:
                exec_time = timestamp() - start_time
                if timeout is not None and exec_time >= timeout:
                    logger.error("Got timeout (%.2f >= %.2f) while rendering %s", exec_time, timeout, handle_name)
                    juggler_status = JugglerCheckStatus.CRIT
                    juggler_message = "Got timeout ({:.2f} >= {:.2f}) while rendering {}".format(
                        exec_time, timeout, handle_name
                    )
                else:
                    logger.info("Executed handle %s in %.2f sec", handle_name, exec_time)

                send_event(
                    "wall-e{}-handle".format(handle_name.replace("/", "-")),
                    status=juggler_status,
                    message=juggler_message,
                )

        return decorated_function

    return decorator


# TODO(rocco66): add IAM auth, replace @app.api_blueprint.route to @api_handler
@app.api_blueprint.route("/cauth/projects-to-hosts", methods=["GET"])
@cauth_response_handler("/cauth/projects-to-hosts", timeout=PROJECTS_TO_HOSTS_TIMEOUT)
def projects_to_hosts_view():
    out = ""

    projects_containing_hosts = []
    for project, hosts_iter in _iter_projects_to_hosts():
        projects_containing_hosts.append(project)
        out += "{}:{}\n".format(project, ",".join(hosts_iter))

    for project in _iter_projects_without_hosts(projects_containing_hosts):
        out += "{}:\n".format(project)

    return out


@api_handler(
    "/cauth/projects-to-responsibles",
    "GET",
    iam_permissions=iam.NoOneApiIamPermission(),
)
@cauth_response_handler("/cauth/projects-to-responsibles", timeout=PROJECTS_TO_RESPONSIBLES_TIMEOUT)
def projects_to_responsibles_view():
    project_to_owners = _get_projects_to_owners()
    return api_response(project_to_owners)


@api_handler(
    "/cauth/projects-cauth-settings",
    "GET",
    iam_permissions=iam.NoOneApiIamPermission(),
)
@cauth_response_handler("/cauth/projects-cauth-settings", timeout=PROJECTS_CAUTH_SETTINGS_TIMEOUT)
def projects_cauth_settings_view():
    projects_cauth_settings = _get_projects_cauth_settings()
    return api_response(projects_cauth_settings)
