"""Various helpers for view modules."""

import logging

import flask

from sepelib.core.exceptions import LogicalError
from walle import constants as walle_constants

log = logging.getLogger(__name__)

API_ACTION_SET = "set"
API_ACTION_ADD = "add"
API_ACTION_REMOVE = "remove"

_API_ACTION_MAPPING = {
    "PUT": API_ACTION_SET,
    "PATCH": API_ACTION_ADD,
    "POST": API_ACTION_ADD,
    "DELETE": API_ACTION_REMOVE,
}


def get_api_action():
    return _API_ACTION_MAPPING[flask.request.method]


def get_api_action_methods():
    return list(_API_ACTION_MAPPING.keys())


def get_api_action_update_query(field, action, values):
    if values is None and action in (API_ACTION_SET, API_ACTION_REMOVE):
        update_op = "unset"
        values = True
    elif action == API_ACTION_SET:
        update_op = "set"
    elif action == API_ACTION_ADD:
        update_op = "add_to_set"
    elif action == API_ACTION_REMOVE:
        update_op = "pull_all"
    else:
        raise LogicalError()

    return {update_op + "__" + field: values}


def get_vlan_id_schema(description=None):
    schema = {"type": "integer", "minimum": walle_constants.VLAN_ID_MIN, "maximum": walle_constants.VLAN_ID_MAX}
    if description is not None:
        schema["description"] = description
    return schema


def get_eine_tags_schema(for_profile=True):
    description_for_profile = "Einstellung tags to apply to host before profile assigning"
    description_for_deploy = "Einstellung tags to apply to host before deployment"
    return {
        "type": "array",
        "items": {
            "type": "string",
            "pattern": r"^[a-zA-Z0-9_:\-=]+$",
        },
        "maxItems": 20,
        "description": description_for_profile if for_profile else description_for_deploy,
    }


def validated_tags(tags):
    return sorted([tag.lstrip("#") for tag in set(tags)]) or None


def get_action_limits_schema():
    return {
        "type": "array",
        "items": {
            "type": "object",
            "properties": {
                "period": {
                    "type": "string",
                    "pattern": r"^[1-9][0-9]*[smhd]$",
                    "description": "Specifies the period (s - seconds, m - minutes, h - hours, d - days)",
                },
                "limit": {
                    "type": "integer",
                    "minimum": 0,
                    "description": "Maximum number of hosts that can got the resolution for the specified period",
                },
            },
            "required": ["period", "limit"],
            "additionalProperties": False,
        },
        "maxItems": 5,
    }
