"""Wall-E constants"""  # Will be used as category name in API reference

import walle.expert.automation_plot
import walle.scenario.constants as scenario_constants
from walle import audit_log, constants as walle_constants, hosts, projects, restrictions
from walle.clients import cms
from walle.expert.constants import HEALTH_STATUS_FILTERS
from walle.expert.types import CheckType, CheckStatus
from walle.operations_log import constants as oplog_constants
from walle.util.api import api_handler, api_response
from walle.util.deploy_config import DeployConfigPolicies
from walle.util.misc import get_existing_tiers


@api_handler("/constants", "GET")
def get_constants():
    """Provides various Wall-E constants to use them in UI instead of hardcoding on both backend and frontend."""

    return api_response(
        {
            "audit_log_types": sorted(audit_log.TYPES),
            "audit_log_statuses": sorted(audit_log.STATUSES),
            "operation_log_types": sorted(oplog_constants.OPERATION_CHOICES),
            "provisioners": sorted(walle_constants.PROVISIONERS),
            "vlan_schemes": sorted(walle_constants.UI_VLAN_SCHEMES),
            "mtn_ip_methods": sorted(walle_constants.MTN_IP_METHODS),
            "allowed_default_profiles": [walle_constants.FLEXY_EINE_PROFILE],
            "host_states": sorted(hosts.HostState.ALL),
            "host_statuses": _sort(
                hosts.HostStatus.ALL_FILTERS + hosts.HostStatus.ALL,
                hosts.HostStatus.ALL_FILTERS + [hosts.HostStatus.READY],
            ),
            # TODO(rocco66): drop it? (nanny uses it?)
            "health_check_types": CheckType.ALL_UI_TYPES,
            "health_check_groups": CheckType.UI_TYPE_GROUPS,
            "health_statuses": _get_health_statuses(),
            "restrictions": restrictions.ALL,
            "restrictions_dependency_mapping": {
                dependency: list(dependents) for dependency, dependents in restrictions.RESTRICTION_MAPPING.items()
            },
            "project_automation_limits": projects.AUTOMATION_LIMIT_NAMES,
            "host_limits": sorted(projects.HOST_LIMITS_NAMES_WITH_DEFAULT_VALUES.keys()),
            "host_types": walle_constants.HostType.get_choices(),
            "existing_tiers": get_existing_tiers(),
            # UI shows versions from the most recent to the oldest, first item is the default value.
            "cms_api_versions": cms.CmsApiVersion.ALL[::-1],
            # these constants are accessed from yasm templates for panels and alerts
            "juggler_checks": CheckType.ALL,
            "dmc_monitored_checks_statuses": CheckStatus.ALL_YASM_DMC_MONITORED,
            "health_monitored_checks_statuses": CheckStatus.ALL,
            # list all possible automation plots checks
            "automation_plots_checks": list(walle.expert.automation_plot.get_all_automation_plots_checks()),
            # list deploy config policies
            "deploy_config_policies": _list_deploy_policies(),
            # list scenario constants
            "scenario_statuses": list(scenario_constants.ScenarioFsmStatus),
            "scenario_work_statuses": list(scenario_constants.ScenarioWorkStatus),
            "scenario_host_statuses": scenario_constants.HostScenarioStatus.ALL,
            "scenario_types": scenario_constants.ScriptName.ALL,
            "known_queues_for_tickets": list(walle_constants.TicketTrackerQueue),
        }
    )


def _sort(seq, fixed_first_elements):
    """Return sorted list `seq` with `fixed_first_elements` always sorted first."""

    sort_map = dict(zip(fixed_first_elements, range(-len(fixed_first_elements), 0)))
    return sorted(seq, key=lambda x: sort_map.get(x, (hash(x) if hash(x) > 0 else -hash(x))))


def _get_health_statuses():
    return (
        HEALTH_STATUS_FILTERS
        + CheckType.ALL_UI_TYPES
        + list(walle.expert.automation_plot.get_all_automation_plots_checks())
    )


def _list_deploy_policies():
    policies = []
    for policy_name in DeployConfigPolicies.get_all_names():
        entry = {
            "name": policy_name,
            "description": DeployConfigPolicies.get_policy_class(policy_name).description,
        }
        policies.append(entry)
    return policies
